#' Plot significant categorical level differences between data subsets
#'
#' Plot method for S3 objects of class CatDiffs generated by the
#' [CompareCategoricalLevels()] function, creating a horizontal barplot
#' of categorical variable level frequencies that differ significantly
#' between two data subsets
#'
#' @param x an S3 object of class CatDiffs
#' @param ... optional named parameters to be passed to [plot()]
#' @param labelA plot label identifying the first data subset
#' @param labelB plot label identifying the second data subset
#' @param nMax maximum number of levels to include in the barplot
#' (default = 20)
#' @param levelFrac relative position of the level labels on the barplot
#' (default = 0.5)
#' @param xLims two-element vector of x-axis limits for the barplot
#' (default sets the range from 0 to 1.2 times the length of the longest
#' bar on the plot)
#'
#' @return None: this method generates a plot from `x`
#' @export
#'
#' @examples
#' catVar <- c(rep("a", 100), rep("b", 100), rep("c", 100))
#' auxVar <- c(rep("Set1", 30), rep("Set2", 70),
#'            rep("Set1", 50), rep("Set2", 50),
#'            rep("Set1", 90), rep("Set2", 10))
#' DF <- data.frame(catVar = catVar, auxVar = auxVar)
#' indexA <- which(DF$auxVar == "Set1")
#' CatDiffObj <- CompareCategoricalLevels(DF, "catVar", indexA)
#' plot(CatDiffObj, labelA = "Set1", labelB = "Set2")
plot.CatDiffs <- function(x, ..., labelA, labelB,
                          nMax = 20, levelFrac = 0.5,
                          xLims = NULL){
  #
  index <- which(x$signif)
  sigSub <- x[index, c("Level", "pA", "pB")]
  nSig <- nrow(sigSub)
  #
  if (nSig > nMax){
    absDiffs <- abs(sigSub$pA - sigSub$pB)
    sortIndex <- order(absDiffs, decreasing = TRUE)
    sigSub <- sigSub[sortIndex, ]
    sigSub <- sigSub[1:nMax, ]
  }
  #
  probMat <- as.matrix(sigSub[, c("pA", "pB")])
  #
  xMin <- min(probMat)
  xMax <- max(probMat)
  if (is.null(xLims)){
    xLims <- c(0, xMax * 1.2)
  }
  mids <- graphics::barplot(t(probMat), horiz = TRUE, beside = TRUE,
                  las = 1, xlim = xLims, ...)
  delta <- xMax - xMin
  xPos <- xMin + levelFrac * delta
  graphics::text(xPos, mids[2, ], sigSub$Level)
  #
  graphics::legend("right", pch = 15, col = grDevices::gray.colors(2),
         legend = c(labelA, labelB))
}

