

param_emaEM <-
  function(x, k, run_number = 10, max_iter = 3, tol = 1e-6) {
    
  
    X <- as.matrix(x)
    storage.mode(X) <- "double"
    K <- k
    

    em_short <- function(x, k) {
      x <- as.matrix(x)
      storage.mode(x) <- "double"
      
      N <- nrow(x)
      P <- ncol(x)
      K <- k
      
      sigma <- array(0, dim = c(P, P, K))
      
     
      kmeans_init <- suppressWarnings(kmeans(x, k, iter.max = 1))
      pi_k <- kmeans_init$size / sum(kmeans_init$size)
      mu   <- kmeans_init$centers
      
      X_cluster <- split(as.data.frame(x), kmeans_init$cluster)
      for (i in 1:K) {
        Xi <- as.matrix(X_cluster[[i]])
        Xi_center <- sweep(Xi, 2, mu[i, ])
        sigma[,, i] <- t(Xi_center) %*% Xi_center / nrow(Xi)
        sigma[,, i] <- sigma[,, i] + diag(1e-6, P)
      }
      
      gamma <- matrix(NA_real_, N, K)
      log_likelihood_old <- -Inf
      
      for (iter in 1:max_iter) {
        ## E-step
        for (i in 1:K) {
          gamma[, i] <- pi_k[i] * safe_dmv(x, mu = mu[i, ], sigma = sigma[,, i])
        }
        row_den <- rowSums(gamma)
        row_den <- pmax(row_den, .Machine$double.eps)
        gamma   <- gamma / row_den
        
        ### M-step
        Nk   <- colSums(gamma)
        pi_k <- Nk / N
        
        for (i in 1:K) {
          
          mu[i, ] <- colSums(x * gamma[, i]) / Nk[i]
          
         
          centered <- sweep(x, 2, mu[i, ])
          weighted <- centered * gamma[, i]
          sigma[,, i] <- t(centered) %*% weighted / Nk[i]
          sigma[,, i] <- sigma[,, i] + diag(1e-6, P)
        }
        
      
        log_likelihood_new <- sum(log(row_den))
        
       
        diff  <- abs(log_likelihood_new - log_likelihood_old)
        scale <- max(abs(log_likelihood_old), .Machine$double.eps)
        ratio <- diff / scale
        
        if (!is.nan(ratio) && !is.infinite(ratio) && ratio < tol) break
        
        log_likelihood_old <- log_likelihood_new
      }
      
      partition_vector <- apply(gamma, 1, which.max)
      
      list(mu = mu, sigma = sigma, pi_k = pi_k, gamma = gamma,
           partition = partition_vector, ll = log_likelihood_new)
    }
    
    
    safe_em_short <- function(X, K) {
      repeat {
        res <- tryCatch(em_short(X, K), error = function(e) NULL)
        if (!is.null(res)) return(res$partition)
      }
    }
    
    short_em_results <- replicate(run_number,
                                  safe_em_short(X, K),
                                  simplify = TRUE)
    
    
    construct_similarity_matrix <- function(cluster_results) {
      n        <- nrow(cluster_results)
      num_runs <- ncol(cluster_results)
      similarity_matrix <- matrix(0, n, n)
      
      for (run in 1:num_runs) {
        clustering <- cluster_results[, run]
        similarity_matrix <- similarity_matrix +
          outer(clustering, clustering, FUN = "==")
      }
      
      similarity_matrix / num_runs
    }
    
    similarity_matrix   <- construct_similarity_matrix(short_em_results)
    dissimilarity_matrix <- 1 - similarity_matrix
    
    apply_ward_clustering <- function(dissimilarity_matrix, K) {
      distance_matrix <- as.dist(dissimilarity_matrix)
      hc <- hclust(distance_matrix, method = "ward.D2")
      cutree(hc, K)
    }
    
    final_clusters <- apply_ward_clustering(dissimilarity_matrix, K)
    
 
    n <- nrow(X)
    d <- ncol(X)
    k <- length(unique(final_clusters))
    
    mu    <- matrix(0, k, d)
    sigma <- array(0, dim = c(d, d, k))
    pi_k  <- numeric(k)
    
    for (i in 1:k) {
      cluster_points <- X[final_clusters == i, , drop = FALSE]
      Nk <- nrow(cluster_points)
      
      if (Nk > 0) {
        mu[i, ] <- colMeans(cluster_points)
        centered_data <- sweep(cluster_points, 2, mu[i, ])
        sigma[,, i] <- t(centered_data) %*% centered_data / Nk
        sigma[,, i] <- sigma[,, i] + diag(1e-6, d)
        pi_k[i] <- Nk / n
      } else {
  
        sigma[,, i] <- diag(d) * 1e-6
        pi_k[i] <- 1e-6
      }
    }
    
    list(pi_k = pi_k, mu = mu, sigma = sigma)
  }
