#' Update categorical prototype in Gower-SOM
#'
#' Internal helper to update the value of a categorical variable in a neuron
#' prototype, given weighted contributions from neighborhood observations.
#'
#'#' @param values A factor vector of candidate categories (e.g., current prototype
#'   value plus neighborhood observations).
#' @param weights A numeric vector of the same length as \code{values}, giving
#'   the weight for each candidate category.
#' @param sampling Logical. If \code{TRUE}, draw a category at random with
#'   probability proportional to \code{weights}. If \code{FALSE}, select the
#'   category with maximum weighted count.
#'
#' @return A factor with the same levels as \code{values}, containing a single
#'   updated category.
#'
#' @examples
#' vals <- factor(c("A","A","B","C"))
#' wts  <- c(0.2, 0.5, 0.2, 0.1)
#' # Deterministic (weighted mode)
#' gsom_updateCategorical(vals, wts, sampling = FALSE)
#'
#' # Stochastic (multinomial sampling)
#' gsom_updateCategorical(vals, wts, sampling = TRUE)
#'
#' @keywords internal
#' @export
#'

gsom_updateCategorical <- function(values, weights, sampling = FALSE) {
  levels_vals <- levels(values)
  weighted_counts <- tapply(weights, values, sum)
  weighted_counts[is.na(weighted_counts)] <- 0

  if (!sampling) {
    mode_val <- names(which.max(weighted_counts))
    return(factor(mode_val, levels = levels_vals))
  } else {
    probs <- weighted_counts / sum(weighted_counts)
    sampled_val <- sample(names(probs), size = 1, prob = probs)
    return(factor(sampled_val, levels = levels_vals))
  }
}
