NumericalHessianHSMM <- function(x, M = NA, obsdist, dwelldist, HSMM, h = 1e-5, method = "central", verbose = TRUE) {
  # Extract model parameters from fitted HSMM object
  obspar <- HSMM$observationparameters
  dwellpar <- HSMM$dwellparameters
  Pi <- HSMM$Pi
  delta <- HSMM$delta
  J <- length(delta)

  # Set default maximum dwell time if not specified
  if (is.na(M)) M <- min(length(x), 1000)

  # Transform parameters to unconstrained space for numerical stability
  param_vector <- createParamVector(obspar, dwellpar, obsdist, dwelldist, J)
  n_params <- length(param_vector)

  if (verbose) message("Computing numerical Hessian with ", n_params, " parameters...")

  # Define log-likelihood function for optimization
  loglik_function <- function(params) {
    tryCatch({
      # Reconstruct parameter lists from vector
      param_lists <- reconstructParams(params, obspar, dwellpar, obsdist, dwelldist, J)

      # Compute log-likelihood using forward-backward algorithm
      loglik <- computeLogLikHSMM(x, J, M, obsdist, dwelldist,
                                  param_lists$obspar, param_lists$dwellpar, Pi, delta)

      # Return finite log-likelihood or penalty value
      if (is.finite(loglik)) return(loglik) else return(-1e10)
    }, error = function(e) {
      return(-1e10) # Return penalty for parameter combinations causing errors
    })
  }

  # Compute Hessian using specified finite difference method
  if (method == "central") {
    hessian <- computeCentralDifferenceHessian(loglik_function, param_vector, h, verbose)
  } else {
    hessian <- computeForwardDifferenceHessian(loglik_function, param_vector, h, verbose)
  }

  # Add parameter names for interpretability
  param_names <- createParamNames(obspar, dwellpar, obsdist, dwelldist, J)
  rownames(hessian) <- param_names
  colnames(hessian) <- param_names

  if (verbose) message("Hessian computation complete.")

  # Return negative Hessian (for minimization context)
  return(-hessian)
}

createParamVector <- function(obspar, dwellpar, obsdist, dwelldist, J) {
  params <- c()

  # Transform observation distribution parameters
  if (obsdist == "gev") {
    # GEV: loc (unconstrained), log(scale), shape (unconstrained)
    params <- c(params, obspar$loc, log(obspar$scale), obspar$shape)
  } else if (obsdist == "norm") {
    # Normal: mean (unconstrained), log(sd)
    params <- c(params, obspar$mean, log(obspar$sd))
  } else if (obsdist == "pois") {
    # Poisson: log(lambda)
    params <- c(params, log(obspar$lambda))
  } else if (obsdist == "nbinom") {
    # Negative binomial: log(size), log(mu)
    params <- c(params, log(obspar$size), log(obspar$mu))
  } else if (obsdist == "weibull") {
    # Weibull: log(scale), log(shape)
    params <- c(params, log(obspar$scale), log(obspar$shape))
  } else if (obsdist == "gamma") {
    # Gamma: log(shape), log(rate)
    params <- c(params, log(obspar$shape), log(obspar$rate))
  } else if (obsdist == "exp") {
    # Exponential: log(rate)
    params <- c(params, log(obspar$rate))
  } else if (obsdist == "lnorm") {
    # Log-normal: meanlog (unconstrained), log(sdlog)
    params <- c(params, obspar$meanlog, log(obspar$sdlog))
  } else if (obsdist == "zip") {
    # Zero-inflated Poisson: logit(pi), log(lambda)
    params <- c(params, log(obspar$pi / (1 - obspar$pi)), log(obspar$lambda))
  } else if (obsdist == "zinb") {
    # Zero-inflated negative binomial: logit(pi), log(size), log(mu)
    params <- c(params, log(obspar$pi / (1 - obspar$pi)), log(obspar$size), log(obspar$mu))
  } else if (obsdist == "ZInormal") {
    # Zero-inflated normal: logit(pi), mean (unconstrained), log(sd)
    params <- c(params, log(obspar$pi / (1 - obspar$pi)), obspar$mean, log(obspar$sd))
  } else if (obsdist == "ZIgamma") {
    # Zero-inflated gamma: logit(pi), log(shape), log(rate)
    params <- c(params, log(obspar$pi / (1 - obspar$pi)), log(obspar$shape), log(obspar$rate))
  }

  # Transform dwell time distribution parameters
  if (dwelldist == "pois") {
    # Poisson dwell: log(lambda)
    params <- c(params, log(dwellpar$lambda))
  } else if (dwelldist == "nbinom") {
    # Negative binomial dwell: log(size), log(mu)
    params <- c(params, log(dwellpar$size), log(dwellpar$mu))
  } else if (dwelldist == "betabinom") {
    # Beta-binomial dwell: log(size), log(alpha), log(beta)
    params <- c(params, log(dwellpar$size), log(dwellpar$alpha), log(dwellpar$beta))
  } else if (dwelldist == "geom") {
    # Geometric dwell: logit(prob)
    params <- c(params, log(dwellpar$prob / (1 - dwellpar$prob)))
  }

  return(params)
}

createParamNames <- function(obspar, dwellpar, obsdist, dwelldist, J) {
  names <- c()

  # Observation parameter names with transformation indicators
  if (obsdist == "gev") {
    names <- c(names, paste0("loc_", 1:J), paste0("log_scale_", 1:J), paste0("shape_", 1:J))
  } else if (obsdist == "norm") {
    names <- c(names, paste0("mean_", 1:J), paste0("log_sd_", 1:J))
  } else if (obsdist == "pois") {
    names <- c(names, paste0("log_lambda_", 1:J))
  } else if (obsdist == "nbinom") {
    names <- c(names, paste0("log_size_", 1:J), paste0("log_mu_", 1:J))
  } else if (obsdist == "weibull") {
    names <- c(names, paste0("log_scale_", 1:J), paste0("log_shape_", 1:J))
  } else if (obsdist == "gamma") {
    names <- c(names, paste0("log_shape_", 1:J), paste0("log_rate_", 1:J))
  } else if (obsdist == "exp") {
    names <- c(names, paste0("log_rate_", 1:J))
  } else if (obsdist == "lnorm") {
    names <- c(names, paste0("meanlog_", 1:J), paste0("log_sdlog_", 1:J))
  } else if (obsdist == "zip") {
    names <- c(names, paste0("logit_pi_", 1:J), paste0("log_lambda_", 1:J))
  } else if (obsdist == "zinb") {
    names <- c(names, paste0("logit_pi_", 1:J), paste0("log_size_", 1:J), paste0("log_mu_", 1:J))
  } else if (obsdist == "ZInormal") {
    names <- c(names, paste0("logit_pi_", 1:J), paste0("mean_", 1:J), paste0("log_sd_", 1:J))
  } else if (obsdist == "ZIgamma") {
    names <- c(names, paste0("logit_pi_", 1:J), paste0("log_shape_", 1:J), paste0("log_rate_", 1:J))
  }

  # Dwell parameter names with "dwell_" prefix
  if (dwelldist == "pois") {
    names <- c(names, paste0("dwell_log_lambda_", 1:J))
  } else if (dwelldist == "nbinom") {
    names <- c(names, paste0("dwell_log_size_", 1:J), paste0("dwell_log_mu_", 1:J))
  } else if (dwelldist == "betabinom") {
    names <- c(names, paste0("dwell_log_size_", 1:J), paste0("dwell_log_alpha_", 1:J), paste0("dwell_log_beta_", 1:J))
  } else if (dwelldist == "geom") {
    names <- c(names, paste0("dwell_logit_prob_", 1:J))
  }

  return(names)
}

reconstructParams <- function(params, obspar_template, dwellpar_template, obsdist, dwelldist, J) {
  idx <- 1
  new_obspar <- obspar_template

  # Reconstruct observation parameters with inverse transformations
  if (obsdist == "gev") {
    new_obspar$loc <- params[idx:(idx + J - 1)]
    idx <- idx + J
    new_obspar$scale <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_obspar$shape <- params[idx:(idx + J - 1)]
    idx <- idx + J
  } else if (obsdist == "norm") {
    new_obspar$mean <- params[idx:(idx + J - 1)]
    idx <- idx + J
    new_obspar$sd <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "pois") {
    new_obspar$lambda <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "nbinom") {
    new_obspar$size <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_obspar$mu <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "weibull") {
    new_obspar$scale <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_obspar$shape <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "gamma") {
    new_obspar$shape <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_obspar$rate <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "exp") {
    new_obspar$rate <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "lnorm") {
    new_obspar$meanlog <- params[idx:(idx + J - 1)]
    idx <- idx + J
    new_obspar$sdlog <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "zip") {
    # Inverse logit transform for pi
    logit_pi <- params[idx:(idx + J - 1)]
    new_obspar$pi <- exp(logit_pi) / (1 + exp(logit_pi))
    idx <- idx + J
    new_obspar$lambda <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "zinb") {
    # Inverse logit transform for pi
    logit_pi <- params[idx:(idx + J - 1)]
    new_obspar$pi <- exp(logit_pi) / (1 + exp(logit_pi))
    idx <- idx + J
    new_obspar$size <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_obspar$mu <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "ZInormal") {
    # Inverse logit transform for pi
    logit_pi <- params[idx:(idx + J - 1)]
    new_obspar$pi <- exp(logit_pi) / (1 + exp(logit_pi))
    idx <- idx + J
    new_obspar$mean <- params[idx:(idx + J - 1)]
    idx <- idx + J
    new_obspar$sd <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (obsdist == "ZIgamma") {
    # Inverse logit transform for pi
    logit_pi <- params[idx:(idx + J - 1)]
    new_obspar$pi <- exp(logit_pi) / (1 + exp(logit_pi))
    idx <- idx + J
    new_obspar$shape <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_obspar$rate <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  }

  # Reconstruct dwell parameters with inverse transformations
  new_dwellpar <- dwellpar_template
  if (dwelldist == "pois") {
    new_dwellpar$lambda <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (dwelldist == "nbinom") {
    new_dwellpar$size <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_dwellpar$mu <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (dwelldist == "betabinom") {
    new_dwellpar$size <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_dwellpar$alpha <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
    new_dwellpar$beta <- exp(params[idx:(idx + J - 1)]) # Inverse log transform
    idx <- idx + J
  } else if (dwelldist == "geom") {
    # Inverse logit transform for prob
    logit_prob <- params[idx:(idx + J - 1)]
    new_dwellpar$prob <- exp(logit_prob) / (1 + exp(logit_prob))
    idx <- idx + J
  }

  return(list(obspar = new_obspar, dwellpar = new_dwellpar))
}

computeLogLikHSMM <- function(x, J, M, obsdist, dwelldist, obspar, dwellpar, Pi, delta) {
  # Compute dwell time probabilities and survival functions
  dwell <- dwellprobs(M, J, dwelldist, dwellpar)
  dwellprobs <- dwell$dwellprobs
  surv <- dwell$surv

  # Compute observation probabilities
  obsprobs <- obsprobs(x, J, obsdist, obspar)

  tau <- length(x)

  # Call compiled forward-backward algorithm
  forback <- backwards(
    transProb = as.matrix(Pi),
    initDist = as.numeric(delta),
    obsProb = as.matrix(obsprobs),
    durProb = as.matrix(dwellprobs),
    survProb = as.matrix(surv),
    seqLen = as.integer(tau),
    numStates = as.integer(J),
    maxDur = as.integer(rep(M, J)),
    backwardProb = matrix(0, nrow = tau, ncol = J),
    normConst = numeric(tau),
    eta = numeric(M * J),
    forwardVars = matrix(0, nrow = tau, ncol = J),
    stateProb = numeric(J * tau),
    xi = matrix(0, nrow = tau, ncol = J),
    numSeq = as.integer(1),
    totalLen = as.integer(tau),
    backwardVars = matrix(0, nrow = tau, ncol = J)
  )

  # Return sum of log normalization constants
  return(sum(log(forback$normConst)))
}

computeCentralDifferenceHessian <- function(func, params, h, verbose = TRUE) {
  n <- length(params)
  hessian <- matrix(0, n, n)

  # Compute upper triangle of Hessian (symmetric matrix)
  for (i in 1:n) {
    for (j in 1:n) {
      if (i <= j) {
        # Create parameter vectors for central difference approximation
        params_pp <- params_pm <- params_mp <- params_mm <- params

        # Four-point stencil for mixed partial derivatives
        params_pp[i] <- params_pp[i] + h
        params_pp[j] <- params_pp[j] + h

        params_pm[i] <- params_pm[i] + h
        params_pm[j] <- params_pm[j] - h

        params_mp[i] <- params_mp[i] - h
        params_mp[j] <- params_mp[j] + h

        params_mm[i] <- params_mm[i] - h
        params_mm[j] <- params_mm[j] - h

        # Evaluate function at four points
        f_pp <- func(params_pp)
        f_pm <- func(params_pm)
        f_mp <- func(params_mp)
        f_mm <- func(params_mm)

        # Central difference approximation for second mixed partial
        hessian[i, j] <- (f_pp - f_pm - f_mp + f_mm) / (4 * h^2)
        hessian[j, i] <- hessian[i, j] # Exploit symmetry
      }
    }

    # Progress indicator for long computations
    if (verbose && i %% max(1, n %/% 10) == 0) {
      message("Hessian progress: ", round(100 * i / n, 1), "%")
    }
  }

  return(hessian)
}

computeForwardDifferenceHessian <- function(func, params, h, verbose = TRUE) {
  n <- length(params)
  hessian <- matrix(0, n, n)

  # Evaluate function at base point
  f0 <- func(params)

  # Compute gradient using forward differences
  grad <- numeric(n)
  for (i in 1:n) {
    params_h <- params
    params_h[i] <- params_h[i] + h
    grad[i] <- (func(params_h) - f0) / h
  }

  # Compute Hessian as gradient of gradient
  for (i in 1:n) {
    params_h <- params
    params_h[i] <- params_h[i] + h

    # Compute gradient
    grad_h <- numeric(n)
    for (j in 1:n) {
      params_hh <- params_h
      params_hh[j] <- params_hh[j] + h
      grad_h[j] <- (func(params_hh) - func(params_h)) / h
    }

    # Forward difference approximation for Hessian row
    hessian[i, ] <- (grad_h - grad) / h

    # Progress indicator
    if (verbose && i %% max(1, n %/% 10) == 0) {
      message("Hessian progress: ", round(100 * i / n, 1), "%")
    }
  }

  # Symmetrize the matrix (average with transpose)
  hessian <- (hessian + t(hessian)) / 2

  return(hessian)
}

make_positive_definite <- function(mat, min_eigenvalue = 1e-8) {
  # Eigenvalue decomposition
  eigen_decomp <- eigen(mat)
  eigenvalues <- eigen_decomp$values
  eigenvectors <- eigen_decomp$vectors

  # Adjust negative or small eigenvalues
  eigenvalues[eigenvalues < min_eigenvalue] <- min_eigenvalue

  # Reconstruct matrix with adjusted eigenvalues
  return(eigenvectors %*% diag(eigenvalues) %*% t(eigenvectors))
}

HSMMVarianceMatrix <- function(x, HSMM, M = NA, obsdist, dwelldist, h = 1e-5, method = "central", verbose = TRUE) {
  if (verbose) message("Computing numerical Hessian for HSMM model...")
  J <- length(HSMM$delta)

  # Compute negative Hessian (Fisher Information Matrix)
  neg_hessian <- NumericalHessianHSMM(
    x = x,
    M = M,
    obsdist = obsdist,
    dwelldist = dwelldist,
    HSMM = HSMM,
    h = h,
    method = method,
    verbose = verbose
  )

  # Attempt to invert Hessian for variance-covariance matrix
  var_cov <- tryCatch({
    solve(neg_hessian)
  }, error = function(e) {
    if (verbose) message("Warning: Hessian inversion failed, using pseudo-inverse")
    ginv(neg_hessian) # Generalized inverse as fallback
  })

  # Ensure positive definiteness for numerical stability
  var_cov <- make_positive_definite(var_cov)

  return(var_cov)
}
