globaldecodeHSMM <- function(x, M = NA, HSMM, obsdist, dwelldist, shift=FALSE) {

  # Input validation
  if(!dwelldist %in% c("pois", "nbinom", "betabinom", "geom")) {
    stop("dwell distribution is not supported")
  }
  if(!obsdist %in% c("pois", "norm", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma")) {
    stop("observation distribution is not supported")
  }

  # Extract model parameters
  J <- length(HSMM$delta)
  dwellpar <- HSMM$dwellparameters
  obspar <- HSMM$observationparameters
  Pi <- HSMM$Pi

  # Initialize shift parameters
  if(shift==FALSE){
    dwellpar$shift <- rep(1, J)
  }

  # Use stationary distribution if delta not available
  delta <- solve(t(diag(J) - Pi + 1), rep(1, J))

  # Setup algorithm parameters
  tau <- length(x)
  M <- if(is.na(M)) min(tau, 1000) else M
  min <- -1e300  # Log of very small probability

  # Compute dwelltime probabilities
  dwellProbabilities <- dwellprobs(M, J, dwelldist, dwellpar)

  # Convert transition probabilities to log scale
  transitionLogProbs <- matrix(0, J, J)
  for(i in 1:J) {
    for(j in 1:J) {
      if(Pi[i,j] > 0) {
        transitionLogProbs[i,j] <- log(Pi[i,j])
      } else {
        transitionLogProbs[i,j] <- min  # Handle zero probabilities
      }
    }
  }

  # Convert initial probabilities to log scale
  initialLogProbs <- sapply(delta, function(p) {
    if(p > 0 && !is.nan(p)) log(p) else min
  })

  # Convert dwelltime probabilities to log scale
  dwellLogProbs <- lapply(dwellProbabilities$dwellprobs, function(probs) {
    logProbs <- numeric(length(probs))
    for(i in seq_along(probs)) {
      logProbs[i] <- if(probs[i] > 0) log(probs[i]) else min
    }
    return(logProbs)
  })

  # Convert survival probabilities to log scale
  survLogProbs <- lapply(dwellProbabilities$surv, function(probs) {
    logProbs <- numeric(length(probs))
    for(i in seq_along(probs)) {
      logProbs[i] <- if(probs[i] > 0) log(probs[i]) else min
    }
    return(logProbs)
  })

  # Flatten probability matrices for C interface
  flatDwellLogProbs <- unlist(dwellLogProbs)
  flatSurvLogProbs <- unlist(survLogProbs)

  # Convert observation probabilities to log scale
  observationProbabilities <- obsprobs(x, J, obsdist, obspar)
  obsLogProbs <- numeric(length(observationProbabilities))
  for(i in seq_along(observationProbabilities)) {
    obsLogProbs[i] <- if(observationProbabilities[i] > 0) {
      log(observationProbabilities[i])
    } else {
      min
    }
  }

  # Initialize arrays for Viterbi algorithm
  forwardProbsArray <- numeric(tau * J)
  prevStatesArray <- integer(tau * J)
  durationsArray <- integer(tau * J)
  pathArray <- integer(tau)
  maxDurations <- rep(M, J)

  # Call HSMM Viterbi algorithm (implemented in C++)
  # Finds optimal state sequence considering dwell times
  viterbi(
    as.numeric(t(transitionLogProbs)),
    as.numeric(initialLogProbs),
    as.numeric(obsLogProbs),
    as.numeric(flatDwellLogProbs),
    as.numeric(flatSurvLogProbs),
    as.integer(tau),
    as.integer(J),
    as.integer(maxDurations),
    forwardProbsArray,
    pathArray,
    prevStatesArray,
    durationsArray
  )

  # Reshape forward probabilities for analysis (optional)
  forwardMatrix <- matrix(0, nrow = J, ncol = tau)
  for(j in 1:J) {
    for(t in 1:tau) {
      forwardMatrix[j,t] <- forwardProbsArray[(j-1)*tau + t]
    }
  }

  # Extract maximum likelihood and optimal state sequence
  logLikelihood <- max(forwardMatrix[,tau])
  stateSequence <- numeric(tau)
  for(t in 1:tau) {
    stateSequence[t] <- pathArray[t] + 1  # Convert from 0-indexed to 1-indexed
  }

  return(stateSequence)
}

