negloglikHMM <- function(J, x, wparams, obsdist) {

  # Convert working parameters back to natural scale
  nparams <- params.w2n(J, wparams, obsdist)
  n <- length(x)

  # Calculate state-dependent observation probabilities
  stateprobs <- matrix(NA, n, J)
  for (i in 1:n) {
    for (j in 1:J) {
      if (obsdist == "norm") {
        # Normal distribution density
        stateprobs[i, j] <- dnorm(x[i], mean = nparams$mean[j], sd = nparams$sd[j])

      } else if (obsdist == "pois") {
        # Poisson distribution density
        stateprobs[i, j] <- dpois(x[i], lambda = nparams$lambda[j])

      } else if (obsdist == "weibull") {
        # Weibull distribution density
        stateprobs[i, j] <- dweibull(x[i], shape = nparams$shape[j], scale = nparams$scale[j])

      } else if (obsdist == "zip") {
        # Zero-inflated Poisson distribution density
        if (x[i] == 0) {
          # P(X=0) = pi + (1-pi) * P(Poisson=0)
          stateprobs[i, j] <- nparams$pi[j] + (1 - nparams$pi[j]) * dpois(0, nparams$lambda[j])
        } else {
          # P(X>0) = (1-pi) * P(Poisson=x)
          stateprobs[i, j] <- (1 - nparams$pi[j]) * dpois(x[i], nparams$lambda[j])
        }

      } else if (obsdist == "nbinom") {
        # Negative binomial distribution density
        stateprobs[i, j] <- dnbinom(x[i], size = nparams$size[j], mu = nparams$mu[j])

      } else if (obsdist == "zinb") {
        # Zero-inflated negative binomial distribution density
        if (x[i] == 0) {
          # P(X=0) = pi + (1-pi) * P(NegBinom=0)
          stateprobs[i, j] <- nparams$pi[j] + (1 - nparams$pi[j]) * dnbinom(0, size = nparams$size[j], mu = nparams$mu[j])
        } else {
          # P(X>0) = (1-pi) * P(NegBinom=x)
          stateprobs[i, j] <- (1 - nparams$pi[j]) * dnbinom(x[i], size = nparams$size[j], mu = nparams$mu[j])
        }

      } else if (obsdist == "exp") {
        # Exponential distribution density
        stateprobs[i, j] <- dexp(x[i], rate = nparams$rate[j])

      } else if (obsdist == "gamma") {
        # Gamma distribution density
        stateprobs[i, j] <- dgamma(x[i], shape = nparams$shape[j], rate = nparams$rate[j])

      } else if (obsdist == "lnorm") {
        # Log-normal distribution density
        stateprobs[i, j] <- dlnorm(x[i], meanlog = nparams$meanlog[j], sdlog = nparams$sdlog[j])

      } else if (obsdist == "gev") {
        # Generalized extreme value distribution density
        stateprobs[i, j] <- devd(x[i], loc = nparams$loc[j], scale = nparams$scale[j], shape = nparams$shape[j], type = "GEV", log = FALSE)

      } else if (obsdist == "ZInormal") {
        # Zero-inflated normal distribution density
        if (x[i] == 0) {
          # P(X=0) = pi + (1-pi) * f_N(0)
          stateprobs[i, j] <- nparams$pi[j] + (1 - nparams$pi[j]) * dnorm(0, mean = nparams$mean[j], sd = nparams$sd[j])
        } else {
          # P(X=x) = (1-pi) * f_N(x)
          stateprobs[i, j] <- (1 - nparams$pi[j]) * dnorm(x[i], mean = nparams$mean[j], sd = nparams$sd[j])

        }
      } else if (obsdist == "ZIgamma") {
        # Zero-inflated gamma distribution density
        if (x[i] == 0) {
          # P(X=0) = pi + (1-pi) * f_N(0)
          stateprobs[i, j] <- nparams$pi[j] + (1 - nparams$pi[j]) * dgamma(0, shape = nparams$shape[j], rate = nparams$rate[j])
        } else {
          # P(X=x) = (1-pi) * f_N(x)
          stateprobs[i, j] <- (1 - nparams$pi[j]) * dgamma(x[i], shape = nparams$shape[j], rate = nparams$rate[j])

        }
      }
    }
  }

  # Forward algorithm implementation
  # Initialize forward probabilities at t=1
  alpha <- nparams$delta * stateprobs[1, ]
  sumalpha <- sum(alpha)
  loglik <- log(sumalpha)
  alpha <- alpha / sumalpha  # Scale to prevent numerical underflow

  # Forward recursion for t=2 to n
  for (i in 2:n) {
    # Forward step: alpha_t(j) = sum_i(alpha_{t-1}(i) * P_ij) * b_j(x_t)
    alpha <- alpha %*% nparams$Pi * stateprobs[i, ]
    sumalpha <- sum(alpha)
    loglik <- loglik + log(sumalpha)  # Accumulate log-likelihood
    alpha <- alpha / sumalpha  # Scale to prevent numerical underflow
  }

  # Return negative log-likelihood for minimization
  negloglik <- -loglik
  return(negloglik)
}




