residualsHSMM <- function(x, HSMM, obsdist, dwelldist, M = NA, lag.max = 50, nsim = 100, use.theoretical.acf = FALSE, verbose = TRUE, seed = NULL) {
  # Set seed if provided
  if (!is.null(seed)) {
    set.seed(seed)
  }

  # Extract basic parameters
  n <- length(x)
  J <- length(HSMM$delta)
  Pi <- HSMM$Pi

  # Compute stationary distribution
  delta <- solve(t(diag(J) - Pi + 1), rep(1, J))
  obspar <- HSMM$observationparameters
  dwellpar <- HSMM$dwellparameters

  # Set maximum dwell time if not specified
  if (is.na(M)) M <- min(1000, n)

  # Inner function to compute ordinary residuals for given data
  compute_residuals <- function(data) {
    # Calculate observation probabilities and dwell time distributions
    obsprobs_density <- obsprobs(data, J, obsdist, obspar)
    dwell <- dwellprobs(M, J, dwelldist, dwellpar)
    dwellprobs <- dwell$dwellprobs
    surv <- dwell$surv

    # Deep copy parameters to avoid reference issues
    Pi2 <- unserialize(serialize(Pi, NULL))
    delta2 <- unserialize(serialize(delta, NULL))

    # Run forward-backward algorithm to get state probabilities
    forback <- backwards(
      transProb = as.matrix(Pi2),
      initDist = as.numeric(delta2),
      obsProb = as.matrix(obsprobs_density),
      durProb = as.matrix(dwellprobs),
      survProb = as.matrix(surv),
      seqLen = rep(length(data), 1),
      numStates = as.integer(J),
      maxDur = as.integer(rep(M, J)),
      backwardProb = matrix(0, nrow = length(data), ncol = J),
      normConst = numeric(length(data)),
      eta = numeric(M * J),
      forwardVars = matrix(0, nrow = length(data), ncol = J),
      stateProb = numeric(J * length(data)),
      xi = matrix(0, nrow = length(data), ncol = J),
      numSeq = as.integer(1),
      totalLen = as.integer(length(data)),
      backwardVars = matrix(0, nrow = length(data), ncol = J)
    )

    # Extract state probabilities
    xi <- matrix(forback$xi, nrow = length(data), ncol = J, byrow = FALSE)

    # Compute cumulative distribution functions for each observation distribution
    obsprobs_cdf <- matrix(NA, nrow = length(data), ncol = J)
    for (i in 1:length(data)) {
      if (obsdist == "norm") {
        # Normal CDF
        obsprobs_cdf[i, ] <- pnorm(data[i], mean = obspar$mean, sd = obspar$sd)
      } else if (obsdist == "pois") {
        #Poisson CDF
        obsprobs_cdf[i, ] <- ppois(data[i], lambda = obspar$lambda)
      } else if (obsdist == "zip") {
        # Zero-inflated Poisson CDF
        for (j in 1:J) {
          pi0 <- obspar$pi[j]; lambda <- obspar$lambda[j]
          obsprobs_cdf[i, j] <- if (data[i] == 0) pi0 + (1 - pi0) * ppois(0, lambda) else (1 - pi0) * ppois(data[i], lambda)
        }
      } else if (obsdist == "zinb") {
        # Zero-inflated negative binomial CDF
        for (j in 1:J) {
          pi0 <- obspar$pi[j]; mu <- obspar$mu[j]; size <- obspar$size[j]
          obsprobs_cdf[i, j] <- if (data[i] == 0) pi0 + (1 - pi0) * pnbinom(0, size, mu = mu) else (1 - pi0) * pnbinom(data[i], size, mu = mu)
        }
      } else if (obsdist == "nbinom") {
        # Negative binomial CDF
        obsprobs_cdf[i, ] <- pnbinom(data[i], size = obspar$size, mu = obspar$mu)
      } else if (obsdist == "exp") {
        # Exponential CDF
        obsprobs_cdf[i, ] <- pexp(data[i], rate = obspar$rate)
      } else if (obsdist == "gamma") {
        # Gamma CDF
        obsprobs_cdf[i, ] <- pgamma(data[i], shape = obspar$shape, rate = obspar$rate)
      } else if(obsdist=="lognormal"){
        # Log-normal CDF
        obsprobs_cdf[i, ] <- plnorm(data[i], meanlog = obspar$meanlog, sdlog=obspar$sdlog)
      }else if (obsdist == "gev") {
        # Generalized extreme value distribution CDF
        for (j in 1:J) {
          obsprobs_cdf[i, j] <- extRemes::pevd(data[i], loc = obspar$loc[j], scale = obspar$scale[j], shape = obspar$shape[j], type = "GEV")
        }
      } else if (obsdist == "ZInormal") {
        # Zero-inflated normal CDF
        for (j in 1:J) {
          pi0 <- obspar$pi[j]; mean <- obspar$mean[j]; sd <- obspar$sd[j]
          obsprobs_cdf[i, j] <- if (data[i] == 0) pi0 + (1 - pi0) * pnorm(0, mean = mean, sd = sd) else (1 - pi0) * pnorm(0, mean = mean, sd = sd)
        }
      } else if (obsdist == "ZIgamma") {
        # Zero-inflated gamma CDF
        for (j in 1:J) {
          pi0 <- obspar$pi[j]; shape <- obspar$shape[j]; rate <- obspar$rate[j]
          obsprobs_cdf[i, j] <- if (data[i] == 0) pi0 + (1 - pi0) * pgamma(0, shape = shape, rate = rate) else (1 - pi0) * pgamma(data[i], shape = shape, rate = rate)
        }
      } else {
        stop("obsdist not implemented")
      }
    }

    # Calculate ordinary residuals using probability integral transformation
    probs <- rowSums(obsprobs_cdf * xi)
    ordresids <- qnorm(probs)
    ordresids <- ordresids[!is.na(ordresids)]
    return(ordresids)
  }

  # Compute observed residuals
  observed_residuals <- compute_residuals(x)

  if (verbose) message(sprintf("Generating %d simulations for envelope...", nsim))

  # Initialize storage for simulation results
  sim_residuals_list <- vector("list", nsim)
  sim_acf_list <- vector("list", nsim)
  nsuccess <- 0

  # Generate simulation residual envelope
  for (i in 1:nsim) {
    if (verbose && i %% 10 == 0) message(sprintf("Simulation %d/%d", i, nsim))

    # Generate simulated data from fitted HSMM
    sim_data <- tryCatch({
      generateHSMM(n, J, obsdist, dwelldist, obspar, dwellpar, Pi, delta, simtype = "nobs")
    }, error = function(e) {
      if (verbose) message(sprintf("Simulation %d failed: %s", i, e$message))
      return(NULL)
    })

    if (!is.null(sim_data)) {
      # Compute residuals for simulated data
      sim_residuals <- tryCatch({
        compute_residuals(sim_data$x)
      }, error = function(e) {
        if (verbose) message(sprintf("Residual computation failed on simulation %d: %s", i, e$message))
        return(NULL)
      })

      if (!is.null(sim_residuals)) {
        sim_residuals_list[[i]] <- sort(sim_residuals)
        nsuccess <- nsuccess + 1

        # Compute ACF for residuals if sufficient data
        if (length(sim_residuals) > lag.max + 1) {
          sim_acf <- acf(sim_residuals, lag.max = lag.max, plot = FALSE)$acf[-1]
          sim_acf_list[[i]] <- sim_acf
        }
      }
    }
  }

  if (verbose) message(sprintf("Completed %d/%d successful simulations", nsuccess, nsim))

  # Construct quantile-quantile plot envelope
  n_resid <- length(observed_residuals)
  qq_low <- qq_med <- qq_upp <- numeric(n_resid)

  for (j in 1:n_resid) {
    sim_values <- sapply(sim_residuals_list, function(sr) if (!is.null(sr) && length(sr) >= j) sr[j] else NA)
    sim_values <- sim_values[!is.na(sim_values)]
    if (length(sim_values) > 0) {
      qq_low[j] <- quantile(sim_values, 0.025, na.rm = TRUE)
      qq_med[j] <- quantile(sim_values, 0.5, na.rm = TRUE)
      qq_upp[j] <- quantile(sim_values, 0.975, na.rm = TRUE)
    }
  }

  # Store QQ envelope results
  residenv <- list(med = qq_med, low = qq_low, upp = qq_upp)
  valid_acf_sims <- sim_acf_list[!sapply(sim_acf_list, is.null)]

  # Construct ACF envelope (simulation-based or theoretical)
  if (!use.theoretical.acf && length(valid_acf_sims) > 0) {
    # Use simulation-based ACF envelope
    acf_low <- acf_upp <- numeric(lag.max)
    for (lag in 1:lag.max) {
      acf_values <- sapply(valid_acf_sims, function(sa) if (length(sa) >= lag) sa[lag] else NA)
      acf_values <- acf_values[!is.na(acf_values)]
      if (length(acf_values) > 0) {
        acf_low[lag] <- quantile(acf_values, 0.025)
        acf_upp[lag] <- quantile(acf_values, 0.975)
      }
    }
    acfenv <- list(
      low = c(NA_real_, acf_low)[1:(lag.max + 1)],
      upp = c(NA_real_, acf_upp)[1:(lag.max + 1)]
    )
  } else {
    # Use theoretical ACF envelope for white noise
    if (verbose) message("Using theoretical ACF envelope")
    n2 <- length(observed_residuals) - (0:lag.max)
    df <- n2 - 2
    t.low <- qt(0.025, df)
    t.upp <- qt(0.975, df)
    low.acf <- t.low / sqrt(df + t.low^2)
    upp.acf <- t.upp / sqrt(df + t.upp^2)
    acfenv <- list(
      low = c(NA_real_, low.acf)[1:(lag.max + 1)],
      upp = c(NA_real_, upp.acf)[1:(lag.max + 1)]
    )
  }

  # Compute observed ACF
  acf_obs <- acf(observed_residuals, lag.max = lag.max, plot = FALSE)

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  # Create diagnostic plots
  par(mfrow = c(1, 2))

  # QQ plot with simulation envelope
  sorted_obs <- sort(observed_residuals)
  ylimqq <- range(sorted_obs, residenv$upp, residenv$low, na.rm = TRUE)
  plot(residenv$med, sorted_obs, ylab = "", ylim = ylimqq, xlab = "Expected",
       cex.axis = 1.2, cex.lab = 1.2, pch = 16, cex = 0.5)
  title(ylab = "Observed", line = 2, cex.lab = 1.2)
  lines(residenv$med, residenv$med, lwd = 0.6)
  lines(residenv$med, residenv$low, col = "red", lty = 2, lwd = 0.6)
  lines(residenv$med, residenv$upp, col = "red", lty = 2, lwd = 0.6)
  title(main = "QQ Plot (Simulation Envelope)")

  # ACF plot with simulation or theoretical envelope
  acf_title <- if (use.theoretical.acf || length(valid_acf_sims) == 0) {
    "ACF (Theoretical Envelope)"
  } else {
    "ACF (Simulation Envelope)"
  }
  ylimacf <- range(c(acfenv$low, acfenv$upp, acf_obs$acf), na.rm = TRUE)
  plot(acf_obs, main = acf_title, ci = 0, ylim = ylimacf,
       cex.axis = 1.2, cex.lab = 1.2, lwd = 0.6, xlab = "Lag", ylab = "")
  lines(0:lag.max, acfenv$low, lty = 2, col = "blue")
  lines(0:lag.max, acfenv$upp, lty = 2, col = "blue")
  title(ylab = "ACF", line = 2, cex.lab = 1.2)

  # Report diagnostic summary
  propoutside <- mean(sorted_obs < residenv$low | sorted_obs > residenv$upp, na.rm = TRUE)
  if (verbose) {
    message(sprintf("Proportion of ordinary residuals outside 95%% simulation envelope: %.1f%%", 100 * propoutside))
  }

  invisible(list(
    residuals = observed_residuals,
    qq_envelope = residenv,
    acf_values = acf_obs,
    acf_envelope = acfenv,
    proportion_outside = propoutside,
    n_successful_sims = nsuccess
  ))
}



