\name{findmleHSMMnostarting}
\alias{findmleHSMMnostarting}
\title{Fit Hidden Semi-Markov Model (HSMM) Without User-Provided Starting Values}
\description{
Fits a hidden semi-Markov model (HSMM) to a univariate time series by maximum likelihood,
using multiple randomly generated initializations instead of user-provided starting parameters.
The best-fitting model is selected by log-likelihood.
}
\usage{
findmleHSMMnostarting(J, x, obsdist, dwelldist, M = NA, no.initials = 50,
                      verbose = TRUE, seed = NULL, ...)
}
\arguments{
  \item{J}{Integer, number of hidden states (must be at least 2).}
  \item{x}{Numeric vector, observed time series.}
  \item{obsdist}{Character string specifying the observation distribution.
    One of \code{"pois"}, \code{"norm"}, \code{"weibull"}, \code{"zip"},
    \code{"nbinom"}, \code{"zinb"}, \code{"exp"}, \code{"gamma"},
    \code{"lnorm"}, \code{"gev"}, \code{"ZInormal"}, \code{"ZIgamma"}.}
  \item{dwelldist}{Character string specifying the dwell-time distribution.
    One of \code{"pois"}, \code{"nbinom"}, \code{"betabinom"}.}
  \item{M}{Integer, truncation parameter for dwell-time distribution (default \code{NA}, computed automatically).}
  \item{no.initials}{Integer, number of random starting values to try (default 50).}
  \item{verbose}{Logical, if \code{TRUE} (default), progress messages are printed to the console.
    Set to \code{FALSE} to suppress informational output.}
    \item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
  \item{...}{Further arguments to pass to \code{\link{findmleHSMM}}, such as \code{maxiter} and \code{tol}.}
}
\details{
This function automates parameter initialization by sampling plausible random
values for state-dependent observation parameters, dwell-time parameters,
initial distribution \eqn{\delta}, and transition matrix \eqn{\Pi}.
It repeatedly calls \code{\link{findmleHSMM}} with these initializations,
and retains the model with the highest log-likelihood.
When \code{obsdist = "gev"}, initial values are obtained via repeated calls
to \code{evd::fgev} on random data segments until a sufficient number of estimates
are collected.
}
\value{
An object containing:
\item{loglikelihoods}{Numeric vector of log-likelihood values across iterations.}
\item{AIC}{Akaike Information Criteria for the fitted model.}
\item{BIC}{Bayesian Information Criteria for the fitted model.}
\item{delta}{Numeric vector. Estimated initial state probabilities.}
\item{Pi}{Matrix. Estimated \code{J x J} transition probability matrix.}
\item{dwellparameters}{List. Estimated dwell-time distribution parameters.}
\item{observationparameters}{List. Estimated observation distribution parameters.}
}
\examples{
set.seed(321)
J <- 3
Pi <- matrix(c(0.0, 0.6, 0.4,
               0.5, 0.0, 0.5,
               0.3, 0.7, 0.0), nrow = J, byrow = TRUE)
obspar <- list(
  mean = c(-2, 0, 3),
  sd   = c(1, 1.5, 2)
)
dwellpar <- list(
  lambda = c(3, 5, 4)
)
sim <- generateHSMM(n = 200, J = J, obsdist = "norm",
                    dwelldist = "pois", obspar = obspar,
                    dwellpar = dwellpar, Pi = Pi)
\donttest{fit <- findmleHSMMnostarting(J = J, x = sim$x,
                                       obsdist = "norm", dwelldist = "pois",
                                       M = 100, no.initials = 30)
fit$observationparameters
fit$dwellparameters
fit$Pi
fit$delta
fit$loglikelihoods
}
}
\seealso{
\code{\link{findmleHSMM}}, for fitting an HSMM with user-supplied initial values.
\code{\link{generateHSMM}}, for simulating HSMM data.
\code{\link{findmleHMMnostarting}}, for the non-initialised estimation of hidden Markov models.
}
\author{Aimee Cody}
