#' Example Germination Data for PSRI Softmax Calculation
#'
#' A synthetic dataset modeled after prion-exposed germination experiments
#' with corn and barley. Contains time-series germination counts and
#' radicle emergence data across multiple treatments and replicates.
#'
#' @format A data frame with 40 rows and 8 columns:
#' \describe{
#'   \item{species}{Character. Crop species (\code{"corn"} or
#'     \code{"barley"}).}
#'   \item{treatment}{Character. Treatment group (\code{"control"},
#'     \code{"low"}, \code{"medium"}, \code{"high"},
#'     \code{"positive"}).}
#'   \item{replicate}{Integer. Replicate number (1-4).}
#'   \item{total_seeds}{Integer. Total seeds per replicate (25).}
#'   \item{day3}{Integer. Cumulative germination count at day 3.}
#'   \item{day5}{Integer. Cumulative germination count at day 5.}
#'   \item{day7}{Integer. Cumulative germination count at day 7.}
#'   \item{radicle_count}{Integer. Number of seeds with radicle emergence
#'     at day 7.}
#' }
#'
#' @details
#' The dataset demonstrates key properties relevant to PSRI method
#' selection:
#' \itemize{
#'   \item \strong{Corn}: Generally high germination rates with clear
#'     treatment effects; suitable for both geometric and softmax methods.
#'   \item \strong{Barley}: Low germination rates with frequent zero
#'     counts; demonstrates zero-collapse in geometric PSRI and the
#'     advantage of softmax aggregation.
#' }
#'
#' @examples
#' data(germination_example)
#' head(germination_example)
#'
#' # Compute PSRI_SM for first replicate
#' row1 <- germination_example[1, ]
#' compute_psri_sm(
#'   germination_counts = c(row1$day3, row1$day5, row1$day7),
#'   timepoints = c(3, 5, 7),
#'   total_seeds = row1$total_seeds,
#'   radicle_count = row1$radicle_count,
#'   return_components = TRUE
#' )
#'
#' # Batch computation
#' germination_example$psri_sm <- apply(germination_example, 1, function(r) {
#'   compute_psri_sm(
#'     germination_counts = as.numeric(c(r["day3"], r["day5"], r["day7"])),
#'     timepoints = c(3, 5, 7),
#'     total_seeds = as.numeric(r["total_seeds"]),
#'     radicle_count = as.numeric(r["radicle_count"])
#'   )
#' })
#' boxplot(psri_sm ~ treatment, data = germination_example[
#'   germination_example$species == "corn", ],
#'   main = "Corn PSRI_SM by Treatment")
#'
#' @source Synthetic data modeled after prion-germination experiments
#'   conducted at the Minnesota Center for Prion Research and Outreach
#'   (MNPRO), University of Minnesota.
"germination_example"
