% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/testEdges.R
\name{testEdges}
\alias{testEdges}
\title{Test edges from SCORPION networks}
\usage{
testEdges(
  networksDF,
  testType = c("single", "two.sample"),
  group1,
  group2 = NULL,
  paired = FALSE,
  alternative = c("two.sided", "greater", "less"),
  padjustMethod = "BH",
  minLog2FC = 1e-16,
  moderateVariance = TRUE,
  empiricalNull = TRUE
)
}
\arguments{
\item{networksDF}{A data.frame output from \code{\link{runSCORPION}} containing
TF-target pairs as rows and network identifiers as columns.}

\item{testType}{Character specifying the test type. Options are:
\itemize{
  \item{"single": Single-sample test (one-sample t-test against zero)}
  \item{"two.sample": Two-sample comparison (t-test between two groups)}
}}

\item{group1}{Character vector of column names in \code{networksDF} representing
the first group (or the only group for single-sample tests).}

\item{group2}{Character vector of column names in \code{networksDF} representing
the second group. Required for two-sample tests, ignored for single-sample tests.}

\item{paired}{Logical indicating whether to perform a paired t-test. Default FALSE.
When TRUE, group1 and group2 must have the same length and be in matched order
(e.g., group1[1] is paired with group2[1]). Useful for comparing matched samples
such as Tumor vs Normal from the same patient.}

\item{alternative}{Character specifying the alternative hypothesis. Options:
"two.sided" (default), "greater", or "less".}

\item{padjustMethod}{Character specifying the p-value adjustment method for multiple
testing correction. See \code{\link[stats]{p.adjust}} for options. Default "BH"
(Benjamini-Hochberg FDR).}

\item{minLog2FC}{Numeric threshold for minimum absolute log2 fold change to
include in testing. For two-sample and paired tests, edges with |log2FoldChange|
below this threshold are excluded. Not applicable for single-sample tests.
Default 1e-16.}

\item{moderateVariance}{Logical indicating whether to apply SAM-style variance
moderation. When TRUE, adds a fudge factor (s0, the median of all standard errors)
to the denominator of the t-statistic. This prevents edges with very small variance
from producing extreme t-statistics, resulting in volcano plots more similar to
limma output. Default TRUE.}

\item{empiricalNull}{Logical indicating whether to estimate the null distribution
empirically from the observed t-statistics. When TRUE, uses the median and MAD
(median absolute deviation) of all t-statistics to recenter and rescale them,
then computes p-values from the standard normal. This is Efron's empirical null
correction (as in locfdr) and is essential when testing millions of correlated
edges. Runs in O(n) time. Default TRUE.}
}
\value{
A data.frame containing:
  \itemize{
    \item{tf: Transcription factor}
    \item{target: Target gene}
    \item{meanEdge: Mean edge weight}
    \item{tStatistic: Test statistic}
    \item{pValue: Raw p-value}
    \item{pAdj: Adjusted p-value}
    \item{For two-sample tests: meanGroup1, meanGroup2, diffMean (Group1 - Group2), cohensD, log2FoldChange}
  }
}
\description{
Performs statistical testing of network edges from runSCORPION output.
Supports single-sample tests (testing if edges differ from zero) and two-sample
tests (comparing edges between two groups).
}
\details{
For single-sample tests, the function tests whether the mean edge weight across
replicates significantly differs from zero using a one-sample t-test.

For two-sample tests, the function compares edge weights between two groups
using Welch's t-test (unequal variances assumed).

For paired tests, the function calculates the difference between matched pairs
and performs a one-sample t-test on the differences (testing if mean difference
differs from zero). This is appropriate when samples are matched (e.g., Tumor
and Normal from the same patient).

Edges are tested independently, and p-values are adjusted for multiple testing
using the specified method.

The function uses fully vectorized computations for efficiency, making it suitable
for large-scale analyses with millions of edges. T-statistics and p-values are
calculated using matrix operations without iteration.
}
\examples{
\dontrun{
# Load test data and build networks by donor and region
# Note: T = Tumor, N = Normal, B = Border regions
data(scorpionTest)
nets <- runSCORPION(
  gexMatrix = scorpionTest$gex,
  tfMotifs = scorpionTest$tf,
  ppiNet = scorpionTest$ppi,
  cellsMetadata = scorpionTest$metadata,
  groupBy = c("donor", "region")
)

# Single-sample test: Test if edges in Tumor region differ from zero
tumor_nets <- grep("--T$", colnames(nets), value = TRUE)  # T = Tumor
results_single <- testEdges(
  networksDF = nets,
  testType = "single",
  group1 = tumor_nets
)

# Two-sample test: Compare Tumor vs Border regions
tumor_nets <- grep("--T$", colnames(nets), value = TRUE)  # T = Tumor
border_nets <- grep("--B$", colnames(nets), value = TRUE)  # B = Border
results_tumor_vs_border <- testEdges(
  networksDF = nets,
  testType = "two.sample",
  group1 = tumor_nets,
  group2 = border_nets
)

# View top differential edges (Tumor vs Border)
head(results_tumor_vs_border[order(results_tumor_vs_border$pAdj), ])

# Compare Tumor vs Normal regions
normal_nets <- grep("--N$", colnames(nets), value = TRUE)  # N = Normal
results_tumor_vs_normal <- testEdges(
  networksDF = nets,
  testType = "two.sample",
  group1 = tumor_nets,
  group2 = normal_nets
)

# Filter by minimum log2 fold change for focused analysis
results_filtered <- testEdges(
  networksDF = nets,
  testType = "two.sample",
  group1 = tumor_nets,
  group2 = normal_nets,
  minLog2FC = 0.5  # Only test edges with |log2FoldChange| >= 0.5
)

# Paired t-test: Compare matched Tumor vs Normal samples (same patient)
# Ensure columns are ordered by patient: P31--T with P31--N, P32--T with P32--N, etc.
tumor_nets_ordered <- c("P31--T", "P32--T", "P33--T")
normal_nets_ordered <- c("P31--N", "P32--N", "P33--N")
results_paired <- testEdges(
  networksDF = nets,
  testType = "two.sample",
  group1 = tumor_nets_ordered,
  group2 = normal_nets_ordered,
  paired = TRUE
)
}
}
