
#' Weighted ridge regression
#'
#' @name Weightedridge.reg
#'
#' @description
#' Fits a ridge regression model with observation-specific weights. The weights
#' can be supplied as a vector, data frame, or a square weight matrix. If a
#' vector or data frame is supplied, it is internally converted to a diagonal
#' weight matrix.
#' In the example below, the weight vector \code{W} is generated from a
#' Uniform(0, 1) distribution purely to illustrate how to call the function.
#' In practice, users should provide weights that reflect the structure of
#' their data.
#' @param x Explanatory variables. A data.frame or matrix with observations in
#'   rows and predictors in columns.
#' @param y Dependent variable. A numeric vector, data.frame, or matrix. For a
#'   univariate response, this should be a length-\code{n} vector or an
#'   \code{n x 1} matrix.
#' @param W Observation weights. Can be
#'   \itemize{
#'     \item a numeric vector of length \code{n}, or
#'     \item a single-column data.frame of length \code{n}, or
#'     \item an \code{n x n} weight matrix.
#'   }
#'   If \code{W} is a vector or data.frame, the function converts it to
#'   \code{diag(W)} internally.
#'     
#' @return
#' A \code{list} with the following components:
#' \describe{
#'   \item{cc}{Numeric scalar. The selected ridge parameter \code{k}.}
#'   \item{beta}{Numeric matrix (\code{p x 1}). Ridge regression coefficients on the standardized scale (no intercept).}
#'   \item{betaor}{Numeric matrix (\code{(p+1) x 1}). Coefficients on the original (unstandardized) scale, including the intercept in the first row.}
#'   \item{e}{Numeric matrix (\code{n x 1}). Residuals on the standardized scale (\code{yr - yhat}).}
#'   \item{ew}{Numeric matrix (\code{n x 1}). Weighted residuals (\code{W^(1/2) \%*\% e}).}
#'   \item{yhat}{Numeric matrix (\code{n x 1}). Fitted values on the standardized scale (\code{xr \%*\% beta}).}
#'   \item{yhatw}{Numeric matrix (\code{n x 1}). Fitted values in the weighted standardized space (\code{xrw \%*\% beta}).}
#'   \item{yhator}{Numeric matrix (\code{n x 1}). Fitted values on the original scale using \code{betaor}.}
#'   \item{MSE}{Numeric scalar. Mean squared error (MSE) computed from weighted residuals.}
#'   \item{F}{Numeric scalar. Overall model F statistic based on the weighted ANOVA decomposition.}
#'   \item{sig}{Numeric scalar. P-value associated with \code{F}.}
#'   \item{varbeta}{Numeric matrix (\code{p x p}). Estimated covariance matrix of \code{beta} on the standardized scale.}
#'   \item{stdbeta}{Numeric vector (length \code{p}). Standard errors of \code{beta}.}
#'   \item{R2}{Numeric scalar. Weighted coefficient of determination (R-squared).}
#'   \item{R2adj}{Numeric scalar. Adjusted weighted R-squared.}
#'   \item{anovatable}{A \code{data.frame}. ANOVA-style table with sums of squares, degrees of freedom, mean squares, \code{F}, and p-value.}
#'   \item{confint}{Numeric matrix (\code{2 x p}). Confidence intervals for \code{beta}; first row is lower, second row is upper.}
#' }
#'   
#' @export
#'
#' @import isdals
#' @import mctest
#' @import ridgregextra
#' @import Stype.est
#' @importFrom stats sd pf qt
#'
#' @examples
#' ## Example: Weighted ridge regression using the bodyfat data from isdals
#' library(isdals)
#' data(bodyfat)
#'
#' ## Explanatory variables (x) and response (y)
#' x <- bodyfat[ , -1]   # all columns except the first: predictors
#' y <- bodyfat[ ,  1]   # first column: response (body fat percentage)
#'
#' ## Generate observation weights uniformly on [0, 1]
#'
#' n <- nrow(x)
#' W <- runif(n, min = 0, max = 1)
#'
#' ## Fit the weighted ridge regression model
#' fit <- Weightedridge.reg(x, y, W)
#'
#' ## Inspect some key outputs
#' fit$beta        # coefficients in the standardized scale
#' fit$betaor      # coefficients in the original scale (including intercept)
#' fit$R2          # R-squared
#' fit$R2adj       # Adjusted R-squared
#' fit$anovatable  # ANOVA table
#' 
Weightedridge.reg <- function(x,y,W) {
  alpha=0.05
  a=0
  b=1
  if(is.data.frame(W)) W=as.vector(t(W))
  if(is.vector(W)) W=diag(W,nrow=length(W))
  
  m=sum(diag(W))
  
  if (is.vector(x)){
    n=length(x)
    p=1
    x<-matrix(x)
    
  } else {
    n=dim(x)[1]
    p=dim(x)[2]
    
    if(p==1) {
      
      x<-matrix(x)
      
    } else {
      
      colind=2
      xx=cbind(x[,1],x[,2])
      while(colind<p) {
        colind=colind+1
        xx=cbind(xx,x[,colind])
      }
      x=xx }
  }
  
  if (is.vector(y)){
    
    y<-matrix(y)
    
    
  } else {
    cy=dim(y)[2]
    if(cy==1) {
      
      y<-matrix(y)
      
    } else {
      
      colind=2
      yy=cbind(y[,1],y[,2])
      while(colind<cy) {
        colind=colind+1
        yy=cbind(yy,y[,colind])
      }
      y=yy }
  }
  
  ridgeregc= ridgereg_k(x,y,a,b)
  cc=ridgeregc$ridge_reg_results$k
  
  xw <- W^0.5%*%x
  yw <- W^0.5%*%y
  yr <- scale(y)/sqrt(n-1)
  xr <- scale(x)/sqrt(n-1)
  yrw=W^0.5%*%yr
  xrw=W^0.5%*%xr
  X <- xr
  XpX <- t(X)%*%W%*%xr
  XpXplusc <- XpX+cc*diag(p)
  
  Xpy <- t(X)%*%W%*%yr
  invXpXplusc <- solve(XpXplusc)
  beta <- invXpXplusc%*%Xpy
  
  tsdx <- apply(xw,2,sd)
  betaor <- sd(yw)*beta/as.matrix(tsdx)
  beta0 <- c(mean(yw),apply(xw,2,mean))%*%rbind(1,-betaor)
  betaor <- rbind(beta0,betaor)
  yhator <- cbind(matrix(1,n),x)%*%betaor
  
  yhat <- xr%*%beta
  yhatw <- xrw%*%beta
  
  e <- yr-yhat
  ew <- W^(1/2)%*%e
  SSE <- t(ew)%*%ew
  SSE <- as.numeric(SSE)
  MSE <- SSE/(n-(p+1))
  
  s <- 0
  for (i in 1:n) {
    s <- s+W[i,i]*yr[i]
  }
  
  ymeanw <- s/m
  
  SST <- t(yrw)%*%yrw-m*ymeanw^2
  SST <- as.numeric(SST)
  MST <- SST/(n-1)
  
  SSR <- SST-SSE
  SSR <- as.numeric(SSR)
  
  R2 <- SSR/SST
  MSR <- SSR/p
  F <- MSR/MSE
  R2adj <- 1-MSE/MST
  
  sig <- 1-pf(F,p,n-(p+1))
  
  varbeta <- invXpXplusc*MSE
  stdbeta <- sqrt(diag(varbeta))
  #print(cc)
  confint <- rbind(t(beta)-qt(1-alpha/2,n-(p+1))*stdbeta,t(beta)+qt(1-alpha/2,n-(p+1))*stdbeta)
  anovatable <- data.frame("s.v."=c("Regression","Error","Total"),
                           "S.S."=c(SSR,SSE,SST),
                           "d.f."=c(p,n-(p+1),n-1),
                           "M.S."=c(MSR,MSE,MST),
                           "F"=c(F,NA,NA),
                           "sig."=c(sig,NA,NA))
  
  
  z <- list(cc=cc,beta=beta,betaor=betaor,e=e,ew=ew,yhat=yhat,yhatw=yhatw,yhator=yhator,MSE=MSE,F=F,sig=sig,varbeta=varbeta,stdbeta=stdbeta,
            R2=R2,R2adj=R2adj,anovatable=anovatable,confint=confint)
  
  return(z)
}

