#' Process: Recovering Fake Data
#'
#' @description
#' This function processes the synthetic datasets generated by
#'  \code{simulate_list()}. For each of these simulated datasets, it then
#'  fits every model specified within the \code{fit_model} list. In essence,
#'  it iteratively calls the \code{optimize_para()} function for each
#'  generated object.
#'
#' The fitting procedure is analogous to that performed by \code{fit_p},
#'  and it similarly leverages parallel computation across subjects
#'  to significantly accelerate the parameter estimation process.
#' 
#' @param estimate [string] 
#' 
#'   Estimation method. Can be either \code{"MLE"} or \code{"MAP"}.
#'   \itemize{
#'     \item{\strong{Maximum Likelihood Estimation} \code{"MLE"}: (Default): 
#'       This method finds the parameter values that maximize the log-likelihood
#'       of the data. A higher log-likelihood indicates that the parameters
#'       provide a better explanation for the observed human behavior. In
#'       other words, data simulated using these parameters would most
#'       closely resemble the actual human data. This method does not
#'       consider any prior information about the parameters.}
#'       
#'     \item{\strong{Maximum A Posteriori Estimation} \code{"MAP"}: This method
#'       finds the parameter values that maximize the posterior probability.
#'       It is an iterative process based on the Expectation-Maximization
#'       (EM) framework.
#'       
#'       \itemize{
#'         \item{\strong{Initialization}: The process begins by assuming a
#'           uniform distribution as the prior for each parameter, making the
#'           initial log-prior zero. The first optimization is thus
#'           equivalent to MLE.}
#'         \item{\strong{Iteration}: After finding the best parameters for
#'           all subjects, the algorithm assesses the actual distribution of
#'           each parameter and fits a normal distribution to it. This
#'           fitted distribution becomes the new empirical prior.}
#'         \item{\strong{Re-estimation}: The parameters are then re-optimized
#'           to maximize the updated posterior probability.}
#'         \item{\strong{Convergence}: This cycle repeats until the posterior
#'           probability converges or the maximum number of iterations is 
#'           reached.}
#'       }
#'       Using this method requires that the \code{priors} argument
#'       be specified to define the initial prior distributions.
#'     }
#'    }
#' 
#' default: \code{estimate = "MLE"}
#' 
#' @param policy [character]
#' 
#' Specifies the learning policy to be used.
#' This determines how the model updates action values based on observed or
#'   simulated choices. It can be either \code{"off"} or \code{"on"}.
#'   
#'  \itemize{
#'   \item {
#'    \strong{Off-Policy (Q-learning): }
#'    This is the most common approach for modeling
#'     reinforcement learning in Two-Alternative Forced Choice (TAFC) tasks.
#'     In this mode, the model's goal is to learn the underlying value of
#'     each option by observing the human participant's behavior. It achieves
#'     this by consistently updating the value of the option that the
#'     human actually chose. The focus is on understanding the value 
#'     representation that likely drove the participant's decisions.
#'   }
#'   \item {
#'    \strong{On-Policy (SARSA): }
#'    In this mode, the target policy and the behavior policy are identical. 
#'     The model first computes the selection probability for each option based 
#'     on their current values. Critically, it then uses these probabilities to 
#'     sample its own action. The value update is then performed on the action 
#'     that the model itself selected. This approach focuses more on directly 
#'     mimicking the stochastic choice patterns of the agent, rather than just 
#'     learning the underlying values from a fixed sequence of actions.
#'   }
#'  }
#'  
#' default: \code{policy = "off"}
#'  
#' @param list [list] 
#' 
#' A list generated by function \code{simulate_list()}
#' 
#' @param id [vector]
#' 
#' Specifies which subject's data to use. In parameter and model recovery
#'  analyses, the specific subject ID is often irrelevant. Although the
#'  experimental trial order might have some randomness for each subject,
#'  the sequence of reward feedback is typically pseudo-random.
#'
#' The default value for this argument is \code{NULL}. When \code{id = NULL},
#'  the program automatically detects existing subject IDs within the
#'  dataset. It then randomly selects one subject as a sample, and the
#'  parameter and model recovery procedures are performed based on this
#'  selected subject's data.
#'  
#' default: \code{id = NULL}
#' 
#' @param n_trials [integer] 
#' 
#' The total number of trials in your experiment.
#' 
#' @param n_params [integer] 
#' 
#' The number of free parameters in your model. 
#' 
#' @param funcs [character]
#' 
#' A character vector containing the names of all user-defined functions
#'  required for the computation. When parallel computation is enabled
#'  (i.e., `nc > 1`), user-defined models and their custom functions might
#'  not be automatically accessible within the parallel environment.
#'
#' Therefore, if you have created your own reinforcement learning model
#'  that modifies the package's default four default functions 
#'  (default functions: 
#'     \code{util_func = \link[binaryRL]{func_gamma}}, 
#'     \code{rate_func = \link[binaryRL]{func_eta}}, 
#'     \code{expl_func = \link[binaryRL]{func_epsilon}}
#'     \code{bias_func = \link[binaryRL]{func_pi}}
#'     \code{prob_func = \link[binaryRL]{func_tau}}
#'  ), 
#'  you must explicitly provide the names of your custom functions as a 
#'  vector here.
#' 
#' @param model_name [character] 
#' 
#' The name of modal
#'
#' @param fit_model [function] 
#' 
#' fit model object function
#'
#' @param lower [List] 
#' 
#' The lower bounds of model's free parameters.
#' 
#' e.g. \code{lower = c(0, 0, 0)}
#' 
#' @param upper [List] 
#' 
#' The upper bounds of model's free parameters.
#' 
#' e.g. \code{upper = c(1, 1, 5)}
#'
#' @param dfun [List]
#' 
#' A nested list that defines the probability density/mass functions (PDF/PMF)
#'  for each model's parameters. The top-level names of the list must match the
#'  model names. Each element must be another named list, where each name
#'  corresponds to a model parameter and its value is the probability density
#'  function.
#'  
#' e.g., \code{stats::dunif}, \code{stats::dexp}
#'
#' @param initial_params [numeric]
#' 
#' Initial values for the free parameters that the optimization algorithm will
#'  search from. These are primarily relevant when using algorithms that require
#'  an explicit starting point, such as \code{L-BFGS-B}. If not specified,
#'  the function will automatically generate initial values close to zero.
#'  
#'  default: \code{initial_params = NA}.
#'
#' @param initial_size [integer]
#' 
#' This parameter corresponds to the population size in genetic 
#'  algorithms (\code{GA}). It specifies the number of initial candidate
#'  solutions that the algorithm starts with for its evolutionary search.
#'  This parameter is only required for optimization algorithms that operate on
#'  a population, such as `GA` or `DEoptim`. 
#'  
#'  default: \code{initial_size = 50}.
#'  
#' @param tolerance [double] 
#' 
#' Convergence threshold for MAP estimation. If the change in
#'  log posterior probability between iterations is smaller than this value, the
#'  algorithm is considered to have converged and the program will stop.
#' 
#' default: \code{tolerance = 0.001}
#' 
#' @param seed [integer] 
#' 
#' Random seed. This ensures that the results are 
#'  reproducible and remain the same each time the function is run. 
#'  
#'  default: \code{seed = 123}
#'  
#' @param iteration [integer] 
#' 
#' The number of iterations the optimization algorithm will perform
#'  when searching for the best-fitting parameters during the fitting
#'  phase. A higher number of iterations may increase the likelihood of 
#'  finding a global optimum but also increases computation time.
#'  
#' @param nc [integer]
#' 
#' Number of cores to use for parallel processing. Since fitting
#' optimal parameters for each subject is an independent task,
#' parallel computation can significantly speed up the fitting process:
#' \itemize{
#'   \item \strong{`nc = 1`}: The fitting proceeds sequentially.
#'   Parameters for one subject are fitted completely before moving
#'   to the next subject.
#'   \item \strong{`nc > 1`}: The fitting is performed in parallel
#'   across subjects. For example, if `nc = 4`, the algorithm will
#'   simultaneously fit data for four subjects. Once these are complete,
#'   it will proceed to fit the next batch of subjects (e.g., subjects
#'   5-8), and so on, until all subjects are processed.
#' }
#' 
#'  default: \code{nc = 1}
#' 
#' @param algorithm [character] 
#' Choose an algorithm package from
#'  \code{L-BFGS-B}, \code{GenSA},\code{GA},\code{DEoptim},\code{PSO},
#'  \code{Bayesian}, \code{CMA-ES}.
#'  
#' In addition, any algorithm from the \code{nloptr} package is also
#'  supported. If your chosen \code{nloptr} algorithm requires a local search,
#'  you need to input a character vector. The first element represents
#'  the algorithm used for global search, and the second element represents
#'  the algorithm used for local search.
#'  
#' @return a data frame for parameter recovery and model recovery
#' @examples
#' \dontrun{
#' df_recovery <- recovery_data(
#'   list = list_simulated,
#'   policy = "off",
#'   estimate = "MAP", 
#'   model_name = "RSTD",
#'   fit_model = binaryRL::RSTD,
#'   dfun = list(
#'     etan = function(x) { stats::dunif(x, min = 0, max = 1, log = TRUE) }, 
#'     etap = function(x) { stats::dunif(x, min = 0, max = 1, log = TRUE) }, 
#'     tau = function(x) { stats::dexp(x, rate = 1, log = TRUE) }
#'   ),
#'   lower = c(0, 0, 0),
#'   upper = c(1, 1, 10),
#'   iteration = c(10, 3),
#'   nc = 10,
#'   algorithm = c("NLOPT_GN_MLSL", "NLOPT_LN_BOBYQA")
#' )
#' }
recovery_data <- function(
    policy,
    estimate,   
  
    list,
    id = 1,
    n_trials,
    n_params, 
    
    funcs = NULL,
    model_name,
    fit_model,
    dfun,
    
    lower,
    upper,
    
    initial_params = NA,
    initial_size = 50,
    tolerance,
    seed = 123,
    
    iteration,
    
    nc = 1,
    algorithm
){
############################### [ Col-Name ] ###################################
  
  # 创建一个空数据集, 用于存放结果
  recovery <- data.frame(
    fit_model = rep(model_name, length(list)),
    ACC = NA,
    LogL = NA,
    LogPr = NA,
    LogPo = NA,
    AIC = NA,
    BIC = NA
  )
  
  # 检测是都用同一个被试的题目, 还是每次都更换题目
  if (length(id) == 1) {
    id <- rep(id, length(list))
  }
  
  # 增加放置输入参数的列
  n_input_params <- length(list[[1]]$input)
  
  for (i in 1:n_input_params) {
    recovery[, i + 7] <- NA
    names(recovery)[i + 7] <- paste0("input_param_", i)
  }
  # 增加放置输出参数的列
  n_output_params <- length(lower)
  
  for (i in 1:n_output_params) {
    recovery[, i + 7 + n_input_params] <- NA
    names(recovery)[i + 7 + n_input_params] <- paste0("output_param_", i)
  }

############################### [ Parallel ] ###################################
    
  sys <- Sys.info()[["sysname"]]
  
  if (nc == 1) {
    future::plan(future::sequential)
  } 
  # Windows
  else if (sys == "Windows") {
    future::plan(future::multisession, workers = nc)
  } 
  # macOS
  else if (sys == "Darwin") {  
    future::plan(future::multisession, workers = nc)
  } 
  # Linux
  else if (sys == "Linux") {
    future::plan(future::multicore, workers = nc)
  }
  
  doFuture::registerDoFuture()

############################### [ Progress ] ###################################
  
  # 以迭代次数作为进度条
  n_iterations <- length(list) 
  
  progressr::handlers(progressr::handler_txtprogressbar)
  
  progressr::with_progress({
    
    p <- progressr::progressor(steps = n_iterations)
    
    doRNG::registerDoRNG(seed = seed)
    
    # 初始化(定义)foreach中的i
    i <- NA
    
################################# [ MLE ] ######################################
    
    # 抑制每个线程加载包时的信息
    suppressMessages({
      model_result <- foreach::foreach(
        i = 1:n_iterations, .combine = rbind,
        .packages = "binaryRL", 
        .export = funcs
      ) %dorng% {
        
        data_i <- list[[i]][["data"]]
        
        binaryRL.res <- binaryRL::optimize_para(
          policy = policy,
          estimate = estimate,
          
          data = data_i,
          id = id[i],
          n_trials = n_trials,
          n_params = n_params,
          
          obj_func = fit_model,
          lower = lower,
          upper = upper,
          priors = dfun,

          initial_params = initial_params,
          initial_size = initial_size,
          
          iteration = iteration[[1]],
          seed = seed,
          algorithm = algorithm
        )
        
        row_i <- data.frame(
          matrix(NA, nrow = 1, ncol = 7 + n_input_params + n_output_params)
        )
        row_i[1, 1] <- model_name
        row_i[1, 2] <- binaryRL.res$acc
        row_i[1, 3] <- -binaryRL.res$ll
        row_i[1, 4] <- -binaryRL.res$lpr
        row_i[1, 5] <- -binaryRL.res$lpo
        row_i[1, 6] <- binaryRL.res$aic
        row_i[1, 7] <- binaryRL.res$bic
        
        for (j in 1:n_input_params) {
          row_i[1, 7 + j] <- list[[i]]$input[j]
        }
        for (j in 1:n_output_params) {
          row_i[1, 7 + n_input_params + j] <- binaryRL.res$output[j]
        }
        
        # 更新進度條
        p() 
        
        return(row_i)
      }
    })
  })
  
  # 继承recovery的列名
  colnames(model_result) <- colnames(recovery)
  
################################# [ MAP ] ######################################
  
  # Expectation-Maximization Algorithm
  if (estimate == "MAP") {
    
    # 基于上面第一次的结果更新先验概率
    params_priors <- update_priors(
      model_result = model_result,
      priors = dfun, 
      n_params = n_params,
      param_prefix = "output_param_"
    )
    
    # 计算第一次拟合时的似然后验值
    LogPo <- sum(model_result$LogPo)
    
    # MAP需要iteration输入的是二元素向量
    if (length(iteration) == 2) {
      iteration_i <- iteration[[1]]
      iteration_g <- iteration[[2]]
    }
    # 如果iteration_g > 0, 则说明用户希望进行层级贝叶斯, 至少进行一次EM
    if (iteration_g > 0) {
      delta_LogPo <- tolerance + 1
      message(
        paste0(
          "Starting Expectation-Maximization Algorithm", "\n",
          "Log-Posterior Probability: ", round(LogPo, 2)
        )
      )
    }
    # 如果iteration_g == 0, 则意味着被试想只进行MAP或MLE, 而不是层级贝叶斯
    else {
      delta_LogPo <- 0
    }
    
    # 初始化后验迭代次数
    iteration <- 0
    
    while (abs(delta_LogPo) > tolerance) {
      
      # 进度条
      progressr::handlers(progressr::handler_txtprogressbar)
      
      progressr::with_progress({
        
        # 进度条与此时运行的被试数绑定
        p <- progressr::progressor(steps = n_iterations)
        
        # doRNG保证种子不变
        doRNG::registerDoRNG(seed = seed)
        
        # 初始化(定义)foreach中的j (无意义, 仅为通过R CMD check)
        i <- NA
        
        # 抑制每个线程加载包时的信息
        suppressMessages({
          model_result <- foreach::foreach(
            i = 1:n_iterations, .combine = rbind,
            .packages = c("binaryRL"),
            .export = funcs
          ) %dorng% {
            
            data_i <- list[[i]][["data"]]
            
            binaryRL.res <- binaryRL::optimize_para(
              policy = policy,
              estimate = estimate,
              
              data = data_i,
              id = id[i],
              n_trials = n_trials,
              n_params = n_params,
              
              obj_func = fit_model,
              lower = lower,
              upper = upper,
              priors = params_priors,
              
              initial_params = initial_params,
              initial_size = initial_size,
              
              iteration = iteration_i,
              seed = seed,
              algorithm = algorithm
            )
            
            row_i <- data.frame(
              matrix(NA, nrow = 1, ncol = 7 + n_input_params + n_output_params)
            )
            row_i[1, 1] <- model_name
            row_i[1, 2] <- binaryRL.res$acc
            row_i[1, 3] <- -binaryRL.res$ll
            row_i[1, 4] <- -binaryRL.res$lpr
            row_i[1, 5] <- -binaryRL.res$lpo
            row_i[1, 6] <- binaryRL.res$aic
            row_i[1, 7] <- binaryRL.res$bic
            
            for (j in 1:n_input_params) {
              row_i[1, 7 + j] <- list[[i]]$input[j]
            }
            for (j in 1:n_output_params) {
              row_i[1, 7 + n_input_params + j] <- binaryRL.res$output[j]
            }
            
            # 更新進度條
            p() 
            
            return(row_i)
          }
        })
      })
      
      # 继承recovery的列名
      colnames(model_result) <- colnames(recovery)
      
      # 计算此时的似然后验变化量
      delta_LogPo <- sum(model_result$LogPo) - LogPo
      
      # 更新似然后验
      LogPo <- sum(model_result$LogPo)
      
      message(
        paste0(
          "Log-Posterior Probability: ", round(LogPo, 2),
          ", ",
          "\u0394: ", sign_numbers(delta_LogPo), round(delta_LogPo, 2)
        )
      )
      
      # 修改先验分布
      params_priors <- update_priors(
        model_result = model_result,
        priors = params_priors, 
        n_params = n_params,
        param_prefix = "output_param_"
      )
      
      # EM也不会无限制跑下去
      iteration <- iteration + 1
      if (iteration >= iteration_g) {
        break 
      }
    }
  }
  
  # 继承recovery的列名
  colnames(model_result) <- colnames(recovery)
  
  recovery <- model_result

  # 取消注册的线程
  future::plan(future::sequential)

  return(recovery)
}
