% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sensemakr.R
\name{run_sensemakr}
\alias{run_sensemakr}
\title{Sensitivity Analysis to Unobserved Confounding (sensemakr)}
\usage{
run_sensemakr(DT)
}
\arguments{
\item{DT}{A \code{data.frame} or \code{data.table} containing at least
the columns \code{I}, \code{C}, \code{trans_FC}, \code{t_norm},
\code{PopDensity}, and \code{War}.}
}
\value{
A list with components:
\itemize{
\item \code{I}: the \code{sensemakr} object for the model with
outcome \code{I}.
\item \code{C}: the \code{sensemakr} object for the model with
outcome \code{C}.
}
}
\description{
Performs the Cinelli & Hazlett style sensitivity analysis using
\pkg{sensemakr} for two linear models:
\itemize{
\item \code{I ~ trans_FC + t_norm + PopDensity + War}
\item \code{C ~ trans_FC + t_norm + PopDensity + War}
}
treating \code{trans_FC} as the exposure of interest and using
\code{PopDensity} and \code{War} as benchmark covariates.
}
\details{
For each outcome (\code{I} and \code{C}), an OLS model is estimated and
passed to \code{sensemakr::sensemakr()} with:
\itemize{
\item \code{treatment = "trans_FC"}
\item \code{benchmark_covariates = c("PopDensity", "War")}
}
The resulting \code{sensemakr} objects are summarized via
\code{summary()}, converted to data frames, and written to CSV files:
\itemize{
\item \code{"sensemakr_I_FC.csv"} for outcome \code{I}.
\item \code{"sensemakr_C_FC.csv"} for outcome \code{C}.
}

The function assumes that a global character scalar \code{dir_csv} is
defined and points to the directory where CSV outputs should be saved.
}
\examples{
\donttest{
library(data.table)

# 1. Create dummy data with ALL columns required by the lm() formulas
DT <- data.table(
  I = rpois(30, lambda = 5),
  C = rpois(30, lambda = 3),
  trans_FC = sample(0:1, 30, replace = TRUE),   # Treatment
  t_norm = rnorm(30),                           # Trend/Time
  PopDensity = rnorm(30),                       # Benchmark Covariate
  War = sample(0:1, 30, replace = TRUE)         # Benchmark Covariate
)

# 2. Define global path using tempdir() (Fixes CRAN policy)
# run_sensemakr writes output to 'dir_csv', so it must be defined.
tmp_dir <- tempdir()
dir_csv <- file.path(tmp_dir, "csv")
if (!dir.exists(dir_csv)) dir.create(dir_csv, recursive = TRUE)

# 3. Run the function
# This requires the 'sensemakr' package to be installed.
res_sense <- run_sensemakr(DT)

# Inspect results
if (!is.null(res_sense$I)) {
  print(summary(res_sense$I))
}
}

}
