% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/helpers.R
\name{fn}
\alias{fn}
\title{Specify a weight function for multiple-membership models}
\usage{
fn(w = w ~ 1/n, c = TRUE)
}
\arguments{
\item{w}{A two-sided formula specifying the weight function. The left-hand side
must be \code{w}; the right-hand side defines the weighting scheme:
\itemize{
  \item Simple: \code{w ~ 1/n} (equal weights based on group size)
  \item Parameterized: \code{w ~ b0 + b1 * tenure} (weights depend on
    member characteristics and estimated parameters)
  \item With group aggregates: \code{w ~ b1 * min(x) + (1-b1) * mean(x)}
    (weights based on group-level summaries; see Details)
}
Parameters must be named \code{b0}, \code{b1}, \code{b2}, etc.}

\item{c}{Logical; if \code{TRUE} (default), weights are normalized to sum to 1
within each group. Set to \code{FALSE} for unnormalized weights.}
}
\value{
A \code{bml_fn} object containing the parsed weight function specification.
}
\description{
Defines how member-level contributions are weighted when aggregating to the
group level (the "micro-macro link"). The weight function can be a simple
formula (e.g., \code{1/n} for equal weights) or can include parameters to be
estimated from the data.
}
\details{
\strong{Weight Function Components:}
\itemize{
  \item \strong{Variables} (e.g., \code{n}, \code{tenure}): Data from your dataset
  \item \strong{Parameters} (e.g., \code{b0}, \code{b1}): Estimated from the data
  \item \strong{Operations}: Standard R arithmetic (\code{+}, \code{-}, \code{*},
    \code{/}, \code{^}, etc.)
}

\strong{Common Weight Functions:}
\itemize{
  \item Equal weights: \code{w ~ 1/n}
  \item Duration-based: \code{w ~ duration}
  \item Flexible parameterized: \code{w ~ b0 + b1 * seniority}
  \item Group aggregates: \code{w ~ b1 * min(x) + (1-b1) * mean(x)}
}

When \code{c = TRUE}, the weights are constrained: \eqn{\sum_{k \in group} w_k = 1}.

\strong{Group-Level Aggregation Functions:}

The weight function supports aggregation functions that compute summaries
within each group (mainid). These are pre-computed in R before passing to JAGS.
Supported functions:
\itemize{
  \item \code{min(var)}, \code{max(var)}: Minimum/maximum value within the group
  \item \code{mean(var)}, \code{sum(var)}: Mean/sum of values within the group
  \item \code{median(var)}, \code{mode(var)}: Median/mode (most frequent) value within the group
  \item \code{sd(var)}, \code{var(var)}, \code{range(var)}: Standard deviation/variance/range (max-min) within the group
  \item \code{first(var)}, \code{last(var)}: First/last value (based on data order)
  \item \code{quantile(var, prob)}: Quantile at probability \code{prob} (0 to 1).
    For example, \code{quantile(x, 0.25)} computes the 25th percentile.
}

Example: \code{fn(w ~ b1 * min(tenure) + (1-b1) * max(tenure))} creates weights
that blend the minimum and maximum tenure within each group, with the blend
controlled by the estimated parameter \code{b1}.

Example with quantile: \code{fn(w ~ quantile(tenure, 0.75) / max(tenure))} uses
the 75th percentile relative to the maximum within each group.

Note: Nested aggregation functions (e.g., \code{min(max(x))}) are not supported.

\strong{JAGS Mathematical Functions:}

The following mathematical functions are passed directly to JAGS and can be
used in weight formulas:
\itemize{
  \item \code{exp}, \code{log}, \code{log10}, \code{sqrt}, \code{abs}, \code{pow}
  \item \code{sin}, \code{cos}, \code{tan}, \code{asin}, \code{acos}, \code{atan}
  \item \code{sinh}, \code{cosh}, \code{tanh}
  \item \code{logit}, \code{ilogit}, \code{probit}, \code{iprobit}, \code{cloglog}, \code{icloglog}
  \item \code{round}, \code{trunc}, \code{floor}, \code{ceiling}
}

Example: \code{fn(w ~ 1 / (1 + (n - 1) * exp(-(b1 * x))))} uses an exponential
decay function where weights depend on member characteristics.

\strong{Ensuring Numerical Stability:}

Weight functions with estimated parameters (\code{b0}, \code{b1}, ...) must
produce bounded, positive values across all plausible parameter values.
Unbounded weight functions can cause the MCMC sampler to crash
(e.g., \code{"Error in node w.1[...]: Invalid parent values"}).
During sampling, weight parameters can take on extreme values, and if the
weight function is not bounded, this will destabilize the likelihood.

Recommendations:
\itemize{
  \item \strong{Use bounded weight functions.} Two options:
    \itemize{
      \item \code{ilogit()}: Bounds weights between 0 and 1 with a zero-point
        at 0.5: \code{fn(w ~ ilogit(b0 + b1 * x), c = TRUE)}
      \item \strong{Generalized logistic} (Rosche, 2026): Bounds weights between
        0 and 1 with a zero-point at \eqn{1/n} (equal weights), so deviations
        from equal weighting are estimated as a function of covariates:
        \code{fn(w ~ 1 / (1 + (n - 1) * exp(-(b0 + b1 * x))), c = TRUE)}
    }
  \item \strong{Use \code{c = TRUE}} (weight normalization) to prevent weights
    from growing without bound
  \item \strong{Standardize covariates} in the weight function. Variables with
    large ranges (e.g., income in thousands) can cause \code{b * x} to overflow
  \item \strong{Use informative priors} for weight parameters via the
    \code{priors} argument in \code{\link{bml}} (e.g.,
    \code{priors = list("b.w.1[1] ~ dnorm(0, 1)")})
  \item \strong{Avoid unbounded functions} like \code{exp(b * x)} without
    normalization (\code{c = TRUE}) or wrapping (e.g., inside \code{ilogit()})
}

Weight parameters are initialized at 0 by default to ensure numerically stable
starting values. See \code{vignette("faq")} (Question 7) for detailed
troubleshooting of numerical issues.
}
\examples{
\donttest{
# Equal weights (standard multiple-membership)
fn(w ~ 1/n, c = TRUE)

# Tenure-based weights (proportional to time served)
fn(w ~ tenure, c = TRUE)

# Flexible parameterized weights
fn(w ~ b0 + b1 * seniority, c = TRUE)

# Unconstrained weights
fn(w ~ importance, c = FALSE)

# Weights based on group aggregates
fn(w ~ b1 * min(tenure) + (1 - b1) * mean(tenure), c = TRUE)

# Combining individual and aggregate measures
fn(w ~ b0 + b1 * (tenure / max(tenure)), c = TRUE)

# Using median for robust central tendency
fn(w ~ tenure / median(tenure), c = TRUE)

# Using quantiles for percentile-based weights
fn(w ~ quantile(tenure, 0.75) - quantile(tenure, 0.25), c = TRUE)
}

}
\references{
Rosche, B. (2026). A Multilevel Model for Theorizing and Estimating the
Micro-Macro Link. \emph{Political Analysis}.

Browne, W. J., Goldstein, H., & Rasbash, J. (2001). Multiple membership
multiple classification (MMMC) models. \emph{Statistical Modelling}, 1(2), 103-124.
}
\seealso{
\code{\link{mm}}, \code{\link{bml}}, \code{vignette("model")} for
  the model structure, \code{vignette("faq")} for troubleshooting
}
