% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quick_chisq.R
\name{quick_chisq}
\alias{quick_chisq}
\title{Quick Chi-Square Test with Automatic Visualization}
\usage{
quick_chisq(
  data,
  var1,
  var2,
  method = c("auto", "chisq", "fisher", "mcnemar"),
  correct = NULL,
  conf.level = 0.95,
  plot_type = c("bar_grouped", "bar_stacked", "heatmap"),
  show_p_value = TRUE,
  p_label = c("p.format", "p.signif"),
  palette = "qual_vivid",
  verbose = TRUE,
  ...
)
}
\arguments{
\item{data}{A data frame containing the variables.}

\item{var1}{Column name for the first categorical variable (row variable).
Supports both quoted and unquoted names via NSE.}

\item{var2}{Column name for the second categorical variable (column variable).
Supports both quoted and unquoted names via NSE.}

\item{method}{Character. Test method: "auto" (default), "chisq", "fisher", or "mcnemar".
When "auto", the function intelligently selects based on expected frequencies and table size.
\strong{WARNING}: "mcnemar" is ONLY for paired/matched data (e.g., before-after measurements
on the same subjects). It tests marginal homogeneity, NOT independence. Do NOT use McNemar's
test for independent samples - use "chisq" or "fisher" instead.}

\item{correct}{Logical or \code{NULL}. Apply Yates' continuity correction?
If \code{NULL} (default), automatically applied for 2x2 tables with expected frequencies < 10.}

\item{conf.level}{Numeric. Confidence level for the interval. Default is 0.95.}

\item{plot_type}{Character. Type of plot: "bar_grouped" (default),
"bar_stacked", or "heatmap".}

\item{show_p_value}{Logical. Display p-value on the plot? Default is \code{TRUE}.}

\item{p_label}{Character. P-value label format: "p.format" (numeric p-value, default) or
"p.signif" (stars).}

\item{palette}{Character. Color palette name from evanverse palettes.
Default is "qual_vivid". Set to \code{NULL} to use ggplot2 defaults.}

\item{verbose}{Logical. Print diagnostic messages? Default is \code{TRUE}.}

\item{...}{Additional arguments (currently unused, reserved for future extensions).}
}
\value{
An object of class \code{quick_chisq_result} containing:
\describe{
\item{plot}{A ggplot object with the association visualization}
\item{test_result}{The htest object from \code{chisq.test()} or \code{fisher.test()}}
\item{method_used}{Character string of the test method used}
\item{contingency_table}{The contingency table (counts)}
\item{expected_freq}{Matrix of expected frequencies}
\item{pearson_residuals}{Pearson residuals for each cell}
\item{effect_size}{Cramer's V effect size measure}
\item{descriptive_stats}{Data frame with frequencies and proportions}
\item{auto_decision}{Details about automatic method selection}
\item{timestamp}{POSIXct timestamp of analysis}
}
}
\description{
Perform chi-square test of independence or Fisher's exact test (automatically
selected based on expected frequencies) with publication-ready visualization.
Designed for analyzing the association between two categorical variables.
}
\details{
\strong{"Quick" means easy to use, not simplified or inaccurate.}

This function performs full statistical testing with proper assumption checking:

\subsection{Automatic Method Selection (method = "auto")}{
The function uses an intelligent algorithm based on expected frequencies:

\itemize{
\item \strong{All expected frequencies >= 5}: Standard chi-square test
\item \strong{2x2 table with any expected frequency < 5}: Fisher's exact test
\item \strong{Larger table with expected frequency < 5}: Chi-square with warning
\item \strong{2x2 table with 5 <= expected frequency < 10}: Chi-square with Yates' correction
}
}

\subsection{Effect Size}{
Cramer's V is calculated as a measure of effect size:
\itemize{
\item Small effect: V = 0.1
\item Medium effect: V = 0.3
\item Large effect: V = 0.5
}
}

\subsection{Pearson Residuals}{
Pearson residuals are calculated for each cell as (observed - expected) / sqrt(expected):
\itemize{
\item Values > |2| indicate significant deviation from independence
\item Values > |3| indicate very significant deviation
}
}

\subsection{Visualization Options}{
\itemize{
\item \strong{bar_grouped}: Grouped bar chart (default)
\item \strong{bar_stacked}: Stacked bar chart (100\\% stacked)
\item \strong{heatmap}: Heatmap of Pearson residuals
}
}
}
\section{Important Notes}{


\itemize{
\item \strong{Categorical variables}: Both variables must be categorical or will be coerced to factors.
\item \strong{Sample size}: Fisher's exact test may be computationally intensive for large tables.
\item \strong{Missing values}: Automatically removed with a warning.
\item \strong{Low frequencies}: Cells with expected frequency < 5 may lead to unreliable results.
}
}

\examples{
# Example 1: Basic usage with automatic method selection
set.seed(123)
data <- data.frame(
  treatment = sample(c("A", "B", "C"), 100, replace = TRUE),
  response = sample(c("Success", "Failure"), 100, replace = TRUE,
                    prob = c(0.6, 0.4))
)

result <- quick_chisq(data, var1 = treatment, var2 = response)
print(result)

# Example 2: 2x2 table
data_2x2 <- data.frame(
  gender = rep(c("Male", "Female"), each = 50),
  disease = sample(c("Yes", "No"), 100, replace = TRUE)
)

result <- quick_chisq(data_2x2, var1 = gender, var2 = disease)

# Example 3: Customize visualization
result <- quick_chisq(data,
                      var1 = treatment,
                      var2 = response,
                      plot_type = "bar_grouped",
                      palette = "qual_balanced")

# Example 4: Manual method selection
result <- quick_chisq(data,
                      var1 = treatment,
                      var2 = response,
                      method = "chisq",
                      correct = FALSE)

# Access components
result$plot                      # ggplot object
result$test_result               # htest object
result$contingency_table         # Contingency table
result$pearson_residuals         # Pearson residuals
summary(result)                  # Detailed summary

}
\seealso{
\code{\link[stats]{chisq.test}}, \code{\link[stats]{fisher.test}},
\code{\link{quick_ttest}}, \code{\link{quick_anova}}
}
