% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/diagnostics.R
\name{plot.dfr_dist}
\alias{plot.dfr_dist}
\title{Plot DFR Distribution Functions}
\usage{
\method{plot}{dfr_dist}(
  x,
  data = NULL,
  par = NULL,
  what = c("survival", "hazard", "cumhaz"),
  xlim = NULL,
  n = 200,
  add = FALSE,
  col = "black",
  lwd = 2,
  empirical = TRUE,
  empirical_col = "steelblue",
  ...
)
}
\arguments{
\item{x}{A \code{dfr_dist} object}

\item{data}{Optional data frame with survival data for empirical overlay}

\item{par}{Parameter vector. If NULL, uses object's stored parameters.}

\item{what}{Which function to plot:
\describe{
\item{"survival"}{S(t) = exp(-H(t))}
\item{"hazard"}{h(t) - instantaneous failure rate}
\item{"cumhaz"}{H(t) - cumulative hazard}
}}

\item{xlim}{x-axis limits. If NULL, determined from data or defaults to c(0, 10).}

\item{n}{Number of points for smooth curve (default 200)}

\item{add}{If TRUE, add to existing plot}

\item{col}{Line color for theoretical curve}

\item{lwd}{Line width for theoretical curve}

\item{empirical}{If TRUE and data provided, overlay Kaplan-Meier estimate}

\item{empirical_col}{Color for empirical curve}

\item{...}{Additional arguments passed to plot()}
}
\value{
Invisibly returns the plotted values as a list with elements
\code{t} (time points) and \code{y} (function values).
}
\description{
Visualizes the survival, hazard, or cumulative hazard function of a
DFR distribution. Optionally overlays empirical estimates from data.
}
\details{
When \code{empirical = TRUE} and data is provided, overlays:
\itemize{
\item For survival: Kaplan-Meier estimate (step function)
\item For cumhaz: Nelson-Aalen estimate (step function)
\item For hazard: Kernel-smoothed hazard estimate
}
}
\examples{
# Plot survival function for Weibull distribution
weib <- dfr_weibull(shape = 2, scale = 5)
plot(weib, what = "survival", xlim = c(0, 10))

# Overlay hazard functions for different shapes
plot(weib, what = "hazard", xlim = c(0, 10), col = "blue")
weib_k1 <- dfr_weibull(shape = 1, scale = 5)  # Exponential
plot(weib_k1, what = "hazard", add = TRUE, col = "green")
weib_k3 <- dfr_weibull(shape = 3, scale = 5)  # Steeper wear-out
plot(weib_k3, what = "hazard", add = TRUE, col = "red")
legend("topleft", c("k=2", "k=1 (exp)", "k=3"),
       col = c("blue", "green", "red"), lwd = 2)

# Compare fitted model to data
set.seed(123)
true_weib <- dfr_weibull(shape = 2.5, scale = 10)
sim_data <- data.frame(t = sampler(true_weib)(100), delta = 1)
solver <- fit(dfr_weibull())
result <- solver(sim_data, par = c(2, 8))
fitted_weib <- dfr_weibull(shape = coef(result)[1], scale = coef(result)[2])
plot(fitted_weib, data = sim_data, what = "survival",
     xlim = c(0, max(sim_data$t)), empirical = TRUE)

}
