% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hrf-functions.R
\name{hrf_boxcar}
\alias{hrf_boxcar}
\title{Boxcar HRF (No Hemodynamic Delay)}
\usage{
hrf_boxcar(width, amplitude = 1, normalize = FALSE)
}
\arguments{
\item{width}{Duration of the boxcar window in seconds.}

\item{amplitude}{Height of the boxcar (default: 1).}

\item{normalize}{Logical; if \code{TRUE}, the boxcar is scaled so that its
integral equals 1 (i.e., amplitude = 1/width). This makes the regression
coefficient interpretable as the mean signal in the window.
Default is \code{FALSE}.}
}
\value{
An HRF object that can be used with \code{regressor()} and other
  fmrihrf functions.
}
\description{
Creates a simple boxcar (step function) HRF that is constant within a time
window starting at t=0 and zero outside. Unlike traditional HRFs, this has
no hemodynamic delay - it represents an instantaneous response.
}
\details{
When used in a GLM, the estimated coefficient represents a (weighted) average
of the data within the specified time window. If \code{normalize = TRUE}, the
coefficient directly estimates the mean signal in that window.

For delayed windows (not starting at t=0), use \code{\link{lag_hrf}} to shift
the boxcar in time.
}
\section{Note on durations}{

The \code{width} is fixed when the HRF is created. The \code{duration}
parameter in \code{\link{regressor}()} does \strong{not} modify the boxcar
width---it controls how long the neural input is sustained (which then gets
convolved with this HRF). For trial-varying boxcar widths, use a list of HRFs:
\preformatted{
widths <- c(4, 6, 8)
hrfs <- lapply(widths, function(w) hrf_boxcar(width = w, normalize = TRUE))
reg <- regressor(onsets = c(0, 20, 40), hrf = hrfs)
}
}

\examples{
# Simple boxcar of 5 seconds width
hrf1 <- hrf_boxcar(width = 5)
t <- seq(-1, 10, by = 0.1)
plot(t, evaluate(hrf1, t), type = "s", main = "Simple Boxcar HRF")

# Normalized boxcar - coefficient will estimate mean signal in window
hrf2 <- hrf_boxcar(width = 5, normalize = TRUE)
# integral is now 1, so beta estimates mean(Y[0:5])

# Use in a regressor with trial-varying widths
hrf_short <- hrf_boxcar(width = 4, normalize = TRUE)
hrf_long <- hrf_boxcar(width = 8, normalize = TRUE)
reg <- regressor(onsets = c(0, 20), hrf = list(hrf_short, hrf_long))

# For delayed windows, use lag_hrf decorator
hrf_delayed <- lag_hrf(hrf_boxcar(width = 5), lag = 10)  # Window from 10-15s
}
\seealso{
\code{\link{hrf_weighted}} for weighted/shaped boxcars,
  \code{\link{lag_hrf}} to shift the window in time

Other hrf_functions: 
\code{\link{hrf_basis_lwu}()},
\code{\link{hrf_bspline}()},
\code{\link{hrf_gamma}()},
\code{\link{hrf_gaussian}()},
\code{\link{hrf_inv_logit}()},
\code{\link{hrf_lwu}()},
\code{\link{hrf_mexhat}()},
\code{\link{hrf_sine}()},
\code{\link{hrf_spmg1}()},
\code{\link{hrf_time}()},
\code{\link{hrf_weighted}()}
}
\concept{hrf_functions}
