% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parse_front_matter.R
\name{parse_front_matter}
\alias{parse_front_matter}
\alias{read_front_matter}
\title{Parse YAML or TOML Front Matter}
\usage{
parse_front_matter(text, parse_yaml = NULL, parse_toml = NULL)

read_front_matter(path, parse_yaml = NULL, parse_toml = NULL)
}
\arguments{
\item{text}{A character string or vector containing the document text. If a
vector with multiple elements, they are joined with newlines (as from
\code{readLines()}).}

\item{parse_yaml, parse_toml}{A function that takes a string and returns a
parsed R object, or \code{NULL} to use the default parser. Use \code{identity} to
return the raw string without parsing.}

\item{path}{A character string specifying the path to a file. The file is
assumed to be UTF-8 encoded. A UTF-8 BOM (byte order mark) at the start
of the file is automatically stripped if present.}
}
\value{
A named list with two elements:
\itemize{
\item \code{data}: The parsed front matter as an R object, or \code{NULL} if no valid
front matter was found.
\item \code{body}: The document content after the front matter, with leading empty
lines removed. If no front matter is found, this is the original text.
}
}
\description{
Extract and parse YAML or TOML front matter from a file or a text string.
Front matter is structured metadata at the beginning of a document, delimited
by fences (\verb{---} for YAML, \verb{+++} for TOML). \code{parse_front_matter()} processes
a character string, while \code{read_front_matter()} reads from a file. Both
functions return a list with the parsed front matter and the document body.
}
\section{Functions}{
\itemize{
\item \code{parse_front_matter()}: Parse front matter from text

\item \code{read_front_matter()}: Parse front matter from a file.

}}
\section{Custom Parsers}{


By default, the package uses \code{\link[yaml12:parse_yaml]{yaml12::parse_yaml()}} for YAML and
\code{\link[tomledit:read]{tomledit::parse_toml()}} for TOML. You can provide custom parser functions
via \code{parse_yaml} and \code{parse_toml} to override these defaults.

Use \code{identity} to return the raw YAML or TOML string without parsing.
}

\section{YAML Specification Version}{


The default YAML parser uses YAML 1.2 via \code{\link[yaml12:parse_yaml]{yaml12::parse_yaml()}}. To use
YAML 1.1 parsing instead (via \code{\link[yaml:yaml.load]{yaml::yaml.load()}}), set either:
\itemize{
\item The R option \code{frontmatter.parse_yaml.spec} to \code{"1.1"}
\item The environment variable \code{FRONTMATTER_PARSE_YAML_SPEC} to \code{"1.1"}
}

The option takes precedence over the environment variable. Valid values are
\code{"1.1"} and \code{"1.2"} (the default).

YAML 1.1 differs from YAML 1.2 in several ways, most notably in how it
handles boolean values (e.g., \code{yes}/\code{no} are booleans in 1.1 but strings
in 1.2).
}

\examples{
# Parse YAML front matter
text <- "---
title: My Document
date: 2024-01-01
---
Document content here"

result <- parse_front_matter(text)
result$data$title  # "My Document"
result$body        # "Document content here"

# Parse TOML front matter
text <- "+++
title = 'My Document'
date = 2024-01-01
+++
Document content"

result <- parse_front_matter(text)

# Get raw YAML without parsing
result <- parse_front_matter(text, parse_yaml = identity)

# Use a custom parser that adds metadata
result <- parse_front_matter(
  text,
  parse_yaml = function(x) {
    data <- yaml12::parse_yaml(x)
    data$parsed_at <- Sys.time()
    data
  }
)

# Or read from a file
tmpfile <- tempfile(fileext = ".md")
writeLines(text, tmpfile)

read_front_matter(tmpfile)

}
