% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grangersearch.R
\name{granger_causality_test}
\alias{granger_causality_test}
\title{Perform a Granger Causality Test on Two Time Series}
\usage{
granger_causality_test(.data = NULL, x, y, lag = 1, alpha = 0.05, test = "F")
}
\arguments{
\item{.data}{A data frame, tibble, or NULL. If provided, \code{x} and \code{y} are
evaluated as column names within this data frame (tidyverse-style).}

\item{x}{Either a numeric vector/time series, or (if \code{.data} is provided)
an unquoted column name.}

\item{y}{Either a numeric vector/time series of the same length as \code{x}, or
(if \code{.data} is provided) an unquoted column name.}

\item{lag}{Integer. The lag order for the VAR model. Default is 1.}

\item{alpha}{Numeric. Significance level for the causality test (between 0 and 1).
Default is 0.05.}

\item{test}{Character. Type of test to perform. Currently only "F" (F-test) is
supported. Default is "F".}
}
\value{
An object of class \code{granger_result} containing:
\describe{
\item{x_causes_y}{Logical. TRUE if X Granger-causes Y at the specified alpha level.}
\item{y_causes_x}{Logical. TRUE if Y Granger-causes X at the specified alpha level.}
\item{p_value_xy}{Numeric. P-value for the test of X causing Y.}
\item{p_value_yx}{Numeric. P-value for the test of Y causing X.}
\item{test_statistic_xy}{Numeric. Test statistic for X causing Y.}
\item{test_statistic_yx}{Numeric. Test statistic for Y causing X.}
\item{lag}{Integer. The lag order used.}
\item{alpha}{Numeric. The significance level used.}
\item{test}{Character. The test type used.}
\item{n}{Integer. Number of observations.}
\item{x_name}{Character. Name of the X variable.}
\item{y_name}{Character. Name of the Y variable.}
\item{call}{The matched call.}
}
}
\description{
Tests whether one time series Granger-causes another and vice versa.
A variable X is said to Granger-cause Y if past values of X help predict Y
beyond what past values of Y alone provide.
}
\details{
The Granger causality test is based on the idea that if X causes Y, then past
values of X should contain information that helps predict Y above and beyond
the information contained in past values of Y alone.

This function fits Vector Autoregressive (VAR) models using the \pkg{vars}
package and performs F-tests to compare restricted and unrestricted models.
The test is performed in both directions to detect unidirectional or
bidirectional causality.

Note that Granger causality is a statistical concept based on prediction and
temporal precedence. It does not necessarily imply true causal mechanisms.
}
\section{Tidyverse Compatibility}{

This function supports tidyverse-style syntax:
\itemize{
\item Pipe-friendly: use with \verb{\%>\%} or \verb{|>}
\item NSE column selection: pass unquoted column names when using a data frame
\item Use \code{\link[=tidy.granger_result]{tidy.granger_result()}} to get a tibble of results
\item Use \code{\link[=glance.granger_result]{glance.granger_result()}} for model-level summary
}
}

\examples{
# Vector-based usage
set.seed(123)
n <- 100
x <- cumsum(rnorm(n))
y <- c(0, x[1:(n-1)]) + rnorm(n, sd = 0.5)

result <- granger_causality_test(x = x, y = y)
print(result)

# Tidyverse-style with data frame
library(tibble)
df <- tibble(
  price = cumsum(rnorm(100)),
  volume = c(0, cumsum(rnorm(99)))
)

# Using pipe and column names
df |> granger_causality_test(price, volume)

# Get tidy results as tibble
result |> tidy()

# Different lag order
df |> granger_causality_test(price, volume, lag = 2)

}
\references{
Granger, C. W. J. (1969). Investigating Causal Relations by Econometric Models
and Cross-spectral Methods. \emph{Econometrica}, 37(3), 424-438.
}
\seealso{
\code{\link[vars]{VAR}} for the underlying VAR model,
\code{\link[vars]{causality}} for an alternative implementation,
\code{\link[=tidy.granger_result]{tidy.granger_result()}} for tidying results.
}
