% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mar_model.R
\name{mar_model}
\alias{mar_model}
\title{Specify parameters for a Mixture Autoregressive model}
\usage{
mar_model(
  k = NULL,
  p = NULL,
  d = NULL,
  phi = NULL,
  P = NULL,
  D = NULL,
  Phi = NULL,
  constants = NULL,
  sigmas = NULL,
  weights = NULL,
  seasonal_periods = 1L
)
}
\arguments{
\item{k}{Number of components.}

\item{p}{Non-negative integer vector giving the orders of non-seasonal AR polynomials \eqn{\phi_i(B)}.
Ignored if \code{phi} provided.}

\item{d}{Non-negative integer vector giving the orders of non-seasonal differencing.}

\item{phi}{A max(p) x k numeric matrix containing the non-seasonal AR parameters
(\eqn{\phi_{1,i},\dots,\phi_{p,i}}), \eqn{i=1,\dots,k} for each component.}

\item{P}{Non-negative integer giving the orders of seasonal AR polynomiasl \eqn{\Phi_i(B)}.
Ignored if \code{seasonal.periods==1} or \code{Phi} provided.}

\item{D}{Non-negative integer giving the orders of seasonal differencing.
Ignored if \code{seasonal.periods==1}.}

\item{Phi}{A max(P) x k numeric matrix containing the seasonal AR parameters
(\eqn{\Phi_{1,i},\dots,\phi_{P,i}}), \eqn{i=1,\dots,k} for each component.
Ignored if \code{seasonal.periods==1}.}

\item{constants}{A numeric vector of length k containing \eqn{c_1,\dots,c_k}.}

\item{sigmas}{A numeric vector of length k or a list of k GARCH specifications.
If it is a vector, it is assumed \eqn{\sigma_{i,t} = \sigma_i} and
\code{sigmas} = \eqn{\sigma_1,\dots,\sigma_k}.
If it is a list, each element should be the output from \code{fGarch::\link[fGarch]{garchSpec}()}.}

\item{weights}{A numeric vector of length k containing the probability of
each of the component processes, \eqn{\alpha_1,\dots,\alpha_k}.}

\item{seasonal_periods}{Either a scalar or a numeric vector of length k containing
the seasonal period of each component.}
}
\value{
A `mar` object containing a list of \code{k}, \code{m},
\code{p}, \code{d}, \code{P}, \code{D},
\code{phi}, \code{Phi}, \code{sigmas} and \code{weights}.
}
\description{
This function allows the parameters of a mixture of k Gaussian ARIMA(p,d,0)(P,D,0)[m] processes
to be specified. The output is used in \code{\link{simulate.mar}()} and \code{\link{generate.mar}}.
The model is of the form
\deqn{(1-B)^{d_i}(1-B^{m_i})^{D_i} (1-\phi_i(B))(1-\Phi_i(B)) y_t = c_i + \sigma_{i,t}\epsilon_t}
with probability \eqn{\alpha_i}, where \eqn{B} is the backshift operator,
\eqn{m_i} is the seasonal period, \eqn{\epsilon_t} is a N(0,1) variate, and
\eqn{\phi_i(B)} and \eqn{\Phi_i(B)} are polynomials in B of
order \eqn{d_i} and \eqn{D_i} respectively.
If any argument is \code{NULL}, the corresponding parameters are randomly selected.
When randomly selected, the AR parameters are uniformly sampled from the stationary region,
p is in \{0,1,2,3\}, d is in \{0,1,2\}, P is in \{0,1,2\} and D is in \{0,1\}.
The model orders are uniformly sampled. The constants are uniformly sampled on (-3,3).
The sigmas are uniformly sampled on (1,5) and the weights are uniformly sampled on (0,1).
The number of components is uniformly sampled on \{1,2,3,4,5\}.
}
\examples{
\dontshow{if (require("fGarch", quietly=TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
n <- 100
# Quarterly MAR model with randomly selected parameters
model1 <- mar_model(seasonal_periods = 4)

# Daily MAR model with randomly selected parameters
model2 <- mar_model(seasonal_periods = c(7, 365))

# MAR model with constant variances
# containing an AR(1) component and an AR(2) component
phi <- cbind(c(0, 0.8, 0), c(0, 0.6, 0.3))
weights <- c(0.8, 0.2)
model3 <- mar_model(phi = phi, d = 0, sigmas = c(1, 2), weights = weights)

# MAR model with heteroskedastic errors
sigmas.spec <- list(
  fGarch::garchSpec(model = list(alpha = c(0.05, 0.06))),
  fGarch::garchSpec(model = list(alpha = c(0.05, 0.05)))
)
model4 <- mar_model(phi = phi, sigmas = sigmas.spec, weights = weights)
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link{simulate.mar}}
}
\author{
Rob J Hyndman
}
