#' @title Function to solve for the baseline recurrent event rate given a target mean frequency for one-sample simulations.
#' @description
#' Computes the value of the baseline recurrent event rate parameter.
#'
#' @param lambda_star Rate parameter of an exponential distribution in generating the terminal event.
#' @param gamma_shape Shape parameter of the Gamma frailty distribution.
#' @param gamma_scale Scale parameter of the Gamma frailty distribution.
#' @param t Event time (time since enrollment) at which the mean frequency value \code{mu0} is defined.
#' @param mu0 Target value of the mean frequency function at time \code{t}.
#'
#' @returns A numeric value giving the baseline recurrent event rate \eqn{\lambda_0} that satisfies \eqn{\mu(t) = \mu(0)}.
#' @export
#' @importFrom stats uniroot
#'
#' @examples
#'find.lambda_0.given.mu0(lambda_star = 0.1, gamma_shape = 2, gamma_scale = 0.5, t = 1, mu0 = 2)
find.lambda_0.given.mu0 <- function(lambda_star, gamma_shape, gamma_scale, t, mu0){

  f = function(lambda_0){
    (lambda_0/lambda_star)*(1 - (lambda_star*t*gamma_scale + 1)^(-gamma_shape)) - mu0
  }
  return(uniroot(f, c(0, 10))$root)
}
