#-------------------------------------------------------------------------------
#' @encoding UTF-8
#' 
#' @title Hasse diagram of the restricted layout structure
#' @description Returns a Hasse diagram of the restricted layout structure of an experimental design
#'
#' @param datadesign A data frame, list or environment (or object coercible by \code{\link[base]{as.data.frame}} to a data frame) containing the factors in the experimental design. The data frame should \strong{only} include the factors/columns that the user wants to include in the Hasse diagram. All factors are treated as categorical. Moreover, the first two letters of factor names are used for interactions between factors so it is advised that these be unique.
#' @param rand.objects A character vector specifying the randomisation objects 
#' that define the Restricted Layout Structure (RLS).  
#' 
#' The vector must have the same length and order as the structural objects 
#' of the layout structure as produced by \code{\link[hassediagrams]{hasselayout}}. 
#' Each element is either \code{"NULL"} (if there is no randomisation object in 
#' the restricted layout structure that corresponds to the structural object) 
#' or the name of the randomisation object corresponding to the structural object. 
#' The user supplies labels for the randomisation objects
#'
#' The recommended workflow is:
#' \enumerate{
#'   \item Run \code{\link[hassediagrams]{hasselayout}} to obtain the list of structural 
#'         objects (in the order required for \code{\link[hassediagrams]{hasserls}}) 
#'         and a template for the randomisation objects. 
#'   \item Observe the structural objects using 
#'   \code{hasselayout(datadesign)$str_objects}
#'   and create a vector of the randomisation objects potential using the
#'   suggested vector using  \code{hasselayout(datadesign)$rand_template}.
#'   \item Modify only those entries that correspond to randomisation objects 
#'         present in the restricted layout structure.  
#' }
#' The labels specified in \code{rand.objects} represent the labels of the 
#' randomisation objects on the Hasse diagram of the restricted layout structure. 
#' If the labels include Unicode symbols (e.g., the Kronecker product symbol), 
#' a Unicode-friendly font is required. 
#' @param rand.arrows A matrix of two columns that takes integer entries. 
#' Each row of the matrix corresponds to one randomisation arrow on the Hasse diagram of the restricted layout structure. 
#' The entries in the first column contain the object(s) at the start of the randomisation arrow 
#' and the second column contains the object(s) at the end. 
#' The values correspond to the entry number for the randomisation object in \code{rand.objects}.
#' Therefore, any randomisation object(s) involved in the randomisation arrow(s) must first be specified in the \code{rand.objects} argument.
#' The randomisation arrows must point downwards, hence, in each row of the matrix the second column entry must be larger than the first column entry. 
#' @param randomfacsid An optional vector specifying whether the factors are defined as fixed (entry = 0) or random (entry = 1). The default choice is NULL and the function automatically sets all entries to 0. The length of the vector should be equal to the number of factors in the design, i.e., the length of the vector should be equal to the number of columns of the argument \code{datadesign}.
#' @param showRLS logical. If \code{FALSE} then generation of the Hasse diagram of the restricted layout structure is suppressed. The default is TRUE.
#' @param showpartialRLS logical. If \code{FALSE} then the partial crossing between randomisation objects (using dotted connecting lines) is not illustrated  
#' on the Hasse diagram of the restricted layout structure. The default is TRUE.
#' @param showdfRLS logical. If \code{FALSE} then the randomisation object label is not displayed on the Hasse diagram of the restricted layout structure. The default is TRUE.
#' @param showrandRLS logical. If \code{FALSE} then the randomisations are not illustrated (using arrows) on the Hasse diagram of the restricted layout structure. 
#' The default is TRUE. If \code{rand.arrows=NULL}, then \code{showrandRLS} defaults to FALSE.
#' @param check.confound.df logical. If \code{FALSE} then the check for confounded degrees of freedom is not performed. The default is TRUE.
#' @param maxlevels.df logical. If \code{FALSE} then the potential maximum number of levels of a generalised factor is removed from the randomisation object 
#' label on the Hasse diagram of the restricted layout structure. The default is TRUE.
#' @param table.out logical. If \code{TRUE} then a table that shows the relationships between the randomisation objects in the restricted layout structure is printed. 
#' The default is FALSE.
#' @param equation.out logical. If \code{TRUE} then a recommended mixed model to use in the statistical analysis is printed. The default is FALSE.
#' @param pdf logical. If \code{TRUE} then a pdf file containing the Hasse diagram of the restricted layout structure is generated. 
#' The default is FALSE, i.e., a pdf file is not generated.
#' @param example character. Filename for the pdf output file containing the Hasse diagram. The default is set to "example".
#' @param outdir character. Location of the pdf output file if \code{pdf=TRUE}. The default is set to \code{NULL} and in this case the pdf output file 
#' containing the Hasse diagram output is stored to a temporary file. To specify a permanent location this argument needs be specified.
#' @param hasse.font character. The name of the font family used for all text included on the Hasse diagram. 
#' Standard and safe font families to choose are "sans", "serif", and "mono". 
#' If any of the labels of the randomisation objects (as defined in the second column of \code{rand.objects} matrix) 
#' contain Unicode characters, a Unicode friendly font family should be selected. 
#' For more details on Unicode friendly family options see the Details section. 
#' If the font family selected fails to render, the font is automatically changed to "sans" instead.
#' The default is "sans".
#' @param produceBWPlot logical. If \code{TRUE} then the Hasse diagram will be generated in black and white format. 
#' The default is set to FALSE, i.e., a coloured version of the plot is produced.
#' @param structural.colour character. The colour of the structural lines that connect randomisation objects on the Hasse diagram. The default colour is "grey".
#' @param structural.width numeric. The width of the structural lines on the Hasse diagram. The default width is 2.
#' @param partial.colour character. The colour of the partial crossing dotted lines of the connecting randomisation objects on the Hasse diagram. 
#' The default colour is "orange".
#' @param partial.width numeric. The width of the partial crossing dotted lines on the Hasse diagram. The default width is 1.5.
#' @param objects.colour character. The colour of the labels of the randomisation objects on the Hasse diagram. The default colour is "mediumblue".
#' @param df.colour character. The colour of the degrees of the freedom labels on the Hasse diagram. The default colour is "red".
#' @param arrow.colour character. The colour of the randomisation arrows on the Hasse diagram. The default colour is "mediumblue".
#' @param arrow.width numeric. The randomisation arrows width on the Hasse diagram. The default width is 1.5.
#' @param arrow.pos numeric. Specifies the position of the randomisation arrows, i.e., how far the randomisation arrows will be from 
#' the objects they point at. The default is 7.5. A smaller number specifies longer arrows and a higher number specifies shorter arrows.
#' @param larger.fontlabelmultiplier numeric. The large font multiplier is the multiplier for the font used for the labels of objects on the 
#' Hasse diagram where there are four or less objects at that level in the diagram. The default is 1.
#' @param middle.fontlabelmultiplier numeric. The medium font multiplier is the multiplier for the font used for the labels of objects on the 
#' Hasse diagram involving a factor that is equivalent to a generalised factor. The default is 1.
#' @param smaller.fontlabelmultiplier numeric. The small font multiplier is the multiplier for the font used for the labels of objects on the 
#' Hasse diagram where there are five or more objects at that level of the diagram. The default is 1.
#'
#' @return The function \code{\link[hassediagrams]{hasserls}} returns:
#' 1. The Hasse diagram of the restricted layout structure (if \code{showRLS = TRUE}).
#' 
#' 2. The restricted layout structure table shows the relationships between the randomisation objects in the restricted layout structure 
#' (if \code{table.out=TRUE}). The individual entries in the table consist of blanks on the main diagonal and 0’s, (0)’s or 1’s elsewhere. 
#' If the factor (or generalised factor) corresponding to the ith row and the factor (or generalised factor) corresponding to the jth column are fully crossed, 
#' then a 0 is entered in the (i,j)th entry in the table. If these factors (or generalised factors) are partially crossed, or the ith row factor 
#' (or generalised factor) only has one level and nests the jth column factor (or generalised factor), then the (i,j)th entry is (0). 
#' If the ith row factor (or generalised factor) is nested within the jth column factor (or generalised factor), then a 1 is entered in the 
#' (i,j)th entry. If two factors (or generalised factor) are equivalent, then they share a single row and column in the table, 
#' where the row and column headers include both factor (or generalised factor) names, separated by an "=" sign.
#' 
#' 3. An equation that suggests the mixed model to be fitted (if \code{equation.out=TRUE}).
#' 
#' 4. If there are confounded degrees of freedom, a table of the structural objects and a description of the associated degrees of freedom is printed.
#'
#' @details
#' The \code{\link[hassediagrams]{hasserls}} function generates the Hasse diagram of the restricted layout structure. 
#' The Hasse diagram consists of a set of randomisation objects, corresponding to the factors and generalised factors, 
#' and the relationships between the objects (either crossed, nested, partially crossed or equivalent), 
#' as defined by the structure of the experimental design and the randomisation performed, see Bate and Chatfield (2016b).
#' 
#' Where present, two partially crossed factors are illustrated on the diagram with a dotted line connecting them. 
#' This feature can be excluded using the \code{showpartialRLS} option.
#' 
#' The maximum number of possible levels of each generalised factor, along with the actual number present in the design 
#' and the "skeleton ANOVA" degrees of freedom, can be included in the randomisation object label on the Hasse diagram.
#'
#' The randomisation arrows that illustrate the randomisation performed can be included on the Hasse diagram.
#' 
#' The \code{\link[hassediagrams]{hasserls}} function evaluates the design in order to identify if there are any 
#' confounded degrees of freedom across the design. It is not recommended to perform this evaluation for large designs, 
#' due to the potential high computational cost. This can be controlled using the \code{check.confound.df = FALSE} option. 
#' 
#' The rendering of Unicode symbols (e.g., u2297, u2192 for Kronecker symbol and arrow, respectively) in the Hasse diagram depends on the operating system and the font selected in hasse.font.
#' 
#' macOS / Linux:
#' Most system fonts that support Unicode work directly in plotting and PDF output without explicit registration. In many cases, the default "sans" family is sufficient for PDF rendering of these symbols.
#' However, for on-screen rendering usually unicode-friendly fonts like "AppleMyungjo", "Arial Unicode MS", .SF Compact, and "Noto Sans Math" are needed. 
#' 
#' Windows:
#' Base R plotting often requires explicit font registration before the font can be used. Even if the font is installed, it may not be accessible to the graphics device until registered.
#' Unicode-friendly fonts are "Lucida Sans Unicode", "Arial Unicode MS", "Segoe UI Symbol", "Cambria", "Noto Sans Math" and "Ebrima". 
#' The aforementioned fonts may not be available in your R session. The available system fonts can be printed by systemfonts::system_fonts()$family.
#' System available fonts can be imported by running showtext::font_import() or extrafont::font_import().
#' To check which fonts have been successfully imported, run showtext::fonts() or extrafont::fonts().
#' The Arial Unicode MS font can be downloaded from online sources.
#' The Noto Sans Math font can be installed using sysfonts::font_add_google("Noto Sans Math").
#' For Windows, fonts might not be accessible to the graphics device until registered using:
#' windowsFonts(LucidaSansUnicode = windowsFont("Lucida Sans Unicode"))
#' adapted to the font need to use.
#' 
#' For ease of execution in the examples below, we are using --> for the arrow symbol and (x) for the kronecker symbol.
#' 
#' @author
#' Damianos Michaelides, Simon Bate, and Marion Chatfield
#'
#' @references
#' Bate, S.T. and Chatfield, M.J. (2016a), Identifying the structure of the experimental design. Journal of Quality Technology, 48, 343-364.
#' 
#' Bate, S.T. and Chatfield, M.J. (2016b), Using the structure of the experimental design and the randomization to construct a mixed model. Journal of Quality Technology, 48, 365-387.
#' 
#' Box, G.E.P., Hunter, J.S., and Hunter, W.G., (1978), Statistics for Experimenters. Wiley.
#' 
#' Joshi, D.D. (1987), Linear Estimation and Design of Experiments. Wiley Eastern, New Delhi.
#' 
#' Williams, E.R., Matheson, A.C. and Harwood, C.E. (2002), Experimental design and analysis for tree improvement. 2nd edition. CSIRO, Melbourne, Australia.
#'
#' @export
#' 
#' @importFrom stats anova aov as.formula model.matrix.default model.matrix
#' @importFrom utils capture.output
#' 
#' 
#' @examples
#' \donttest{
#' ## NOTE TO USERS:
#' ## In the examples below you may use Unicode symbols (e.g., "u2297 and "u2192"
#' ## with a backslash, for the Kronecker and arrow symbols respectively),
#' ## but we use ASCII fallbacks such as "(x)" and "-->" to ensure
#' ## compatibility across systems.
#' ## To render proper Unicode symbols in diagrams, update the labels manually
#' ## and set a Unicode-friendly font via the hasse.font argument.
#'
#' ### Example: Asphalt concrete production (fractional factorial design)
#' ## Obtain the structural objects from the layout structure
#' ls_concrete <- hasselayout(datadesign = concrete, 
#'                            showLS = FALSE,
#'                            showpartialLS = FALSE,
#'                            showdfLS = FALSE)
#' ## Observe the structural objects and then use the suggested 
#' ## template for randomisation objects
#' ls_concrete$str_objects
#' rand_spec <- ls_concrete$rand_template
#' ## Fill in the randomisation objects that occur in the RLS
#' rand_spec[] <- ls_concrete$str_objects       
#' rand_spec[length(rand_spec)] <- "AC^AG^CC^CoT^CuT --> Run"
#' 
#' ## Generate the Hasse diagram of the restricted layout structure
#' hasserls(datadesign = concrete,
#'          rand.objects = rand_spec,
#'          larger.fontlabelmultiplier = 1.6,
#'          smaller.fontlabelmultiplier = 1.3)
#' 
#'
#' ### Example: Crossover dental study
#' ## Obtain the structural objects from the layout structure
#' ls_dental <- hasselayout(datadesign = dental, 
#'                          randomfacsid = c(0, 1, 0, 0, 0),
#'                          showLS = FALSE,
#'                          showpartialLS = FALSE,
#'                          showdfLS = FALSE)
#' ## Observe the structural objects and then use the suggested 
#' ## template for randomisation objects
#' ls_dental$str_objects
#' rand_spec <- ls_dental$rand_template
#' ## Fill in the randomisation objects that occur in the RLS
#' rand_spec[c(2:5, 7, 8)] <- c("Period", "Sequence",
#'                              "Treatment", "Subject[Sequence]",
#'                              "Period (x) Sequence",
#'                              "Observation")
#' ## Create a matrix for the randomisation arrows 
#' dental_rand_arrows <- matrix(c(3, 5, 4, 7), ncol = 2, byrow = TRUE)
#' ## Generate the Hasse diagram of the restricted layout structure
#' hasserls(datadesign = dental,
#'          rand.objects = rand_spec,
#'          rand.arrows = dental_rand_arrows,
#'          randomfacsid = c(0, 1, 0, 0, 0),
#'          larger.fontlabelmultiplier = 1.6,
#'          arrow.pos = 15)
#'
#'
#' ## Conditionally run examples requiring 'dae'
#' if (requireNamespace("dae", quietly = TRUE)) {
#'   data(BIBDWheat.dat, package = "dae")
#'   BIBDWheat <- BIBDWheat.dat[, -4]
#'   BIBDWheat$Plots <- 1:30
## Obtain the structural objects from the layout structure
#' ls_BIBDWheat <- hasselayout(datadesign = BIBDWheat,
#'                             showLS = FALSE,
#'                             showpartialLS = FALSE,
#'                             showdfLS = FALSE)
#' ## Observe the structural objects and then use the suggested 
#' ## template for randomisation objects
#' ls_BIBDWheat$str_objects
#' rand_spec <- ls_BIBDWheat$rand_template
#' ## Fill in the randomisation objects that occur in the RLS
#' rand_spec[c(2:4)] <- c("Blocks", "Varieties", "Plot[Block]")
#' ## Create a matrix for the randomisation arrows 
#' IBDWheat_rand_arrows <- matrix(c(3, 4), ncol = 2, byrow = TRUE)
#' ## Generate the Hasse diagram of the restricted layout structure
#' hasserls(datadesign = BIBDWheat,
#'          rand.objects = rand_spec,
#'          rand.arrows = IBDWheat_rand_arrows,
#'          equation.out = TRUE)
#' 
#'
#' data(Fac4Proc.dat, package = "dae")
#' Fac4Proc <- Fac4Proc.dat[, -6]
#' ## Obtain the structural objects from the layout structure
#' ls_Fac4Proc <- hasselayout(datadesign = Fac4Proc,
#'                            showLS = FALSE,
#'                            showpartialLS = FALSE,
#'                            showdfLS = FALSE)
#' ## Observe the structural objects and then use the suggested 
#' ## template for randomisation objects
#' ls_Fac4Proc$str_objects
#' rand_spec <- ls_Fac4Proc$rand_template
#' ## Fill in the randomisation objects that occur in the RLS
#' rand_spec[] <- ls_Fac4Proc$str_objects       
#' rand_spec[length(rand_spec)] <- "Catal^Conc^Press^Temp --> Run"
#' ## Generate the Hasse diagram of the restricted layout structure
#' hasserls(datadesign = Fac4Proc,
#'          rand.objects = rand_spec,
#'          showpartialRLS = FALSE,
#'          smaller.fontlabelmultiplier = 2)
#'
#' } else {
#'   message("Install package 'dae' to run the final examples.")
#' }
#' }


hasserls <- function(datadesign,
                     rand.objects,
                     rand.arrows = NULL,
                     randomfacsid = NULL,
                     showRLS = TRUE,
                     showpartialRLS = TRUE,
                     showdfRLS = TRUE,
                     showrandRLS = TRUE,
                     check.confound.df = TRUE,
                     maxlevels.df = TRUE,
                     table.out = FALSE,
                     equation.out = FALSE,
                     pdf = FALSE,
                     example = "example",
                     outdir = NULL,
                     hasse.font = "sans",
                     produceBWPlot = FALSE,
                     structural.colour = "grey",
                     structural.width = 2,
                     partial.colour = "orange",
                     partial.width = 1.5,
                     objects.colour = "mediumblue",
                     df.colour = "red",
                     arrow.colour = "mediumblue",
                     arrow.width = 1.5,
                     arrow.pos = 7.5,
                     larger.fontlabelmultiplier = 1,   
                     middle.fontlabelmultiplier = 1,   
                     smaller.fontlabelmultiplier = 1) {
  
  rls_results <- .build_layout_structure(datadesign = datadesign, randomfacsid = randomfacsid)
  
  logical_args <- list(
    showRLS = showRLS,
    showpartialRLS = showpartialRLS,
    showdfRLS = showdfRLS,
    showrandRLS = showrandRLS,
    check.confound.df = check.confound.df,
    maxlevels.df = maxlevels.df,
    table.out = table.out,
    equation.out = equation.out,
    pdf = pdf,
    produceBWPlot = produceBWPlot
  )
  
  for (arg_name in names(logical_args)) {
    arg_value <- logical_args[[arg_name]]
    if (!is.logical(arg_value) || length(arg_value) != 1) {
      stop(sprintf("Argument '%s' must be a single logical value (TRUE or FALSE).", arg_name), call. = FALSE)
    }
  }
  
  if (!is.character(example) || length(example) != 1) {
    stop("Argument 'example' must be a single character string.", call. = FALSE)
  }
  
  if (!is.null(outdir) && (!dir.exists(outdir))) {
    stop("Argument 'outdir' must be NULL or an existing directory path.", call. = FALSE)
  }
  
  if (!is.character(hasse.font) || length(hasse.font) != 1) {
    stop("Argument 'hasse.font' must be a single character string naming a font family.", call. = FALSE)
  }
  if (!(hasse.font %in% c("sans", "serif", "mono"))) {
    warning("hasse.font is safe to be used for 'sans', 'serif', and 'mono'. \nYour selected font is not in that list, which may lead to potential errors.", call.=FALSE)
  }
  
  colour_args <- list(
    structural.colour = structural.colour,
    partial.colour = partial.colour,
    objects.colour = objects.colour,
    df.colour = df.colour,
    arrow.colour = arrow.colour
  )
  for (arg_name in names(colour_args)) {
    if (!is.character(colour_args[[arg_name]]) || length(colour_args[[arg_name]]) != 1) {
      stop(sprintf("Argument '%s' must be a single character colour name.", arg_name), call. = FALSE)
    }
  }
  
  numeric_args <- list(
    structural.width = structural.width,
    partial.width = partial.width,
    larger.fontlabelmultiplier = larger.fontlabelmultiplier,
    middle.fontlabelmultiplier = middle.fontlabelmultiplier,
    smaller.fontlabelmultiplier = smaller.fontlabelmultiplier,
    arrow.width = arrow.width,
    arrow.pos = arrow.pos
  )
  for (arg_name in names(numeric_args)) {
    arg_value <- numeric_args[[arg_name]]
    if (!is.numeric(arg_value) || length(arg_value) != 1 || is.na(arg_value) || arg_value <= 0) {
      stop(sprintf("Argument '%s' must be a single positive numeric value.", arg_name), call. = FALSE)
    }
  }
  
  
  contains_symbols <- grepl("\u2297", rand.objects, fixed = TRUE) | grepl("\u2192", rand.objects, fixed = TRUE)
  if (any(contains_symbols==TRUE)) {
    warning(
      "The rand.objects argument contains Unicode characters ('\u2297' or '\u2192').\n",
      "Rendering depends on your operating system and font availability:\n",
      " - On Windows, you may need to register the font with windowsFonts(), e.g.,\n",
      "     windowsFonts(LucidaSansUnicode = windowsFont('Lucida Sans Unicode'))\n",
      "   and then set hasse.font = 'LucidaSansUnicode'.\n",
      " - On macOS, most system fonts support Unicode without registration.\n",
      "Also note: PDF output on Windows may not display Unicode correctly with the default 'sans' font,\n",
      "while macOS PDF output typically works with 'sans'.\n",
      "For more guidance, see the Details section of ?hasserls.", call. = FALSE
    )
  }
  
  if (showpartialRLS==TRUE || showdfRLS==TRUE|| showrandRLS==TRUE) showRLS <- TRUE
  
  if(is.null(outdir)) {
    data.folder.location <- tempdir()
  } else {
    data.folder.location <- outdir
  }
  
  if(pdf == TRUE) {
    edgewidth <- structural.width
    dottedline <- 2
    cairo_pdf(filename = file.path(data.folder.location, paste0(example, "_output.pdf")), width = 11, height = 8)
  } else {
    edgewidth <- structural.width
    dottedline <- 2
  }
  
  if (!(table.out %in% c(TRUE, FALSE))) {
    stop("table.out should be Y or N", call. = FALSE)
  }
  
  if (!(equation.out %in% c(TRUE, FALSE))) {
    stop("equation.out should be Y or N", call. = FALSE)
  }
  
  if (any(rand.arrows == 1)) {
    stop("Randomisation arrows should not involve the Mean (should not contain entry 1)", call. = FALSE)
  }

  
  noall <- length(colnames(rls_results$outputlistip1$designi))
  
  nreff <- length(rand.objects) 
  if (rand.objects[nreff] == "NULL") {
    rand.objects[nreff] <- "Obs Unit"
    message("Setting final randomisation object to 'Obs Unit' if was set to 'NULL'.")
  } else {
    rand.objects[nreff] <- rand.objects[nreff]
  }
  if (rand.objects[1] == "NULL") {
    rand.objects[1] <- "Mean"
    message("Setting first randomisation object to 'Mean' if was set to 'NULL'.")
  } else {
    rand.objects[1] <- rand.objects[1]
  }
  
  if (nreff != length(rls_results$finaleffectsnames)) {
    stop("The number of elements in rand.objects should be the same as
          the number of structural objects. Structural objects that do not occur in
          the restricted layout structure you should be retained as NULL.", call. = FALSE)
  }
  
  
  fixed <- which(as.vector(rand.objects) != "NULL")
  briefrename <- as.vector(rand.objects[fixed])
  
  if (!is.null(rand.arrows)) {
    if( !( all(rand.arrows %in% fixed) ) ) stop("There must not be objects that define randomisation arrows which are NULL in the rand.objects", call. = FALSE)
  } 
  
  if (is.null(rand.arrows)) {
    showrandRLS <- FALSE
  }
  
  Colourblue<-"mediumblue"
  Colourred<-"red"
  Colourpurple<-"purple"
  Colourorange<-"orange"
  
  if (produceBWPlot == TRUE) {	
    Colourblue <- "black"
    Colourred <- "black"
    Colourpurple <- "black"
    Colourorange <- "black"
    structural.colour <- "black"
    partial.colour <- "black"
    objects.colour="black"
    df.colour="black"
    arrow.colour="black"
  }
  
  
  old_opts <- options(show.error.messages = FALSE)
  on.exit(options(old_opts), add = TRUE)
  
  if (showRLS==TRUE) {
    
    finaldesign.effects <- rownames(rls_results$finalstructure)[fixed]
    brief.finaldesign.effects <- as.vector(colnames(rls_results$finalstructure)[fixed])
    
    finalnames.wrong <- FALSE 
    if (sum(finaldesign.effects %in% rownames(rls_results$finalstructure)) != length(finaldesign.effects)) {
      message("\nThe selected final terms ", paste(finaldesign.effects[!(finaldesign.effects %in% rownames(rls_results$finalstructure))], collapse = ", "), 
              " are not in the layout structure terms:\n", 
              paste(rownames(rls_results$finalstructure), collapse = ", "))
      finalnames.wrong <- TRUE
    }
    
    if (!is.null(rand.arrows)) {
      nr.row <- nrow(rand.arrows)
      nr.col <- ncol(rand.arrows)
      randomised <- matrix(0, nrow=nr.row, ncol=nr.col)
      for (r in 1:nr.row) {
        for(cc in 1:nr.col) {
          randomised[r,cc] <- rownames(rls_results$finalstructure)[rand.arrows[r,cc]]
        }
      }
    } else{
      randomised <- NULL
    }

    if ((!is.null(randomised)) && (sum(randomised %in% rownames(rls_results$finalstructure)) != length(randomised))) { 
      message("\nThe selected terms indicating the randomisation ", 
              paste(randomised[!(randomised %in% rownames(rls_results$finalstructure))], collapse = ", "), 
              " are not in the objects in the layout structure:\n", 
              paste(rownames(rls_results$finalstructure), collapse = ", "))
      finalnames.wrong <- TRUE
    }
    

    finalceffects.table.final.brief <- rls_results$finalstructure[finaldesign.effects,brief.finaldesign.effects]
    
    #select final effects in final design structure
    finalfinaleffects <- rls_results$finaleffects[rls_results$finaleffectsnames %in% finaldesign.effects]
    
    if (table.out == TRUE) {
      message("\nThe following table shows the relationships between the randomisation objects in the Restricted Layout Structure\n")
      print(finalceffects.table.final.brief)
    } 
    
    names(finalfinaleffects)<- rls_results$finaleffectsnames[rls_results$finaleffectsnames %in% finaldesign.effects]
    
    if (!is.null(rand.arrows)) {
      nr.row <- nrow(rand.arrows)
      nr.col <- ncol(rand.arrows)
      for (aa in 1:nr.row) {
        raa1 <- randomised[aa,1]
        raa2 <- randomised[aa,2]
        level.raa1 <- as.numeric(finalfinaleffects[raa1])
        level.raa2 <- as.numeric(finalfinaleffects[raa2])
        if( identical(level.raa1, level.raa2) ) {
          stop("The two randomisation objects at either end of a randomisation arrow
        cannot be at the same level unless the two objects are equivalent. 
        Make sure that the two entries of the same row in the rand.arrows argument
        are not at the same level.", call. = FALSE)
        }
        
        rbb1 <- rand.arrows[aa,1]
        rbb2 <- rand.arrows[aa,2]
        if(rbb1 > rbb2) {
          rand.arrows[aa,1] <- rbb2
          rand.arrows[aa,2] <- rbb1
          warning("In the Hasse diagram the randomisation arrows must point downwards rather than upwards. 
          Such entries in the rand.arrows argument have been switched automatically.", call. = FALSE)
          randomised[aa,1] <- raa2
          randomised[aa,2] <- raa1
        }
      }
    }
    
    
    #Check that nested effects are random
    names(rls_results$finalrandomeffects)<- rls_results$finaleffectsnames
    typenested <- cbind(rls_results$finalrandomeffects[names(finalfinaleffects)][names(finalfinaleffects) %in% rls_results$nestednames[ ,2]], rls_results$nestednames[ ,2])       
    if(ncol(typenested) > 1) {
      fixednested <- typenested[typenested[ ,1]==0,2]
    } else {
      fixednested <- 0
    }
    
    
    #Constructs the right hand side of equation terms with as.factor in front and replacing nested and * parts of effects with :
    model.equation.final <- model.equation.fun(model.effects.fun(rownames(finalceffects.table.final.brief)))
    if (equation.out == TRUE) {
      message("\nThe suggested mixed model to be fitted is: \n", substring(model.equation.final,6))
    }
    
    #Need a matrix indicating which effects are to be linked by lines (indicated by 1s in the matrix)
    #First treat partially crossed and fully crossed as the same
    fadjm <- matrix(NA,nrow=nrow(finalceffects.table.final.brief),ncol=ncol(finalceffects.table.final.brief),dimnames=dimnames(finalceffects.table.final.brief))
    
    colnames(fadjm) <- briefrename

    for (i in 1:nrow(finalceffects.table.final.brief)) {
      for (j in 1:ncol(finalceffects.table.final.brief)) {
        if (finalceffects.table.final.brief[i,j]=="1") fadjm[i,j] <- 1
        if (finalceffects.table.final.brief[i,j]=="0") fadjm[i,j] <- 0
        if (finalceffects.table.final.brief[i,j]=="(0)") fadjm[i,j] <- 0
        if (finalceffects.table.final.brief[i,j]==" ") fadjm[i,j] <- 0
      }
    }
    
    #Then remove lines which link to nested effects lower in the design e.g. A links to A*B and A*B links to A*B*C but A does not link to A*B*C in diagram.
    fadjm.adjust <- fadjm
    for (j in 1:ncol(fadjm)) {
      for (i in 1:nrow(fadjm)) {
        if (fadjm.adjust[i,j]==1) {
          for (k in 1:nrow(fadjm)) {
            fadjm.adjust[k,j] <- max(fadjm.adjust[k,j]-fadjm[k,i],0)
          }
        }
      }
    }
    
    #This adds dummy variables so that whole width of plotting space is used
    fadjm.adjust <- rbind(aaadum1=c(0), Mean=fadjm.adjust[1, ], zzzdum2=c(0), fadjm.adjust[-1, ])
    fadjm.adjust <- cbind(aaadum1=c(0), Mean=fadjm.adjust[ ,1], zzzdum2=c(0), fadjm.adjust[ ,-1])
    
    #This removes a line if it is to be drawn as a randomised arrow  
    if (showrandRLS==TRUE) {  
      randomisedmat <- matrix(0,nrow=nrow(fadjm.adjust),ncol=ncol(fadjm.adjust),dimnames=list(rownames(fadjm.adjust),rownames(fadjm.adjust)))
      if (!is.null(randomised)) {
        for (i in 1:nrow(randomised)) {
          randomisedmat[randomised[i,2],randomised[i,1]] <- 1
        }
      }
      
      colnames(randomisedmat) <- colnames(fadjm.adjust) 
      
      if (!is.null(briefrename)) {
        colnames(randomisedmat)[c(-1,-3)] <- briefrename
      }       
      
      fadjm.adjust <- fadjm.adjust - randomisedmat
    } 
    
    # For plot to work need to reverse the order of effects                      
    fadjm.reverse <- fadjm.adjust[nrow(fadjm.adjust):1,ncol(fadjm.adjust):1]
    
    if ( any(fadjm.reverse < 0) ) {
      fadjm.reverse[fadjm.reverse < 0] <- 0
      warning("One or more of the arrows do not correspond to structural lines. You may want to check that your randomisation arrows are defined correctly.", call. = FALSE)
    }
    
    fg1 <- graph_from_adjacency_matrix(fadjm.reverse, mode="max")
    
    fg <- simplify(fg1)
    V(fg)$label <- V(fg)$name
    
    #This section calculates the coordinates for the vertices of the Hasse diagram  
    dscoords <- dscoords.fun(DStype="RLS", feffects=finalfinaleffects, ceffects.table.fb=finalceffects.table.final.brief, 
                             larger.fontlabelmultiplier, smaller.fontlabelmultiplier, middle.fontlabelmultiplier)  
    fg$layout <- dscoords$coords
    
    font_used <- hasse.font
    vertex.label.font <- rep(2,length(colnames(fadjm.reverse)))
    vertex.label.color.objects <- c(rep(objects.colour,length(colnames(fadjm.reverse))-3),"transparent",objects.colour,"transparent")
    vertex.label.color.black <- c(rep("black",length(colnames(fadjm.reverse))-3),"transparent","black","transparent")
    vertex.label.color.df <- c(rep(df.colour,length(colnames(fadjm.reverse))-3),"transparent",df.colour,"transparent")
    
    # Set up plot for underlining random effects
    # Default assumes that interaction of two fixed effects is fixed - should allow user to modify
    # Put list identifying random effects in reverse order
    finalfinaleffectrandom <- rls_results$finalrandomeffects[rls_results$finaleffectsnames %in% finaldesign.effects]
    finalfinaleffectrandom.reverse <- c(finalfinaleffectrandom[length(finalfinaleffectrandom):1],0,0)
    
    fadjm.reverse.blank <- fadjm.reverse
    #Replace characters by underscores to produce underlines
    for (m in 1:length(colnames(fadjm.reverse.blank))) {
      if (finalfinaleffectrandom.reverse[m]==1) {
        colnames(fadjm.reverse.blank)[m] <- paste("",paste(rep("_",nchar(colnames(fadjm.reverse.blank)[m])),collapse=""))
      } else {
        colnames(fadjm.reverse.blank)[m] <- ""}
    }
    
    fg2 <- graph_from_adjacency_matrix(fadjm.reverse.blank,mode="max")
    
    fg2a <- simplify(fg2)
    V(fg2a)$label <- V(fg2a)$name
    fg2a$layout <- dscoords$coords
    vcount(fg2a) 
    fg2a.edges <- get.edges(fg2a,1:ecount(fg2a))[ ,1]-1   
    node.dumg <- c(vcount(fg2a)-3,vcount(fg2a)-1) 
    edge.color<-rep(structural.colour,length(fg2a.edges))
    edge.color[fg2a.edges %in% node.dumg]<-"transparent"
    
    if (all(par("mfrow") == c(1, 1))) {
      
      default_mar <- c(5.1, 4.1, 4.1, 2.1)
      current_mar <- par("mar")
      
      if (isTRUE(all.equal(current_mar, default_mar))) {
        
        old_mar <- current_mar
        on.exit(par(mar = old_mar), add = TRUE)
        
        par(mar = c(
          (2 * (max(larger.fontlabelmultiplier, smaller.fontlabelmultiplier) - 1) + 1) * 0.8,
          (5 * (max(larger.fontlabelmultiplier, smaller.fontlabelmultiplier) - 1) + 1) * 0.4,
          0.2,
          (5 * (max(larger.fontlabelmultiplier, smaller.fontlabelmultiplier) - 1) + 1) * 0.4
        ))
      }
    }
    

    tryCatch({
      plot(fg2a, asp=FALSE, add=F,vertex.label.color=vertex.label.color.black, vertex.label.cex=dscoords$textlabel.size, vertex.label.font=vertex.label.font, vertex.label.degree=pi/2, vertex.label.dist=0.4, vertex.size=5, vertex.color="transparent", vertex.shape="circle", vertex.frame.color="white", edge.color=edge.color, edge.width = edgewidth, vertex.label.family=font_used)
    }, error = function(e) {
      message("The font selected in hasse.font is not available in the system's fonts and rendering failed. See the Details section for more information on fonts. The hasse.font is set to 'sans' instead.")
      font_used <<- "sans"
      plot(fg2a, asp=FALSE, add=F,vertex.label.color=vertex.label.color.black, vertex.label.cex=dscoords$textlabel.size, vertex.label.font=vertex.label.font, vertex.label.degree=pi/2, vertex.label.dist=0.4, vertex.size=5, vertex.color="transparent", vertex.shape="circle", vertex.frame.color="white", edge.color=edge.color, edge.width = edgewidth, vertex.label.family=font_used)
    })
    
    
    #-----------------------------------------------------------------------------------------------------------    
    #Identify effects which are partially crossed
    #I have not checked whether the derivation of partially crossed effects is correct 
    #Another way which may be better is by using E(g)[DEFINE HERE WHICH ONES]$lty <- 2
    #Note for can choose type of dotted/dashed line (2-6) if wish
    if (showpartialRLS==TRUE)  {
      fadjm3 <- matrix(0,nrow=nrow(fadjm),ncol=ncol(fadjm),dimnames=dimnames(fadjm))
      for (i in 1:nrow(finalceffects.table.final.brief)) {
        for (j in 1:ncol(finalceffects.table.final.brief)) {
          if (finalceffects.table.final.brief[i,j]=="(0)" && finalceffects.table.final.brief[j,i]=="(0)") fadjm3[i,j] <- 1
        }
      }
      #Then remove lines which link to nested effects lower in the design e.g. A links to A*B and A*B links to A*B*C but A does not link to A*B*C in diagram.
      fadjm3.adjust <- fadjm3
      for (j in 1:ncol(fadjm3)) {
        for (i in 1:nrow(fadjm3)) {
          if (fadjm3.adjust[i,j]==1) {
            for (k in 1:nrow(fadjm3)) {
              fadjm3.adjust[k,j] <- max(fadjm3.adjust[k,j]-fadjm3[k,i],0)
            }
          }
        }
      }
      
      #This adds dummy variables so that whole width of plotting space is used 
      fadjm3.adjust <-rbind(aaadum1=c(0), Mean=fadjm3.adjust[1, ], zzzdum2=c(0), fadjm3.adjust[-1, ])
      fadjm3.adjust <-cbind(aaadum1=c(0), Mean=fadjm3.adjust[ ,1], zzzdum2=c(0), fadjm3.adjust[ ,-1])
      
      # For plot to work need to reverse the order of effects
      fadjm3.reverse <- fadjm3.adjust[nrow(fadjm3.adjust):1,ncol(fadjm3.adjust):1]
      
      fg3 <- graph_from_adjacency_matrix(fadjm3.reverse,mode="max")   #Change this to "directed" for randomization arrows                     
      fg3 <- simplify(fg3)
      V(fg3)$label <- V(fg3)$name
      fg3$layout <- dscoords$coords
      
      # par(mar=c((2*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.8, (5*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.4, 0.2, (5*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.4))     
      
      plot(fg3, asp=FALSE, add=TRUE, vertex.label.color="transparent",vertex.label.cex=dscoords$textlabel.size, vertex.label.font=2, vertex.size=0, vertex.color="transparent", vertex.frame.color="transparent",  edge.label.color=Colourred, edge.label.font=2, edge.color=partial.colour, edge.lty=dottedline, edge.width = partial.width, vertex.label.family=font_used)
      
    }
    
    #Adds names of effects
    # par(mar=c((2*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.8, (5*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.4, 0.2, (5*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.4))     
    
    plot(fg, asp=FALSE, add=T,vertex.label.color=vertex.label.color.objects, vertex.label.cex=dscoords$textlabel.size, vertex.label.font=vertex.label.font, vertex.size=0, vertex.color="transparent", vertex.frame.color="transparent", vertex.shape="circle", edge.lty=0, edge.width = edgewidth, vertex.label.family=font_used)
  }



#Sets up matrix for degrees of freedom, 1st col = Tier, 2nd col=max number of levels, 3rd col = actual number of levels, 4th col = dfs
#Adds degrees of freedom to plot

  if (showdfRLS==TRUE) {
    RLS.output <- dfs.fun("RLS", noall, finalfinaleffects, finalceffects.table.final.brief, fadjm, rls_results$outputlistip1, rls_results$nfactors, maxlevels.df, check.confound.df, datadesign, rls_results$finaleffectsnames)
    RLS.output$xdfs
    fg4fix<-fg
    fg4rand<-fg
    dflab4fix<-NULL
    dflab4rand<-NULL
    for (i in 1: length(V(fg2a)$name)){
      if (V(fg2a)$name[i]=="") {
        dflab4fix[i]<-paste(sep="", "[",RLS.output$xdfs.reverse[i,3],RLS.output$maxlevelsf.reverse[i],",",RLS.output$xdfs.reverse[i,4],"]")
        dflab4rand[i]<-""
      } else {
        dflab4rand[i]<-paste(sep="", "[",RLS.output$xdfs.reverse[i,3],RLS.output$maxlevelsf.reverse[i],",",RLS.output$xdfs.reverse[i,4],"]")
        dflab4fix[i]<-""    
      }
    }
    
    V(fg4fix)$label <- dflab4fix
    V(fg4rand)$label <- dflab4rand
    
    vertex.label.dist.df4fix<-(1*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*1
    vertex.label.dist.df4rand<-(1*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*1
    
    #Add degrees of freedom
    # par(mar=c((2*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.8, (5*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.4, 0.2, (5*(max(larger.fontlabelmultiplier,smaller.fontlabelmultiplier)-1)+1)*0.4))     
    
    plot(fg4fix, asp=FALSE, add=T,vertex.label.color=vertex.label.color.df, vertex.label.cex=dscoords$textlabel.size.df,  vertex.label.font=vertex.label.font, vertex.label.degree=pi/2, vertex.label.dist=vertex.label.dist.df4fix,vertex.size=0, vertex.color="transparent", vertex.frame.color="transparent", vertex.shape="circle", edge.lty=0, edge.width = edgewidth, vertex.label.family=font_used)
    plot(fg4rand, asp=FALSE, add=T,vertex.label.color=vertex.label.color.df, vertex.label.cex=dscoords$textlabel.size.df, vertex.label.font=vertex.label.font, vertex.label.degree=pi/2, vertex.label.dist=vertex.label.dist.df4rand,vertex.size=0, vertex.color="transparent", vertex.frame.color="transparent", vertex.shape="circle", edge.lty=0, edge.width = edgewidth, vertex.label.family=font_used)
  }

  #-----------------------------------------------------------------------------------------------------------    
  #Identify effects which are randomised to others
  #Another way which may be better is by using E(g)[DEFINE HERE WHICH ONES]$lty <- 2
  #Note for can choose type of dotted/dashed line (2-6) if wish
  if (showrandRLS==TRUE) {
    
    # For plot to work need to reverse the order of effects
    randomisedmat.reverse <- randomisedmat[nrow(randomisedmat):1,ncol(randomisedmat):1]
    randomisedmat.reverse
    
    fg5 <- graph_from_adjacency_matrix(randomisedmat.reverse,mode="directed")                      
    fg5 <- simplify(fg5)
    V(fg5)$label <- V(fg5)$name
    fg5$layout <- dscoords$coords
    
    plot(fg5, asp=FALSE, add=TRUE, vertex.label.color="transparent",vertex.label.cex=dscoords$textlabel.size, vertex.label.font=1, vertex.size=arrow.pos, 
         vertex.color="transparent", vertex.frame.color="transparent", edge.color=arrow.colour, edge.lty=2, edge.arrow.mode=1, edge.width=arrow.width, edge.arrow.size=0.4)
  }

  if (pdf==TRUE) hidedevoff <- dev.off()
  
}



