/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Amodule.h" 
#define H5O_FRIEND     

#include "H5private.h"   
#include "H5Apkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5SMprivate.h" 
#include "H5WBprivate.h" 

#define H5A_NAME_BT2_NODE_SIZE  512
#define H5A_NAME_BT2_MERGE_PERC 40
#define H5A_NAME_BT2_SPLIT_PERC 100

#define H5A_CORDER_BT2_NODE_SIZE  512
#define H5A_CORDER_BT2_MERGE_PERC 40
#define H5A_CORDER_BT2_SPLIT_PERC 100

#define H5A_ATTR_BUF_SIZE 128

typedef struct H5A_bt2_od_wrt_t {
    
    H5F_t  *f;               
    H5HF_t *fheap;           
    H5HF_t *shared_fheap;    
    H5A_t  *attr;            
    haddr_t corder_bt2_addr; 
} H5A_bt2_od_wrt_t;

typedef struct {
    
    H5F_t  *f;            
    H5HF_t *fheap;        
    H5HF_t *shared_fheap; 
    hsize_t count;        

    
    hid_t                     loc_id;  
    hsize_t                   skip;    
    const H5A_attr_iter_op_t *attr_op; 
    void                     *op_data; 

    
    int op_ret; 
} H5A_bt2_ud_it_t;

typedef struct {
    
    H5F_t                          *f;      
    const H5A_dense_bt2_name_rec_t *record; 

    
    H5A_t *attr; 
} H5A_fh_ud_cp_t;

typedef struct H5A_bt2_ud_rm_t {
    
    H5A_bt2_ud_common_t common;          
    haddr_t             corder_bt2_addr; 
} H5A_bt2_ud_rm_t;

typedef struct H5A_bt2_ud_rmbi_t {
    
    H5F_t     *f;              
    H5HF_t    *fheap;          
    H5HF_t    *shared_fheap;   
    H5_index_t idx_type;       
    haddr_t    other_bt2_addr; 
} H5A_bt2_ud_rmbi_t;

herr_t
H5A__dense_create(H5F_t *f, H5O_ainfo_t *ainfo)
{
    H5HF_create_t fheap_cparam;         
    H5B2_create_t bt2_cparam;           
    H5HF_t       *fheap      = NULL;    
    H5B2_t       *bt2_name   = NULL;    
    H5B2_t       *bt2_corder = NULL;    
    herr_t        ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);

    
    
    memset(&fheap_cparam, 0, sizeof(fheap_cparam));
    fheap_cparam.managed.width            = H5O_FHEAP_MAN_WIDTH;
    fheap_cparam.managed.start_block_size = H5O_FHEAP_MAN_START_BLOCK_SIZE;
    fheap_cparam.managed.max_direct_size  = H5O_FHEAP_MAN_MAX_DIRECT_SIZE;
    fheap_cparam.managed.max_index        = H5O_FHEAP_MAN_MAX_INDEX;
    fheap_cparam.managed.start_root_rows  = H5O_FHEAP_MAN_START_ROOT_ROWS;
    fheap_cparam.checksum_dblocks         = H5O_FHEAP_CHECKSUM_DBLOCKS;
    fheap_cparam.max_man_size             = H5O_FHEAP_MAX_MAN_SIZE;

    
    if (NULL == (fheap = H5HF_create(f, &fheap_cparam)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "unable to create fractal heap");

    
    if (H5HF_get_heap_addr(fheap, &ainfo->fheap_addr) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGETSIZE, FAIL, "can't get fractal heap address");

#ifndef NDEBUG
    {
        size_t fheap_id_len; 

        
        if (H5HF_get_id_len(fheap, &fheap_id_len) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGETSIZE, FAIL, "can't get fractal heap ID length");
        assert(fheap_id_len == H5O_FHEAP_ID_LEN);
    }
#endif 

    
    memset(&bt2_cparam, 0, sizeof(bt2_cparam));
    bt2_cparam.cls       = H5A_BT2_NAME;
    bt2_cparam.node_size = (size_t)H5A_NAME_BT2_NODE_SIZE;
    bt2_cparam.rrec_size = 4 +               
                           4 +               
                           1 +               
                           H5O_FHEAP_ID_LEN; 
    bt2_cparam.split_percent = H5A_NAME_BT2_SPLIT_PERC;
    bt2_cparam.merge_percent = H5A_NAME_BT2_MERGE_PERC;
    if (NULL == (bt2_name = H5B2_create(f, &bt2_cparam, NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "unable to create v2 B-tree for name index");

    
    if (H5B2_get_addr(bt2_name, &ainfo->name_bt2_addr) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get v2 B-tree address for name index");

    
    if (ainfo->index_corder) {
        
        memset(&bt2_cparam, 0, sizeof(bt2_cparam));
        bt2_cparam.cls       = H5A_BT2_CORDER;
        bt2_cparam.node_size = (size_t)H5A_CORDER_BT2_NODE_SIZE;
        bt2_cparam.rrec_size = 4 +               
                               1 +               
                               H5O_FHEAP_ID_LEN; 
        bt2_cparam.split_percent = H5A_CORDER_BT2_SPLIT_PERC;
        bt2_cparam.merge_percent = H5A_CORDER_BT2_MERGE_PERC;
        if (NULL == (bt2_corder = H5B2_create(f, &bt2_cparam, NULL)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "unable to create v2 B-tree for creation order index");

        
        if (H5B2_get_addr(bt2_corder, &ainfo->corder_bt2_addr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get v2 B-tree address for creation order index");
    } 

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for creation order index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5A__dense_fnd_cb(const H5A_t *attr, bool *took_ownership, void *_user_attr)
{
    const H5A_t **user_attr = (const H5A_t **)_user_attr; 
    herr_t        ret_value = SUCCEED;                    

    FUNC_ENTER_PACKAGE

    
    assert(attr);
    assert(user_attr);
    assert(took_ownership);

    
    if (*user_attr != NULL) {
        H5A_t *old_attr = *(H5A_t **)_user_attr;

        
        if (old_attr->shared)
            if (H5A__shared_free(old_attr) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTRELEASE, FAIL, "can't release attribute info");

        old_attr = H5FL_FREE(H5A_t, old_attr);
    } 

    
    *user_attr      = attr;
    *took_ownership = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5A_t *
H5A__dense_open(H5F_t *f, const H5O_ainfo_t *ainfo, const char *name)
{
    H5A_bt2_ud_common_t udata;               
    H5HF_t             *fheap        = NULL; 
    H5HF_t             *shared_fheap = NULL; 
    H5B2_t             *bt2_name     = NULL; 
    htri_t              attr_sharable;       
    bool                attr_exists;         
    H5A_t              *ret_value = NULL;    

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);
    assert(name);

    
    if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, NULL, "unable to open fractal heap");

    
    if ((attr_sharable = H5SM_type_shared(f, H5O_ATTR_ID)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, NULL, "can't determine if attributes are shared");

    
    if (attr_sharable) {
        haddr_t shared_fheap_addr; 

        
        if (H5SM_get_fheap_addr(f, H5O_ATTR_ID, &shared_fheap_addr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, NULL, "can't get shared message heap address");

        
        if (H5_addr_defined(shared_fheap_addr)) {
            
            if (NULL == (shared_fheap = H5HF_open(f, shared_fheap_addr)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, NULL, "unable to open fractal heap");
        } 
    }     

    
    if (NULL == (bt2_name = H5B2_open(f, ainfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, NULL, "unable to open v2 B-tree for name index");

    
    udata.f             = f;
    udata.fheap         = fheap;
    udata.shared_fheap  = shared_fheap;
    udata.name          = name;
    udata.name_hash     = H5_checksum_lookup3(name, strlen(name), 0);
    udata.flags         = 0;
    udata.corder        = 0;
    udata.found_op      = H5A__dense_fnd_cb; 
    udata.found_op_data = &ret_value;

    
    attr_exists = false;
    if (H5B2_find(bt2_name, &udata, &attr_exists, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, NULL, "can't search for attribute in name index");
    if (attr_exists == false)
        HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, NULL, "can't locate attribute in name index");

done:
    
    if (shared_fheap && H5HF_close(shared_fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, NULL, "can't close fractal heap");
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, NULL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, NULL, "can't close v2 B-tree for name index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5A__dense_insert(H5F_t *f, const H5O_ainfo_t *ainfo, H5A_t *attr)
{
    H5A_bt2_ud_ins_t udata;                       
    H5HF_t          *fheap        = NULL;         
    H5HF_t          *shared_fheap = NULL;         
    H5B2_t          *bt2_name     = NULL;         
    H5B2_t          *bt2_corder   = NULL;         
    H5WB_t          *wb           = NULL;         
    uint8_t          attr_buf[H5A_ATTR_BUF_SIZE]; 
    unsigned         mesg_flags = 0;              
    htri_t           attr_sharable;               
    herr_t           ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);
    assert(attr);

    
    if ((attr_sharable = H5SM_type_shared(f, H5O_ATTR_ID)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't determine if attributes are shared");

    
    if (attr_sharable) {
        haddr_t shared_fheap_addr; 
        htri_t  shared_mesg;       

        
        if ((shared_mesg = H5O_msg_is_shared(H5O_ATTR_ID, attr)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "error determining if message is shared");
        else if (shared_mesg > 0)
            
            mesg_flags |= H5O_MSG_FLAG_SHARED;
        else {
            
            if (H5SM_try_share(f, NULL, 0, H5O_ATTR_ID, attr, &mesg_flags) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_WRITEERROR, FAIL, "error determining if message should be shared");

            
            assert(!(mesg_flags & H5O_MSG_FLAG_SHAREABLE));
        } 

        
        if (H5SM_get_fheap_addr(f, H5O_ATTR_ID, &shared_fheap_addr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get shared message heap address");

        
        if (H5_addr_defined(shared_fheap_addr)) {
            
            if (NULL == (shared_fheap = H5HF_open(f, shared_fheap_addr)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");
        } 
    }     

    
    if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    if (mesg_flags & H5O_MSG_FLAG_SHARED) {
        
        assert(attr_sharable);

        
        udata.id = attr->sh_loc.u.heap_id;
    } 
    else {
        void  *attr_ptr;  
        size_t attr_size; 

        
        if ((attr_size = H5O_msg_raw_size(f, H5O_ATTR_ID, false, attr)) == 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGETSIZE, FAIL, "can't get message size");

        
        if (NULL == (wb = H5WB_wrap(attr_buf, sizeof(attr_buf))))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "can't wrap buffer");

        
        if (NULL == (attr_ptr = H5WB_actual(wb, attr_size)))
            HGOTO_ERROR(H5E_ATTR, H5E_NOSPACE, FAIL, "can't get actual buffer");

        
        if (H5O_msg_encode(f, H5O_ATTR_ID, false, (unsigned char *)attr_ptr, attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTENCODE, FAIL, "can't encode attribute");

        
        
        if (H5HF_insert(fheap, attr_size, attr_ptr, &udata.id) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to insert attribute into fractal heap");
    } 

    
    if (NULL == (bt2_name = H5B2_open(f, ainfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    udata.common.f            = f;
    udata.common.fheap        = fheap;
    udata.common.shared_fheap = shared_fheap;
    udata.common.name         = attr->shared->name;
    udata.common.name_hash    = H5_checksum_lookup3(attr->shared->name, strlen(attr->shared->name), 0);
    H5_CHECKED_ASSIGN(udata.common.flags, uint8_t, mesg_flags, unsigned);
    udata.common.corder        = attr->shared->crt_idx;
    udata.common.found_op      = NULL;
    udata.common.found_op_data = NULL;
    

    
    if (H5B2_insert(bt2_name, &udata) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to insert record into v2 B-tree");

    
    if (ainfo->index_corder) {
        
        assert(H5_addr_defined(ainfo->corder_bt2_addr));
        if (NULL == (bt2_corder = H5B2_open(f, ainfo->corder_bt2_addr, NULL)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for creation order index");

        
        if (H5B2_insert(bt2_corder, &udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to insert record into v2 B-tree");
    } 

done:
    
    if (shared_fheap && H5HF_close(shared_fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for creation order index");
    if (wb && H5WB_unwrap(wb) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close wrapped buffer");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5A__dense_write_bt2_cb2(void *_record, void *_op_data, bool *changed)
{
    H5A_dense_bt2_corder_rec_t *record = (H5A_dense_bt2_corder_rec_t *)_record; 
    H5O_fheap_id_t *new_heap_id        = (H5O_fheap_id_t *)_op_data; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(record);
    assert(new_heap_id);

    
    record->id = *new_heap_id;

    
    *changed = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5A__dense_write_bt2_cb(void *_record, void *_op_data, bool *changed)
{
    H5A_dense_bt2_name_rec_t *record     = (H5A_dense_bt2_name_rec_t *)_record; 
    H5A_bt2_od_wrt_t         *op_data    = (H5A_bt2_od_wrt_t *)_op_data; 
    H5B2_t                   *bt2_corder = NULL;           
    H5WB_t                   *wb         = NULL;           
    uint8_t                   attr_buf[H5A_ATTR_BUF_SIZE]; 
    herr_t                    ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(record);
    assert(op_data);

    
    if (record->flags & H5O_MSG_FLAG_SHARED) {
        
        if (H5O__attr_update_shared(op_data->f, NULL, op_data->attr, NULL) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update attribute in shared storage");

        
        record->id = op_data->attr->sh_loc.u.heap_id;

        
        if (H5_addr_defined(op_data->corder_bt2_addr)) {
            H5A_bt2_ud_common_t udata; 

            
            if (NULL == (bt2_corder = H5B2_open(op_data->f, op_data->corder_bt2_addr, NULL)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for creation order index");

            
            udata.f             = op_data->f;
            udata.fheap         = NULL;
            udata.shared_fheap  = NULL;
            udata.name          = NULL;
            udata.name_hash     = 0;
            udata.flags         = 0;
            udata.corder        = op_data->attr->shared->crt_idx;
            udata.found_op      = NULL;
            udata.found_op_data = NULL;

            
            if (H5B2_modify(bt2_corder, &udata, false, H5A__dense_write_bt2_cb2,
                            &op_data->attr->sh_loc.u.heap_id) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to modify record in v2 B-tree");
        } 

        
        *changed = true;
    } 
    else {
        void  *attr_ptr;  
        size_t attr_size; 

        
        if ((attr_size = H5O_msg_raw_size(op_data->f, H5O_ATTR_ID, false, op_data->attr)) == 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGETSIZE, FAIL, "can't get attribute size");

        
        if (NULL == (wb = H5WB_wrap(attr_buf, sizeof(attr_buf))))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTINIT, FAIL, "can't wrap buffer");

        
        if (NULL == (attr_ptr = H5WB_actual(wb, attr_size)))
            HGOTO_ERROR(H5E_ATTR, H5E_NOSPACE, FAIL, "can't get actual buffer");

        
        if (H5O_msg_encode(op_data->f, H5O_ATTR_ID, false, (unsigned char *)attr_ptr, op_data->attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTENCODE, FAIL, "can't encode attribute");

#ifndef NDEBUG
        {
            size_t obj_len; 

            if (H5HF_get_obj_len(op_data->fheap, &record->id, &obj_len) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTGETSIZE, FAIL, "can't get object size");
            assert(obj_len == attr_size);
        }
#endif 
        
        
        if (H5HF_write(op_data->fheap, &record->id, changed, attr_ptr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTUPDATE, FAIL, "unable to update attribute in heap");
    } 

done:
    
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for creation order index");
    if (wb && H5WB_unwrap(wb) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close wrapped buffer");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5A__dense_write(H5F_t *f, const H5O_ainfo_t *ainfo, H5A_t *attr)
{
    H5A_bt2_ud_common_t udata;               
    H5A_bt2_od_wrt_t    op_data;             
    H5HF_t             *fheap        = NULL; 
    H5HF_t             *shared_fheap = NULL; 
    H5B2_t             *bt2_name     = NULL; 
    htri_t              attr_sharable;       
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);
    assert(H5_addr_defined(ainfo->fheap_addr));
    assert(H5_addr_defined(ainfo->name_bt2_addr));
    assert(attr);

    
    if ((attr_sharable = H5SM_type_shared(f, H5O_ATTR_ID)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't determine if attributes are shared");

    
    if (attr_sharable) {
        haddr_t shared_fheap_addr; 

        
        if (H5SM_get_fheap_addr(f, H5O_ATTR_ID, &shared_fheap_addr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get shared message heap address");

        
        if (H5_addr_defined(shared_fheap_addr)) {
            
            if (NULL == (shared_fheap = H5HF_open(f, shared_fheap_addr)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");
        } 
    }     

    
    if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    if (NULL == (bt2_name = H5B2_open(f, ainfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    udata.f             = f;
    udata.fheap         = fheap;
    udata.shared_fheap  = shared_fheap;
    udata.name          = attr->shared->name;
    udata.name_hash     = H5_checksum_lookup3(attr->shared->name, strlen(attr->shared->name), 0);
    udata.flags         = 0;
    udata.corder        = 0;
    udata.found_op      = NULL;
    udata.found_op_data = NULL;

    
    op_data.f               = f;
    op_data.fheap           = fheap;
    op_data.shared_fheap    = shared_fheap;
    op_data.attr            = attr;
    op_data.corder_bt2_addr = ainfo->corder_bt2_addr;

    
    if (H5B2_modify(bt2_name, &udata, false, H5A__dense_write_bt2_cb, &op_data) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to modify record in v2 B-tree");

done:
    
    if (shared_fheap && H5HF_close(shared_fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5A__dense_copy_fh_cb(const void *obj, size_t obj_len, void *_udata)
{
    H5A_fh_ud_cp_t *udata     = (H5A_fh_ud_cp_t *)_udata; 
    herr_t          ret_value = SUCCEED;                  

    FUNC_ENTER_PACKAGE

    
    
    if (NULL == (udata->attr = (H5A_t *)H5O_msg_decode(udata->f, NULL, H5O_ATTR_ID, obj_len,
                                                       (const unsigned char *)obj)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTDECODE, FAIL, "can't decode attribute");

    
    udata->attr->shared->crt_idx = udata->record->corder;

    
    if (udata->record->flags & H5O_MSG_FLAG_SHARED)
        H5SM_reconstitute(&(udata->attr->sh_loc), udata->f, H5O_ATTR_ID, udata->record->id);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5A__dense_rename(H5F_t *f, const H5O_ainfo_t *ainfo, const char *old_name, const char *new_name)
{
    H5A_bt2_ud_common_t udata;               
    H5HF_t             *fheap        = NULL; 
    H5HF_t             *shared_fheap = NULL; 
    H5B2_t             *bt2_name     = NULL; 
    H5B2_t             *bt2_corder   = NULL; 
    H5A_t              *attr_copy    = NULL; 
    htri_t              attr_sharable;       
    htri_t              shared_mesg;         
    bool                attr_exists;         
    H5O_ainfo_t         tainfo    = *ainfo;  
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);
    assert(old_name);
    assert(new_name);

    
    if ((attr_sharable = H5SM_type_shared(f, H5O_ATTR_ID)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't determine if attributes are shared");

    
    if (attr_sharable) {
        haddr_t shared_fheap_addr; 

        
        if (H5SM_get_fheap_addr(f, H5O_ATTR_ID, &shared_fheap_addr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get shared message heap address");

        
        if (H5_addr_defined(shared_fheap_addr)) {
            
            if (NULL == (shared_fheap = H5HF_open(f, shared_fheap_addr)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");
        } 
    }     

    
    if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    if (NULL == (bt2_name = H5B2_open(f, ainfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    udata.f             = f;
    udata.fheap         = fheap;
    udata.shared_fheap  = shared_fheap;
    udata.name          = old_name;
    udata.name_hash     = H5_checksum_lookup3(old_name, strlen(old_name), 0);
    udata.flags         = 0;
    udata.corder        = 0;
    udata.found_op      = H5A__dense_fnd_cb; 
    udata.found_op_data = &attr_copy;

    
    attr_exists = false;
    if (H5B2_find(bt2_name, &udata, &attr_exists, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, FAIL, "can't search for attribute in name index");
    if (attr_exists == false)
        HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, FAIL, "can't locate attribute in name index");
    assert(attr_copy);

    
    if ((shared_mesg = H5O_msg_is_shared(H5O_ATTR_ID, attr_copy)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "error determining if message is shared");
    else if (shared_mesg > 0) {
        
        
        attr_copy->sh_loc.type = H5O_SHARE_TYPE_UNSHARED;
    } 

    
    H5MM_xfree(attr_copy->shared->name);
    attr_copy->shared->name = H5MM_xstrdup(new_name);

    
    if (H5A__set_version(f, attr_copy) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTSET, FAIL, "unable to update attribute version");

    
    if (ainfo->index_corder) {
        bool corder_attr_exists; 

        
        assert(H5_addr_defined(ainfo->corder_bt2_addr));
        if (NULL == (bt2_corder = H5B2_open(f, ainfo->corder_bt2_addr, NULL)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for creation index");

        
        udata.corder = attr_copy->shared->crt_idx;

        corder_attr_exists = false;
        if (H5B2_find(bt2_corder, &udata, &corder_attr_exists, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, FAIL, "can't search for attribute in name index");

        if (corder_attr_exists) {
            H5A_bt2_ud_rm_t rm_udata;

            
            rm_udata.common.corder = attr_copy->shared->crt_idx;

            
            if (H5B2_remove(bt2_corder, &rm_udata, NULL, NULL) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTREMOVE, FAIL,
                            "unable to remove attribute from creation order index v2 B-tree");
        }
    }

    
    
    if (H5A__dense_insert(f, ainfo, attr_copy) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTINSERT, FAIL, "unable to add to dense storage");

    
    if ((shared_mesg = H5O_msg_is_shared(H5O_ATTR_ID, attr_copy)) > 0) {
        hsize_t attr_rc; 

        
        if (H5SM_get_refcount(f, H5O_ATTR_ID, &attr_copy->sh_loc, &attr_rc) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't retrieve shared message ref count");

        
        if (attr_rc == 1) {
            
            if (H5O__attr_link(f, NULL, attr_copy) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_LINKCOUNT, FAIL, "unable to adjust attribute link count");
        } 
    }     
    else if (shared_mesg == 0) {
        
        
        if (H5O__attr_link(f, NULL, attr_copy) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_LINKCOUNT, FAIL, "unable to adjust attribute link count");
    } 
    else if (shared_mesg < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_WRITEERROR, FAIL, "error determining if message should be shared");

    
    tainfo.corder_bt2_addr = HADDR_UNDEF;

    
    if (H5A__dense_remove(f, (const H5O_ainfo_t *)&tainfo, old_name) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute in dense storage");

done:
    
    if (shared_fheap && H5HF_close(shared_fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for creation order index");
    if (attr_copy)
        H5O_msg_free(H5O_ATTR_ID, attr_copy);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5A__dense_iterate_bt2_cb(const void *_record, void *_bt2_udata)
{
    const H5A_dense_bt2_name_rec_t *record =
        (const H5A_dense_bt2_name_rec_t *)_record;              
    H5A_bt2_ud_it_t *bt2_udata = (H5A_bt2_ud_it_t *)_bt2_udata; 
    herr_t           ret_value = H5_ITER_CONT;                  

    FUNC_ENTER_PACKAGE

    
    if (bt2_udata->skip > 0)
        --bt2_udata->skip;
    else {
        H5A_fh_ud_cp_t fh_udata; 
        H5HF_t        *fheap;    

        
        if (record->flags & H5O_MSG_FLAG_SHARED)
            fheap = bt2_udata->shared_fheap;
        else
            fheap = bt2_udata->fheap;

        
        
        fh_udata.f      = bt2_udata->f;
        fh_udata.record = record;
        fh_udata.attr   = NULL;

        
        if (H5HF_op(fheap, &record->id, H5A__dense_copy_fh_cb, &fh_udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPERATE, H5_ITER_ERROR, "heap op callback failed");

        
        switch (bt2_udata->attr_op->op_type) {
            case H5A_ATTR_OP_APP2: {
                H5A_info_t ainfo; 

                
                if (H5A__get_info(fh_udata.attr, &ainfo) < 0)
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, H5_ITER_ERROR, "unable to get attribute info");

                
                H5_BEFORE_USER_CB(H5_ITER_ERROR)
                    {
                        
                        ret_value = (bt2_udata->attr_op->u.app_op2)(
                            bt2_udata->loc_id, fh_udata.attr->shared->name, &ainfo, bt2_udata->op_data);
                    }
                H5_AFTER_USER_CB(H5_ITER_ERROR)
                break;
            }

#ifndef H5_NO_DEPRECATED_SYMBOLS
            case H5A_ATTR_OP_APP:
                
                H5_BEFORE_USER_CB(H5_ITER_ERROR)
                    {
                        
                        ret_value = (bt2_udata->attr_op->u.app_op)(
                            bt2_udata->loc_id, fh_udata.attr->shared->name, bt2_udata->op_data);
                    }
                H5_AFTER_USER_CB(H5_ITER_ERROR)
                break;
#endif 

            case H5A_ATTR_OP_LIB:
                
                ret_value = (bt2_udata->attr_op->u.lib_op)(fh_udata.attr, bt2_udata->op_data);
                break;

            default:
                assert("unknown attribute op type" && 0);
#ifdef NDEBUG
                HGOTO_ERROR(H5E_ATTR, H5E_UNSUPPORTED, FAIL, "unsupported attribute op type");
#endif    
        } 

        
        H5O_msg_free(H5O_ATTR_ID, fh_udata.attr);
    } 

    
    
    bt2_udata->count++;

    
    if (ret_value < 0)
        HERROR(H5E_ATTR, H5E_CANTNEXT, "iteration operator failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5A__dense_iterate(H5F_t *f, hid_t loc_id, const H5O_ainfo_t *ainfo, H5_index_t idx_type,
                   H5_iter_order_t order, hsize_t skip, hsize_t *last_attr, const H5A_attr_iter_op_t *attr_op,
                   void *op_data)
{
    H5HF_t          *fheap        = NULL;         
    H5HF_t          *shared_fheap = NULL;         
    H5A_attr_table_t atable       = {0, 0, NULL}; 
    H5B2_t          *bt2          = NULL;         
    haddr_t          bt2_addr;                    
    herr_t           ret_value = FAIL;            

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);
    assert(H5_addr_defined(ainfo->fheap_addr));
    assert(H5_addr_defined(ainfo->name_bt2_addr));
    assert(attr_op);

    
    if (idx_type == H5_INDEX_NAME) {
        
        if (order == H5_ITER_NATIVE) {
            assert(H5_addr_defined(ainfo->name_bt2_addr));
            bt2_addr = ainfo->name_bt2_addr;
        } 
        else
            bt2_addr = HADDR_UNDEF;
    } 
    else {
        assert(idx_type == H5_INDEX_CRT_ORDER);

        
        bt2_addr = ainfo->corder_bt2_addr;
    } 

    
    if (order == H5_ITER_NATIVE && H5_addr_defined(bt2_addr)) {
        H5A_bt2_ud_it_t udata;         
        htri_t          attr_sharable; 

        
        if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

        
        if ((attr_sharable = H5SM_type_shared(f, H5O_ATTR_ID)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't determine if attributes are shared");

        
        if (attr_sharable) {
            haddr_t shared_fheap_addr; 

            
            if (H5SM_get_fheap_addr(f, H5O_ATTR_ID, &shared_fheap_addr) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get shared message heap address");

            
            if (H5_addr_defined(shared_fheap_addr)) {
                
                if (NULL == (shared_fheap = H5HF_open(f, shared_fheap_addr)))
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");
            } 
        }     

        
        if (NULL == (bt2 = H5B2_open(f, bt2_addr, NULL)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for index");

        
        udata.f            = f;
        udata.fheap        = fheap;
        udata.shared_fheap = shared_fheap;
        udata.loc_id       = loc_id;
        udata.skip         = skip;
        udata.count        = 0;
        udata.attr_op      = attr_op;
        udata.op_data      = op_data;

        
        
        if ((ret_value = H5B2_iterate(bt2, H5A__dense_iterate_bt2_cb, &udata)) < 0)
            HERROR(H5E_ATTR, H5E_BADITER, "attribute iteration failed");

        
        if (last_attr)
            *last_attr = udata.count;
    } 
    else {
        
        
        if (H5A__dense_build_table(f, ainfo, idx_type, order, &atable) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "error building table of attributes");

        
        if ((ret_value = H5A__attr_iterate_table(&atable, skip, last_attr, loc_id, attr_op, op_data)) < 0)
            HERROR(H5E_ATTR, H5E_CANTNEXT, "iteration operator failed");
    } 

done:
    
    if (shared_fheap && H5HF_close(shared_fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for index");
    if (atable.attrs && H5A__attr_release_table(&atable) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to release attribute table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5A__dense_remove_bt2_cb(const void *_record, void *_udata)
{
    const H5A_dense_bt2_name_rec_t *record = (const H5A_dense_bt2_name_rec_t *)_record;
    H5A_bt2_ud_rm_t                *udata  = (H5A_bt2_ud_rm_t *)_udata; 
    H5A_t  *attr       = *(H5A_t **)udata->common.found_op_data;        
    H5B2_t *bt2_corder = NULL;    
    herr_t  ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5_addr_defined(udata->corder_bt2_addr)) {
        
        if (NULL == (bt2_corder = H5B2_open(udata->common.f, udata->corder_bt2_addr, NULL)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for creation order index");

        
        udata->common.corder = attr->shared->crt_idx;

        
        if (H5B2_remove(bt2_corder, udata, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTREMOVE, FAIL,
                        "unable to remove attribute from creation order index v2 B-tree");
    } 

    
    if (record->flags & H5O_MSG_FLAG_SHARED) {
        
        if (H5SM_delete(udata->common.f, NULL, &(attr->sh_loc)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to delete shared attribute");
    } 
    else {
        
        
        if (H5O__attr_delete(udata->common.f, NULL, attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute");

        
        if (H5HF_remove(udata->common.fheap, &record->id) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTREMOVE, FAIL, "unable to remove attribute from fractal heap");
    } 

done:
    
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for creation order index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5A__dense_remove(H5F_t *f, const H5O_ainfo_t *ainfo, const char *name)
{
    H5A_bt2_ud_rm_t udata;               
    H5HF_t         *fheap        = NULL; 
    H5HF_t         *shared_fheap = NULL; 
    H5B2_t         *bt2_name     = NULL; 
    H5A_t          *attr_copy    = NULL; 
    htri_t          attr_sharable;       
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);
    assert(name && *name);

    
    if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    if ((attr_sharable = H5SM_type_shared(f, H5O_ATTR_ID)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't determine if attributes are shared");

    
    if (attr_sharable) {
        haddr_t shared_fheap_addr; 

        
        if (H5SM_get_fheap_addr(f, H5O_ATTR_ID, &shared_fheap_addr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get shared message heap address");

        
        if (H5_addr_defined(shared_fheap_addr)) {
            
            if (NULL == (shared_fheap = H5HF_open(f, shared_fheap_addr)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");
        } 
    }     

    
    if (NULL == (bt2_name = H5B2_open(f, ainfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    udata.common.f             = f;
    udata.common.fheap         = fheap;
    udata.common.shared_fheap  = shared_fheap;
    udata.common.name          = name;
    udata.common.name_hash     = H5_checksum_lookup3(name, strlen(name), 0);
    udata.common.found_op      = H5A__dense_fnd_cb; 
    udata.common.found_op_data = &attr_copy;
    udata.corder_bt2_addr      = ainfo->corder_bt2_addr;

    
    if (H5B2_remove(bt2_name, &udata, H5A__dense_remove_bt2_cb, &udata) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTREMOVE, FAIL, "unable to remove attribute from name index v2 B-tree");

done:
    
    if (shared_fheap && H5HF_close(shared_fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");
    if (attr_copy)
        H5O_msg_free_real(H5O_MSG_ATTR, attr_copy);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5A__dense_remove_by_idx_bt2_cb(const void *_record, void *_bt2_udata)
{
    H5HF_t                         *fheap;         
    H5B2_t                         *bt2    = NULL; 
    const H5A_dense_bt2_name_rec_t *record = (const H5A_dense_bt2_name_rec_t *)_record; 
    H5A_bt2_ud_rmbi_t              *bt2_udata = (H5A_bt2_ud_rmbi_t *)_bt2_udata; 
    H5A_fh_ud_cp_t                  fh_udata; 
    H5O_shared_t                    sh_loc;   
    bool   use_sh_loc;          
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    fh_udata.f      = bt2_udata->f;
    fh_udata.record = record;
    fh_udata.attr   = NULL;

    
    if (record->flags & H5O_MSG_FLAG_SHARED)
        fheap = bt2_udata->shared_fheap;
    else
        fheap = bt2_udata->fheap;

    
    if (H5_addr_defined(bt2_udata->other_bt2_addr) || !(record->flags & H5O_MSG_FLAG_SHARED)) {
        
        if (H5HF_op(fheap, &record->id, H5A__dense_copy_fh_cb, &fh_udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPERATE, FAIL, "attribute removal callback failed");
        assert(fh_udata.attr);

        
        use_sh_loc = false;
    } 
    else {
        
        H5SM_reconstitute(&sh_loc, bt2_udata->f, H5O_ATTR_ID, record->id);

        
        use_sh_loc = true;
    } 

    
    if (H5_addr_defined(bt2_udata->other_bt2_addr)) {
        H5A_bt2_ud_common_t other_bt2_udata; 

        
        if (bt2_udata->idx_type == H5_INDEX_NAME) {
            
            other_bt2_udata.corder = fh_udata.attr->shared->crt_idx;
        } 
        else {
            assert(bt2_udata->idx_type == H5_INDEX_CRT_ORDER);

            
            other_bt2_udata.f            = bt2_udata->f;
            other_bt2_udata.fheap        = bt2_udata->fheap;
            other_bt2_udata.shared_fheap = bt2_udata->shared_fheap;
            other_bt2_udata.name         = fh_udata.attr->shared->name;
            other_bt2_udata.name_hash =
                H5_checksum_lookup3(fh_udata.attr->shared->name, strlen(fh_udata.attr->shared->name), 0);
            other_bt2_udata.found_op      = NULL;
            other_bt2_udata.found_op_data = NULL;
        } 

        
        if (NULL == (bt2 = H5B2_open(bt2_udata->f, bt2_udata->other_bt2_addr, NULL)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for index");

        

        
        if (H5B2_remove(bt2, &other_bt2_udata, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTREMOVE, FAIL,
                        "unable to remove record from 'other' index v2 B-tree");
    } 

    
    if (record->flags & H5O_MSG_FLAG_SHARED) {
        H5O_shared_t *sh_loc_ptr; 

        
        if (use_sh_loc)
            sh_loc_ptr = &sh_loc;
        else
            sh_loc_ptr = &(fh_udata.attr->sh_loc);

        
        if (H5SM_delete(bt2_udata->f, NULL, sh_loc_ptr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to delete shared attribute");
    } 
    else {
        
        
        if (H5O__attr_delete(bt2_udata->f, NULL, fh_udata.attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute");

        
        if (H5HF_remove(fheap, &record->id) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTREMOVE, FAIL, "unable to remove attribute from fractal heap");
    } 

done:
    
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for index");
    if (fh_udata.attr)
        H5O_msg_free(H5O_ATTR_ID, fh_udata.attr);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5A__dense_remove_by_idx(H5F_t *f, const H5O_ainfo_t *ainfo, H5_index_t idx_type, H5_iter_order_t order,
                         hsize_t n)
{
    H5HF_t          *fheap        = NULL;         
    H5HF_t          *shared_fheap = NULL;         
    H5A_attr_table_t atable       = {0, 0, NULL}; 
    H5B2_t          *bt2          = NULL;         
    haddr_t          bt2_addr;                    
    herr_t           ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);

    
    if (idx_type == H5_INDEX_NAME) {
        
        if (order == H5_ITER_NATIVE) {
            bt2_addr = ainfo->name_bt2_addr;
            assert(H5_addr_defined(bt2_addr));
        } 
        else
            bt2_addr = HADDR_UNDEF;
    } 
    else {
        assert(idx_type == H5_INDEX_CRT_ORDER);

        
        bt2_addr = ainfo->corder_bt2_addr;
    } 

    
    if (H5_addr_defined(bt2_addr)) {
        H5A_bt2_ud_rmbi_t udata;         
        htri_t            attr_sharable; 

        
        if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

        
        if ((attr_sharable = H5SM_type_shared(f, H5O_ATTR_ID)) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't determine if attributes are shared");

        
        if (attr_sharable) {
            haddr_t shared_fheap_addr; 

            
            if (H5SM_get_fheap_addr(f, H5O_ATTR_ID, &shared_fheap_addr) < 0)
                HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get shared message heap address");

            
            if (H5_addr_defined(shared_fheap_addr)) {
                
                if (NULL == (shared_fheap = H5HF_open(f, shared_fheap_addr)))
                    HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");
            } 
        }     

        
        if (NULL == (bt2 = H5B2_open(f, bt2_addr, NULL)))
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for index");

        
        udata.f              = f;
        udata.fheap          = fheap;
        udata.shared_fheap   = shared_fheap;
        udata.idx_type       = idx_type;
        udata.other_bt2_addr = idx_type == H5_INDEX_NAME ? ainfo->corder_bt2_addr : ainfo->name_bt2_addr;

        
        if (H5B2_remove_by_idx(bt2, order, n, H5A__dense_remove_by_idx_bt2_cb, &udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTREMOVE, FAIL, "unable to remove attribute from v2 B-tree index");
    } 
    else {
        
        
        if (H5A__dense_build_table(f, ainfo, idx_type, order, &atable) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "error building table of attributes");

        
        if (n >= atable.num_attrs)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index specified");

        
        if (H5A__dense_remove(f, ainfo, ((atable.attrs[n])->shared)->name) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute in dense storage");
    } 

done:
    
    if (shared_fheap && H5HF_close(shared_fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2 && H5B2_close(bt2) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for index");
    if (atable.attrs && H5A__attr_release_table(&atable) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to release attribute table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5A__dense_exists(H5F_t *f, const H5O_ainfo_t *ainfo, const char *name, bool *attr_exists)
{
    H5A_bt2_ud_common_t udata;               
    H5HF_t             *fheap        = NULL; 
    H5HF_t             *shared_fheap = NULL; 
    H5B2_t             *bt2_name     = NULL; 
    htri_t              attr_sharable;       
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);
    assert(name);
    assert(attr_exists);

    
    if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    if ((attr_sharable = H5SM_type_shared(f, H5O_ATTR_ID)) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't determine if attributes are shared");

    
    if (attr_sharable) {
        haddr_t shared_fheap_addr; 

        
        if (H5SM_get_fheap_addr(f, H5O_ATTR_ID, &shared_fheap_addr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTGET, FAIL, "can't get shared message heap address");

        
        if (H5_addr_defined(shared_fheap_addr)) {
            
            if (NULL == (shared_fheap = H5HF_open(f, shared_fheap_addr)))
                HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");
        } 
    }     

    
    if (NULL == (bt2_name = H5B2_open(f, ainfo->name_bt2_addr, NULL)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    udata.f             = f;
    udata.fheap         = fheap;
    udata.shared_fheap  = shared_fheap;
    udata.name          = name;
    udata.name_hash     = H5_checksum_lookup3(name, strlen(name), 0);
    udata.flags         = 0;
    udata.corder        = 0;
    udata.found_op      = NULL; 
    udata.found_op_data = NULL;

    
    if (H5B2_find(bt2_name, &udata, attr_exists, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_NOTFOUND, FAIL, "can't search for attribute in name index");

done:
    
    if (shared_fheap && H5HF_close(shared_fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5A__dense_delete_bt2_cb(const void *_record, void *_bt2_udata)
{
    const H5A_dense_bt2_name_rec_t *record =
        (const H5A_dense_bt2_name_rec_t *)_record;                      
    H5A_bt2_ud_common_t *bt2_udata = (H5A_bt2_ud_common_t *)_bt2_udata; 
    H5A_t               *attr      = NULL;                              
    herr_t               ret_value = SUCCEED;                           

    FUNC_ENTER_PACKAGE

    
    if (record->flags & H5O_MSG_FLAG_SHARED) {
        H5O_shared_t sh_mesg; 

        
        H5SM_reconstitute(&sh_mesg, bt2_udata->f, H5O_ATTR_ID, record->id);

        
        if (H5SM_delete(bt2_udata->f, NULL, &sh_mesg) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTFREE, FAIL, "unable to delete shared attribute");
    } 
    else {
        H5A_fh_ud_cp_t fh_udata; 

        
        
        fh_udata.f      = bt2_udata->f;
        fh_udata.record = record;
        
        fh_udata.attr = NULL;

        
        if (H5HF_op(bt2_udata->fheap, &record->id, H5A__dense_copy_fh_cb, &fh_udata) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTOPERATE, FAIL, "heap op callback failed");
        attr = fh_udata.attr;

        
        
        if (H5O__attr_delete(bt2_udata->f, NULL, fh_udata.attr) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete attribute");
    } 

done:
    
    if (attr)
        H5O_msg_free_real(H5O_MSG_ATTR, attr);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5A__dense_delete(H5F_t *f, H5O_ainfo_t *ainfo)
{
    H5A_bt2_ud_common_t udata;               
    H5HF_t             *fheap     = NULL;    
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);

    
    if (NULL == (fheap = H5HF_open(f, ainfo->fheap_addr)))
        HGOTO_ERROR(H5E_ATTR, H5E_CANTOPENOBJ, FAIL, "unable to open fractal heap");

    
    udata.f             = f;
    udata.fheap         = fheap;
    udata.shared_fheap  = NULL;
    udata.name          = NULL;
    udata.name_hash     = 0;
    udata.flags         = 0;
    udata.found_op      = NULL; 
    udata.found_op_data = NULL;

    
    if (H5B2_delete(f, ainfo->name_bt2_addr, NULL, H5A__dense_delete_bt2_cb, &udata) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete v2 B-tree for name index");
    ainfo->name_bt2_addr = HADDR_UNDEF;

    
    if (H5HF_close(fheap) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");
    fheap = NULL;

    
    if (H5_addr_defined(ainfo->corder_bt2_addr)) {
        
        if (H5B2_delete(f, ainfo->corder_bt2_addr, NULL, NULL, NULL) < 0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL,
                        "unable to delete v2 B-tree for creation order index");
        ainfo->corder_bt2_addr = HADDR_UNDEF;
    } 

    
    if (H5HF_delete(f, ainfo->fheap_addr) < 0)
        HGOTO_ERROR(H5E_ATTR, H5E_CANTDELETE, FAIL, "unable to delete fractal heap");
    ainfo->fheap_addr = HADDR_UNDEF;

done:
    
    if (fheap && H5HF_close(fheap) < 0)
        HDONE_ERROR(H5E_ATTR, H5E_CLOSEERROR, FAIL, "can't close fractal heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 
