/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FApkg.h"     
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5MMprivate.h" 

H5FL_DEFINE_STATIC(H5FA_hdr_t);

H5FA_hdr_t *
H5FA__hdr_alloc(H5F_t *f)
{
    H5FA_hdr_t *hdr       = NULL; 
    H5FA_hdr_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    if (NULL == (hdr = H5FL_CALLOC(H5FA_hdr_t)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for Fixed Array shared header");

    
    hdr->addr = HADDR_UNDEF;

    
    hdr->f           = f;
    hdr->swmr_write  = (H5F_INTENT(f) & H5F_ACC_SWMR_WRITE) > 0;
    hdr->sizeof_addr = H5F_SIZEOF_ADDR(f);
    hdr->sizeof_size = H5F_SIZEOF_SIZE(f);

    
    ret_value = hdr;

done:
    if (!ret_value)
        if (hdr && H5FA__hdr_dest(hdr) < 0)
            HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, NULL, "unable to destroy fixed array header");
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__hdr_init(H5FA_hdr_t *hdr, void *ctx_udata)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    hdr->stats.hdr_size = hdr->size = H5FA_HEADER_SIZE_HDR(hdr);

    
    hdr->stats.nelmts = hdr->cparam.nelmts;

    
    if (hdr->cparam.cls->crt_context)
        if (NULL == (hdr->cb_ctx = (*hdr->cparam.cls->crt_context)(ctx_udata)))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTCREATE, FAIL,
                        "unable to create fixed array client callback context");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5FA__hdr_create(H5F_t *f, const H5FA_create_t *cparam, void *ctx_udata)
{
    H5FA_hdr_t *hdr       = NULL;  
    bool        inserted  = false; 
    haddr_t     ret_value = HADDR_UNDEF;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(cparam);

#ifndef NDEBUG
    {
        
        if (cparam->raw_elmt_size == 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, HADDR_UNDEF, "element size must be greater than zero");
        if (cparam->max_dblk_page_nelmts_bits == 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, HADDR_UNDEF,
                        "max. # of elements bits must be greater than zero");
        if (cparam->nelmts == 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_BADVALUE, HADDR_UNDEF, "# of elements must be greater than zero");
    }
#endif 

    
    if (NULL == (hdr = H5FA__hdr_alloc(f)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "memory allocation failed for Fixed Array shared header");

    hdr->dblk_addr = HADDR_UNDEF;

    
    H5MM_memcpy(&hdr->cparam, cparam, sizeof(hdr->cparam));

    
    if (H5FA__hdr_init(hdr, ctx_udata) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINIT, HADDR_UNDEF, "initialization failed for fixed array header");

    
    if (HADDR_UNDEF == (hdr->addr = H5MF_alloc(f, H5FD_MEM_FARRAY_HDR, (hsize_t)hdr->size)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTALLOC, HADDR_UNDEF, "file allocation failed for Fixed Array header");

    
    if (hdr->swmr_write)
        if (NULL == (hdr->top_proxy = H5AC_proxy_entry_create()))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTCREATE, HADDR_UNDEF, "can't create fixed array entry proxy");

    
    if (H5AC_insert_entry(f, H5AC_FARRAY_HDR, hdr->addr, hdr, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTINSERT, HADDR_UNDEF, "can't add fixed array header to cache");
    inserted = true;

    
    if (hdr->top_proxy)
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, f, hdr) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "unable to add fixed array entry as child of array proxy");

    
    ret_value = hdr->addr;

done:
    if (!H5_addr_defined(ret_value))
        if (hdr) {
            
            if (inserted)
                if (H5AC_remove_entry(hdr) < 0)
                    HDONE_ERROR(H5E_FARRAY, H5E_CANTREMOVE, HADDR_UNDEF,
                                "unable to remove fixed array header from cache");

            
            if (H5_addr_defined(hdr->addr) &&
                H5MF_xfree(f, H5FD_MEM_FARRAY_HDR, hdr->addr, (hsize_t)hdr->size) < 0)
                HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, HADDR_UNDEF, "unable to free Fixed Array header");

            
            if (H5FA__hdr_dest(hdr) < 0)
                HDONE_ERROR(H5E_FARRAY, H5E_CANTFREE, HADDR_UNDEF, "unable to destroy Fixed Array header");
        }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__hdr_incr(H5FA_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (hdr->rc == 0)
        if (H5AC_pin_protected_entry(hdr) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTPIN, FAIL, "unable to pin fixed array header");

    
    hdr->rc++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__hdr_decr(H5FA_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->rc);

    
    hdr->rc--;

    
    if (hdr->rc == 0) {
        assert(hdr->file_rc == 0);
        if (H5AC_unpin_entry(hdr) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNPIN, FAIL, "unable to unpin fixed array header");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__hdr_fuse_incr(H5FA_hdr_t *hdr)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);

    
    hdr->file_rc++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

size_t
H5FA__hdr_fuse_decr(H5FA_hdr_t *hdr)
{
    size_t ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(hdr);
    assert(hdr->file_rc);

    
    hdr->file_rc--;

    
    ret_value = hdr->file_rc;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__hdr_modified(H5FA_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5AC_mark_entry_dirty(hdr) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTMARKDIRTY, FAIL, "unable to mark fixed array header as dirty");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5FA_hdr_t *
H5FA__hdr_protect(H5F_t *f, haddr_t fa_addr, void *ctx_udata, unsigned flags)
{
    H5FA_hdr_t         *hdr;   
    H5FA_hdr_cache_ud_t udata; 
    H5FA_hdr_t         *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(fa_addr));

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.f         = f;
    udata.addr      = fa_addr;
    udata.ctx_udata = ctx_udata;

    
    if (NULL == (hdr = (H5FA_hdr_t *)H5AC_protect(f, H5AC_FARRAY_HDR, fa_addr, &udata, flags)))
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTPROTECT, NULL, "unable to protect fixed array header, address = %llu",
                    (unsigned long long)fa_addr);
    hdr->f = f; 

    
    if (hdr->swmr_write && NULL == hdr->top_proxy) {
        
        if (NULL == (hdr->top_proxy = H5AC_proxy_entry_create()))
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTCREATE, NULL, "can't create fixed array entry proxy");

        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, f, hdr) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTSET, NULL,
                        "unable to add fixed array entry as child of array proxy");
    }

    
    ret_value = hdr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__hdr_unprotect(H5FA_hdr_t *hdr, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (H5AC_unprotect(hdr->f, H5AC_FARRAY_HDR, hdr->addr, hdr, cache_flags) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect fixed array hdr, address = %llu", (unsigned long long)hdr->addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__hdr_delete(H5FA_hdr_t *hdr)
{
    unsigned cache_flags = H5AC__NO_FLAGS_SET; 
    herr_t   ret_value   = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(!hdr->file_rc);

#ifndef NDEBUG

    unsigned hdr_status = 0; 

    
    if (H5AC_get_entry_status(hdr->f, hdr->addr, &hdr_status) < 0)
        HGOTO_ERROR(H5E_FARRAY, H5E_CANTGET, FAIL, "unable to check metadata cache status for array header");

    
    assert(hdr_status & H5AC_ES__IN_CACHE);
    assert(hdr_status & H5AC_ES__IS_PROTECTED);

#endif 

    
    if (H5_addr_defined(hdr->dblk_addr)) {
        
        if (H5FA__dblock_delete(hdr, hdr->dblk_addr) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTDELETE, FAIL, "unable to delete fixed array data block");
    }

    
    cache_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG | H5AC__FREE_FILE_SPACE_FLAG;

done:
    
    if (H5AC_unprotect(hdr->f, H5AC_FARRAY_HDR, hdr->addr, hdr, cache_flags) < 0)
        HDONE_ERROR(H5E_FARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release fixed array header");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FA__hdr_dest(H5FA_hdr_t *hdr)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->rc == 0);

    
    if (hdr->cb_ctx) {
        if ((*hdr->cparam.cls->dst_context)(hdr->cb_ctx) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTRELEASE, FAIL,
                        "unable to destroy fixed array client callback context");
    }
    hdr->cb_ctx = NULL;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_dest(hdr->top_proxy) < 0)
            HGOTO_ERROR(H5E_FARRAY, H5E_CANTRELEASE, FAIL, "unable to destroy fixed array 'top' proxy");
        hdr->top_proxy = NULL;
    }

    
    hdr = H5FL_FREE(H5FA_hdr_t, hdr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
