/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5FDmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FDpkg.h"     
#include "H5FDsec2.h"    
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5Pprivate.h"  

hid_t H5FD_SEC2_id_g = H5I_INVALID_HID;

typedef struct H5FD_sec2_t {
    H5FD_t  pub; 
    int     fd;  
    haddr_t eoa; 
    haddr_t eof; 
#ifndef H5_HAVE_PREADWRITE
    haddr_t        pos; 
    H5FD_file_op_t op;  
#endif                  
    bool ignore_disabled_file_locks;
    char filename[H5FD_MAX_FILENAME_LEN]; 
#ifndef H5_HAVE_WIN32_API
    
    dev_t device; 
    ino_t inode;  
#else
    
    DWORD nFileIndexLow;
    DWORD nFileIndexHigh;
    DWORD dwVolumeSerialNumber;

    HANDLE hFile; 
#endif 

    
    bool fam_to_single;
} H5FD_sec2_t;

static H5FD_t *H5FD__sec2_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr);
static herr_t  H5FD__sec2_close(H5FD_t *_file);
static int     H5FD__sec2_cmp(const H5FD_t *_f1, const H5FD_t *_f2);
static herr_t  H5FD__sec2_query(const H5FD_t *_f1, unsigned long *flags);
static haddr_t H5FD__sec2_get_eoa(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD__sec2_set_eoa(H5FD_t *_file, H5FD_mem_t type, haddr_t addr);
static haddr_t H5FD__sec2_get_eof(const H5FD_t *_file, H5FD_mem_t type);
static herr_t  H5FD__sec2_get_handle(H5FD_t *_file, hid_t fapl, void **file_handle);
static herr_t  H5FD__sec2_read(H5FD_t *_file, H5FD_mem_t type, hid_t fapl_id, haddr_t addr, size_t size,
                               void *buf);
static herr_t  H5FD__sec2_write(H5FD_t *_file, H5FD_mem_t type, hid_t fapl_id, haddr_t addr, size_t size,
                                const void *buf);
static herr_t  H5FD__sec2_truncate(H5FD_t *_file, hid_t dxpl_id, bool closing);
static herr_t  H5FD__sec2_lock(H5FD_t *_file, bool rw);
static herr_t  H5FD__sec2_unlock(H5FD_t *_file);
static herr_t  H5FD__sec2_delete(const char *filename, hid_t fapl_id);
static herr_t  H5FD__sec2_ctl(H5FD_t *_file, uint64_t op_code, uint64_t flags, const void *input,
                              void **output);

static const H5FD_class_t H5FD_sec2_g = {
    H5FD_CLASS_VERSION,    
    H5FD_SEC2_VALUE,       
    "sec2",                
    H5FD_MAXADDR,          
    H5F_CLOSE_WEAK,        
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    0,                     
    NULL,                  
    NULL,                  
    NULL,                  
    0,                     
    NULL,                  
    NULL,                  
    H5FD__sec2_open,       
    H5FD__sec2_close,      
    H5FD__sec2_cmp,        
    H5FD__sec2_query,      
    NULL,                  
    NULL,                  
    NULL,                  
    H5FD__sec2_get_eoa,    
    H5FD__sec2_set_eoa,    
    H5FD__sec2_get_eof,    
    H5FD__sec2_get_handle, 
    H5FD__sec2_read,       
    H5FD__sec2_write,      
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    NULL,                  
    H5FD__sec2_truncate,   
    H5FD__sec2_lock,       
    H5FD__sec2_unlock,     
    H5FD__sec2_delete,     
    H5FD__sec2_ctl,        
    H5FD_FLMAP_DICHOTOMY   
};

H5FL_DEFINE_STATIC(H5FD_sec2_t);

herr_t
H5FD__sec2_register(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5I_VFL != H5I_get_type(H5FD_SEC2_id_g))
        if ((H5FD_SEC2_id_g = H5FD_register(&H5FD_sec2_g, sizeof(H5FD_class_t), false)) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTREGISTER, FAIL, "unable to register sec2 driver");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5FD__sec2_unregister(void)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    H5FD_SEC2_id_g = H5I_INVALID_HID;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_fapl_sec2(hid_t fapl_id)
{
    H5P_genplist_t *plist; 
    herr_t          ret_value;

    FUNC_ENTER_API(FAIL)

    if (NULL == (plist = H5P_object_verify(fapl_id, H5P_FILE_ACCESS, false)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file access property list");

    ret_value = H5P_set_driver(plist, H5FD_SEC2, NULL, NULL);

done:
    FUNC_LEAVE_API(ret_value)
} 

static H5FD_t *
H5FD__sec2_open(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr)
{
    H5FD_sec2_t *file = NULL; 
    int          fd   = -1;   
    int          o_flags;     
#ifdef H5_HAVE_WIN32_API
    struct _BY_HANDLE_FILE_INFORMATION fileinfo;
#endif
    h5_stat_t       sb;
    H5P_genplist_t *plist;            
    H5FD_t         *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    HDcompile_assert(sizeof(HDoff_t) >= sizeof(size_t));

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "invalid file name");
    if (0 == maxaddr || HADDR_UNDEF == maxaddr)
        HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, NULL, "bogus maxaddr");
    if (H5FD_ADDR_OVERFLOW(maxaddr))
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, NULL, "bogus maxaddr");

    
    o_flags = (H5F_ACC_RDWR & flags) ? O_RDWR : O_RDONLY;
    if (H5F_ACC_TRUNC & flags)
        o_flags |= O_TRUNC;
    if (H5F_ACC_CREAT & flags)
        o_flags |= O_CREAT;
    if (H5F_ACC_EXCL & flags)
        o_flags |= O_EXCL;

    
    if ((fd = HDopen(name, o_flags, H5_POSIX_CREATE_MODE_RW)) < 0) {
        int myerrno = errno;
        HGOTO_ERROR(
            H5E_FILE, H5E_CANTOPENFILE, NULL,
            "unable to open file: name = '%s', errno = %d, error message = '%s', flags = %x, o_flags = %x",
            name, myerrno, strerror(myerrno), flags, (unsigned)o_flags);
    } 

    memset(&sb, 0, sizeof(h5_stat_t));
    if (HDfstat(fd, &sb) < 0)
        HSYS_GOTO_ERROR(H5E_FILE, H5E_BADFILE, NULL, "unable to fstat file");

    
    if (NULL == (file = H5FL_CALLOC(H5FD_sec2_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "unable to allocate file struct");

    file->fd = fd;
    H5_CHECKED_ASSIGN(file->eof, haddr_t, sb.st_size, h5_stat_size_t);
#ifndef H5_HAVE_PREADWRITE
    file->pos = HADDR_UNDEF;
    file->op  = OP_UNKNOWN;
#endif 
#ifdef H5_HAVE_WIN32_API
    file->hFile = (HANDLE)_get_osfhandle(fd);
    if (INVALID_HANDLE_VALUE == file->hFile)
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, NULL, "unable to get Windows file handle");

    if (!GetFileInformationByHandle((HANDLE)file->hFile, &fileinfo))
        HGOTO_ERROR(H5E_FILE, H5E_CANTOPENFILE, NULL, "unable to get Windows file information");

    file->nFileIndexHigh       = fileinfo.nFileIndexHigh;
    file->nFileIndexLow        = fileinfo.nFileIndexLow;
    file->dwVolumeSerialNumber = fileinfo.dwVolumeSerialNumber;
#else  
    file->device = sb.st_dev;
    file->inode  = sb.st_ino;
#endif 

    
    if (NULL == (plist = (H5P_genplist_t *)H5I_object(fapl_id)))
        HGOTO_ERROR(H5E_VFL, H5E_BADTYPE, NULL, "not a file access property list");

    
    if (H5FD_ignore_disabled_file_locks_p != FAIL)
        
        file->ignore_disabled_file_locks = H5FD_ignore_disabled_file_locks_p;
    else {
        
        if (H5P_get(plist, H5F_ACS_IGNORE_DISABLED_FILE_LOCKS_NAME, &file->ignore_disabled_file_locks) < 0)
            HGOTO_ERROR(H5E_VFL, H5E_CANTGET, NULL, "can't get ignore disabled file locks property");
    }

    
    strncpy(file->filename, name, sizeof(file->filename) - 1);
    file->filename[sizeof(file->filename) - 1] = '\0';

    
    if (H5P_FILE_ACCESS_DEFAULT != fapl_id) {

        
        if (H5P_exist_plist(plist, H5F_ACS_FAMILY_TO_SINGLE_NAME) > 0)
            if (H5P_get(plist, H5F_ACS_FAMILY_TO_SINGLE_NAME, &file->fam_to_single) < 0)
                HGOTO_ERROR(H5E_VFL, H5E_CANTGET, NULL, "can't get property of changing family to single");
    } 

    
    ret_value = (H5FD_t *)file;

done:
    if (NULL == ret_value) {
        if (fd >= 0)
            HDclose(fd);
        if (file)
            file = H5FL_FREE(H5FD_sec2_t, file);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_close(H5FD_t *_file)
{
    H5FD_sec2_t *file      = (H5FD_sec2_t *)_file;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(file);

    
    if (HDclose(file->fd) < 0)
        HSYS_GOTO_ERROR(H5E_IO, H5E_CANTCLOSEFILE, FAIL, "unable to close file");

    
    file = H5FL_FREE(H5FD_sec2_t, file);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5FD__sec2_cmp(const H5FD_t *_f1, const H5FD_t *_f2)
{
    const H5FD_sec2_t *f1        = (const H5FD_sec2_t *)_f1;
    const H5FD_sec2_t *f2        = (const H5FD_sec2_t *)_f2;
    int                ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

#ifdef H5_HAVE_WIN32_API
    if (f1->dwVolumeSerialNumber < f2->dwVolumeSerialNumber)
        HGOTO_DONE(-1);
    if (f1->dwVolumeSerialNumber > f2->dwVolumeSerialNumber)
        HGOTO_DONE(1);

    if (f1->nFileIndexHigh < f2->nFileIndexHigh)
        HGOTO_DONE(-1);
    if (f1->nFileIndexHigh > f2->nFileIndexHigh)
        HGOTO_DONE(1);

    if (f1->nFileIndexLow < f2->nFileIndexLow)
        HGOTO_DONE(-1);
    if (f1->nFileIndexLow > f2->nFileIndexLow)
        HGOTO_DONE(1);
#else  
    if (f1->device < f2->device)
        HGOTO_DONE(-1);
    if (f1->device > f2->device)
        HGOTO_DONE(1);

    if (f1->inode < f2->inode)
        HGOTO_DONE(-1);
    if (f1->inode > f2->inode)
        HGOTO_DONE(1);
#endif 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_query(const H5FD_t *_file, unsigned long *flags )
{
    const H5FD_sec2_t *file = (const H5FD_sec2_t *)_file; 

    FUNC_ENTER_PACKAGE_NOERR

    
    
    if (flags) {
        *flags = 0;
        *flags |= H5FD_FEAT_AGGREGATE_METADATA;  
        *flags |= H5FD_FEAT_ACCUMULATE_METADATA; 
        *flags |= H5FD_FEAT_DATA_SIEVE; 
        *flags |= H5FD_FEAT_AGGREGATE_SMALLDATA; 
        *flags |= H5FD_FEAT_POSIX_COMPAT_HANDLE; 
        *flags |=
            H5FD_FEAT_SUPPORTS_SWMR_IO; 
        *flags |= H5FD_FEAT_DEFAULT_VFD_COMPATIBLE; 

        
        if (file && file->fam_to_single)
            *flags |= H5FD_FEAT_IGNORE_DRVRINFO; 
    }                                            

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static haddr_t
H5FD__sec2_get_eoa(const H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type)
{
    const H5FD_sec2_t *file = (const H5FD_sec2_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(file->eoa)
} 

static herr_t
H5FD__sec2_set_eoa(H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type, haddr_t addr)
{
    H5FD_sec2_t *file = (H5FD_sec2_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    file->eoa = addr;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static haddr_t
H5FD__sec2_get_eof(const H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type)
{
    const H5FD_sec2_t *file = (const H5FD_sec2_t *)_file;

    FUNC_ENTER_PACKAGE_NOERR

    FUNC_LEAVE_NOAPI(file->eof)
} 

static herr_t
H5FD__sec2_get_handle(H5FD_t *_file, hid_t H5_ATTR_UNUSED fapl, void **file_handle)
{
    H5FD_sec2_t *file      = (H5FD_sec2_t *)_file;
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    if (!file_handle)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "file handle not valid");

    *file_handle = &(file->fd);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_read(H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type, hid_t H5_ATTR_UNUSED dxpl_id, haddr_t addr,
                size_t size, void *buf )
{
    H5FD_sec2_t *file      = (H5FD_sec2_t *)_file;
    HDoff_t      offset    = (HDoff_t)addr;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file && file->pub.cls);
    assert(buf);

    
    if (!H5_addr_defined(addr))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "addr undefined, addr = %llu", (unsigned long long)addr);
    if (H5FD_REGION_OVERFLOW(addr, size))
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, addr = %llu", (unsigned long long)addr);

#ifndef H5_HAVE_PREADWRITE
    
    if (addr != file->pos || OP_READ != file->op)
        if (HDlseek(file->fd, (HDoff_t)addr, SEEK_SET) < 0)
            HSYS_GOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to seek to proper position");
#endif 

    
    while (size > 0) {
        h5_posix_io_t     bytes_in   = 0;  
        h5_posix_io_ret_t bytes_read = -1; 

        
        if (size > H5_POSIX_MAX_IO_BYTES)
            bytes_in = H5_POSIX_MAX_IO_BYTES;
        else
            bytes_in = (h5_posix_io_t)size;

        do {
#ifdef H5_HAVE_PREADWRITE
            bytes_read = HDpread(file->fd, buf, bytes_in, offset);
            if (bytes_read > 0)
                offset += bytes_read;
#else
            bytes_read = HDread(file->fd, buf, bytes_in);
#endif 
        } while (-1 == bytes_read && EINTR == errno);

        if (-1 == bytes_read) { 
            int    myerrno = errno;
            time_t mytime  = time(NULL);

#ifndef H5_HAVE_PREADWRITE
            offset = HDlseek(file->fd, 0, SEEK_CUR);
#endif

            HGOTO_ERROR(H5E_IO, H5E_READERROR, FAIL,
                        "file read failed: time = %s, filename = '%s', file descriptor = %d, errno = %d, "
                        "error message = '%s', buf = %p, total read size = %llu, bytes this sub-read = %llu, "
                        "offset = %llu",
                        ctime(&mytime), file->filename, file->fd, myerrno, strerror(myerrno), buf, 
                        (unsigned long long)size, (unsigned long long)bytes_in, (unsigned long long)offset);
        } 

        if (0 == bytes_read) {
            
            memset(buf, 0, size);
            break;
        } 

        assert(bytes_read >= 0);
        assert((size_t)bytes_read <= size);

        size -= (size_t)bytes_read;
        addr += (haddr_t)bytes_read;
        buf = (char *)buf + bytes_read;
    } 

#ifndef H5_HAVE_PREADWRITE
    
    file->pos = addr;
    file->op  = OP_READ;
#endif 

done:
#ifndef H5_HAVE_PREADWRITE
    if (ret_value < 0) {
        
        file->pos = HADDR_UNDEF;
        file->op  = OP_UNKNOWN;
    }  
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_write(H5FD_t *_file, H5FD_mem_t H5_ATTR_UNUSED type, hid_t H5_ATTR_UNUSED dxpl_id, haddr_t addr,
                 size_t size, const void *buf)
{
    H5FD_sec2_t *file      = (H5FD_sec2_t *)_file;
    HDoff_t      offset    = (HDoff_t)addr;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file && file->pub.cls);
    assert(buf);

    
    if (!H5_addr_defined(addr))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "addr undefined, addr = %llu", (unsigned long long)addr);
    if (H5FD_REGION_OVERFLOW(addr, size))
        HGOTO_ERROR(H5E_ARGS, H5E_OVERFLOW, FAIL, "addr overflow, addr = %llu, size = %llu",
                    (unsigned long long)addr, (unsigned long long)size);

#ifndef H5_HAVE_PREADWRITE
    
    if (addr != file->pos || OP_WRITE != file->op)
        if (HDlseek(file->fd, (HDoff_t)addr, SEEK_SET) < 0)
            HSYS_GOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to seek to proper position");
#endif 

    
    while (size > 0) {
        h5_posix_io_t     bytes_in    = 0;  
        h5_posix_io_ret_t bytes_wrote = -1; 

        
        if (size > H5_POSIX_MAX_IO_BYTES)
            bytes_in = H5_POSIX_MAX_IO_BYTES;
        else
            bytes_in = (h5_posix_io_t)size;

        do {
#ifdef H5_HAVE_PREADWRITE
            bytes_wrote = HDpwrite(file->fd, buf, bytes_in, offset);
            if (bytes_wrote > 0)
                offset += bytes_wrote;
#else
            bytes_wrote = HDwrite(file->fd, buf, bytes_in);
#endif 
        } while (-1 == bytes_wrote && EINTR == errno);

        if (-1 == bytes_wrote) { 
            int    myerrno = errno;
            time_t mytime  = time(NULL);

#ifndef H5_HAVE_PREADWRITE
            offset = HDlseek(file->fd, 0, SEEK_CUR);
#endif

            HGOTO_ERROR(H5E_IO, H5E_WRITEERROR, FAIL,
                        "file write failed: time = %s, filename = '%s', file descriptor = %d, errno = %d, "
                        "error message = '%s', buf = %p, total write size = %llu, bytes this sub-write = "
                        "%llu, offset = %llu",
                        ctime(&mytime), file->filename, file->fd, myerrno, strerror(myerrno), buf, 
                        (unsigned long long)size, (unsigned long long)bytes_in, (unsigned long long)offset);
        } 

        assert(bytes_wrote > 0);
        assert((size_t)bytes_wrote <= size);

        size -= (size_t)bytes_wrote;
        addr += (haddr_t)bytes_wrote;
        buf = (const char *)buf + bytes_wrote;
    } 

    
#ifndef H5_HAVE_PREADWRITE
    file->pos = addr;
    file->op  = OP_WRITE;
#endif 
    if (addr > file->eof)
        file->eof = addr;

done:
#ifndef H5_HAVE_PREADWRITE
    if (ret_value < 0) {
        
        file->pos = HADDR_UNDEF;
        file->op  = OP_UNKNOWN;
    }  
#endif 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_truncate(H5FD_t *_file, hid_t H5_ATTR_UNUSED dxpl_id, bool H5_ATTR_UNUSED closing)
{
    H5FD_sec2_t *file      = (H5FD_sec2_t *)_file;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(file);

    
    if (!H5_addr_eq(file->eoa, file->eof)) {
#ifdef H5_HAVE_WIN32_API
        LARGE_INTEGER li;       
        DWORD         dwPtrLow; 
        DWORD dwError;          
        BOOL  bError;           

        
        li.QuadPart = (LONGLONG)file->eoa;

        
        dwPtrLow = SetFilePointer(file->hFile, li.LowPart, &li.HighPart, FILE_BEGIN);
        if (INVALID_SET_FILE_POINTER == dwPtrLow) {
            dwError = GetLastError();
            if (dwError != NO_ERROR)
                HGOTO_ERROR(H5E_FILE, H5E_FILEOPEN, FAIL, "unable to set file pointer");
        }

        bError = SetEndOfFile(file->hFile);
        if (0 == bError)
            HGOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to extend file properly");
#else  
        if (-1 == HDftruncate(file->fd, (HDoff_t)file->eoa))
            HSYS_GOTO_ERROR(H5E_IO, H5E_SEEKERROR, FAIL, "unable to extend file properly");
#endif 

        
        file->eof = file->eoa;

#ifndef H5_HAVE_PREADWRITE
        
        file->pos = HADDR_UNDEF;
        file->op  = OP_UNKNOWN;
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_lock(H5FD_t *_file, bool rw)
{
    H5FD_sec2_t *file = (H5FD_sec2_t *)_file; 
    int          lock_flags;                  
    herr_t       ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    assert(file);

    
    lock_flags = rw ? LOCK_EX : LOCK_SH;

    
    if (HDflock(file->fd, lock_flags | LOCK_NB) < 0) {
        if (file->ignore_disabled_file_locks && ENOSYS == errno) {
            
            errno = 0;
        }
        else
            HSYS_GOTO_ERROR(H5E_VFL, H5E_CANTLOCKFILE, FAIL, "unable to lock file");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_unlock(H5FD_t *_file)
{
    H5FD_sec2_t *file      = (H5FD_sec2_t *)_file; 
    herr_t       ret_value = SUCCEED;              

    FUNC_ENTER_PACKAGE

    assert(file);

    if (HDflock(file->fd, LOCK_UN) < 0) {
        if (file->ignore_disabled_file_locks && ENOSYS == errno) {
            
            errno = 0;
        }
        else
            HSYS_GOTO_ERROR(H5E_VFL, H5E_CANTUNLOCKFILE, FAIL, "unable to unlock file");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_delete(const char *filename, hid_t H5_ATTR_UNUSED fapl_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(filename);

    if (HDremove(filename) < 0)
        HSYS_GOTO_ERROR(H5E_VFL, H5E_CANTDELETEFILE, FAIL, "unable to delete file");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5FD__sec2_ctl(H5FD_t H5_ATTR_UNUSED *_file, uint64_t H5_ATTR_UNUSED op_code, uint64_t flags,
               const void H5_ATTR_UNUSED *input, void H5_ATTR_UNUSED **output)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    if (flags & H5FD_CTL_FAIL_IF_UNKNOWN_FLAG)
        HGOTO_ERROR(H5E_VFL, H5E_FCNTL, FAIL, "unknown op_code and fail if unknown flag is set");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
