/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Gpkg.h"      
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 
#include "H5Ppublic.h"   
#include "H5WBprivate.h" 

typedef struct {
    
    bool chk_exists; 

    
    H5G_loc_t *obj_loc; 
    bool       exists;  
} H5G_trav_slink_t;

static herr_t H5G__traverse_slink_cb(H5G_loc_t *grp_loc, const char *name, const H5O_link_t *lnk,
                                     H5G_loc_t *obj_loc, void *_udata ,
                                     H5G_own_loc_t *own_loc );
static herr_t H5G__traverse_ud(const H5G_loc_t *grp_loc, const H5O_link_t *lnk, H5G_loc_t *obj_loc ,
                               unsigned target, bool *obj_exists);
static herr_t H5G__traverse_slink(const H5G_loc_t *grp_loc, const H5O_link_t *lnk,
                                  H5G_loc_t *obj_loc , unsigned target, bool *obj_exists);
static herr_t H5G__traverse_real(const H5G_loc_t *loc, const char *name, unsigned target, H5G_traverse_t op,
                                 void *op_data);

static herr_t
H5G__traverse_slink_cb(H5G_loc_t H5_ATTR_UNUSED *grp_loc, const char H5_ATTR_UNUSED *name,
                       const H5O_link_t H5_ATTR_UNUSED *lnk, H5G_loc_t *obj_loc, void *_udata ,
                       H5G_own_loc_t *own_loc )
{
    H5G_trav_slink_t *udata     = (H5G_trav_slink_t *)_udata; 
    herr_t            ret_value = SUCCEED;                    

    FUNC_ENTER_PACKAGE

    
    if (obj_loc == NULL) {
        if (udata->chk_exists)
            udata->exists = false;
        else
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "component not found");
    } 
    else {
        
        H5O_loc_copy_deep(udata->obj_loc->oloc, obj_loc->oloc);

        
        udata->exists = true;
    } 

done:
    
    *own_loc = H5G_OWN_NONE;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__traverse_ud(const H5G_loc_t *grp_loc , const H5O_link_t *lnk, H5G_loc_t *obj_loc ,
                 unsigned target, bool *obj_exists)
{
    const H5L_class_t *link_class;     
    hid_t              cb_return = -1; 
    H5G_loc_t          grp_loc_copy;
    H5G_name_t         grp_path_copy;
    H5O_loc_t          grp_oloc_copy;
    H5G_loc_t          new_loc; 
    H5G_t             *grp;
    hid_t              cur_grp   = (-1);
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_loc);
    assert(lnk);
    assert(lnk->type >= H5L_TYPE_UD_MIN);
    assert(obj_loc);

    
    if (NULL == (link_class = H5L_find_class(lnk->type)))
        HGOTO_ERROR(H5E_SYM, H5E_NOTREGISTERED, FAIL, "unable to get UD link class");

    
    grp_loc_copy.path = &grp_path_copy;
    grp_loc_copy.oloc = &grp_oloc_copy;
    H5G_loc_reset(&grp_loc_copy);
    if (H5G_loc_copy(&grp_loc_copy, grp_loc, H5_COPY_DEEP) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, FAIL, "unable to copy object location");

    
    if (NULL == (grp = H5G_open(&grp_loc_copy)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open group");
    if ((cur_grp = H5VL_wrap_register(H5I_GROUP, grp, false)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREGISTER, FAIL, "unable to register group");

    
    if (target & H5G_TARGET_EXISTS)
        H5E_pause_stack();

        
#ifndef H5_NO_DEPRECATED_SYMBOLS
    
    if (link_class->version == H5L_LINK_CLASS_T_VERS_0) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                cb_return = (((const H5L_class_0_t *)link_class)->trav_func)(
                    lnk->name, cur_grp, lnk->u.ud.udata, lnk->u.ud.size, H5CX_get_lapl());
            }
        H5_AFTER_USER_CB(FAIL)
    }
    else {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                cb_return = (link_class->trav_func)(lnk->name, cur_grp, lnk->u.ud.udata, lnk->u.ud.size,
                                                    H5CX_get_lapl(), H5CX_get_dxpl());
            }
        H5_AFTER_USER_CB(FAIL)
    }
#else  
    
    H5_BEFORE_USER_CB(FAIL)
        {
            cb_return = (link_class->trav_func)(lnk->name, cur_grp, lnk->u.ud.udata, lnk->u.ud.size,
                                                H5CX_get_lapl(), H5CX_get_dxpl());
        }
    H5_AFTER_USER_CB(FAIL)
#endif 

    
    if (target & H5G_TARGET_EXISTS)
        H5E_resume_stack();

    
    if (cb_return < 0) {
        
        if (target & H5G_TARGET_EXISTS) {
            
            *obj_exists = false;

            
            HGOTO_DONE(SUCCEED);
        } 
        
        else
            HGOTO_ERROR(H5E_SYM, H5E_BADID, FAIL, "traversal callback returned invalid ID");
    } 

    
    if (H5G_loc(cb_return, &new_loc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "unable to get object location from ID");

    
    H5G_loc_free(obj_loc);

    
    H5G_loc_copy(obj_loc, &new_loc, H5_COPY_DEEP);

    
    if (H5O_loc_hold_file(obj_loc->oloc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to hold file open");

    
    if (H5I_dec_ref(cb_return) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to close ID from UD callback");
    cb_return = (hid_t)(-1);

done:
    
    if (cur_grp > 0 && H5I_dec_ref(cur_grp) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to close ID for current location");

    if (ret_value < 0 && cb_return > 0 && H5I_dec_ref(cb_return) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to close ID from UD callback");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__traverse_slink(const H5G_loc_t *grp_loc, const H5O_link_t *lnk, H5G_loc_t *obj_loc ,
                    unsigned target, bool *obj_exists)
{
    H5G_trav_slink_t udata;                     
    H5G_name_t       tmp_obj_path;              
    bool             tmp_obj_path_set = false;  
    H5O_loc_t        tmp_grp_oloc;              
    H5G_name_t       tmp_grp_path;              
    H5G_loc_t        tmp_grp_loc;               
    bool             tmp_grp_loc_set = false;   
    herr_t           ret_value       = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_loc);
    assert(lnk);
    assert(lnk->type == H5L_TYPE_SOFT);

    
    tmp_grp_loc.oloc = &tmp_grp_oloc;
    tmp_grp_loc.path = &tmp_grp_path;

    
    H5G_loc_reset(&tmp_grp_loc);
    H5G_name_reset(&tmp_obj_path);

    
    
    H5G_loc_copy(&tmp_grp_loc, grp_loc, H5_COPY_DEEP);
    tmp_grp_loc_set = true;

    
    
    H5G_name_copy(&tmp_obj_path, obj_loc->path, H5_COPY_SHALLOW);
    tmp_obj_path_set = true;

    
    udata.chk_exists = (target & H5G_TARGET_EXISTS) ? true : false;
    udata.exists     = false;
    udata.obj_loc    = obj_loc;

    
    if (H5G__traverse_real(&tmp_grp_loc, lnk->u.soft.name, target, H5G__traverse_slink_cb, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to follow symbolic link");

    
    *obj_exists = udata.exists;

done:
    
    if (tmp_obj_path_set) {
        H5G_name_free(obj_loc->path);
        H5G_name_copy(obj_loc->path, &tmp_obj_path, H5_COPY_SHALLOW);
    } 

    
    if (tmp_grp_loc_set)
        H5G_loc_free(&tmp_grp_loc);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__traverse_special(const H5G_loc_t *grp_loc, const H5O_link_t *lnk, unsigned target, bool last_comp,
                      H5G_loc_t *obj_loc, bool *obj_exists)
{
    size_t nlinks;              
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_loc);
    assert(lnk);
    assert(obj_loc);

    
    if (H5L_TYPE_SOFT == lnk->type && (0 == (target & H5G_TARGET_SLINK) || !last_comp)) {

        
        if (H5CX_get_nlinks(&nlinks) < 0)
            HGOTO_ERROR(H5E_LINK, H5E_CANTGET, FAIL, "unable to retrieve # of soft / UD links to traverse");

        
        if ((nlinks)-- <= 0)
            HGOTO_ERROR(H5E_LINK, H5E_NLINKS, FAIL, "too many links");

        
        if (H5CX_set_nlinks(nlinks) < 0)
            HGOTO_ERROR(H5E_LINK, H5E_CANTSET, FAIL, "can't update # of soft / UD links to traverse");

        
        if (H5G__traverse_slink(grp_loc, lnk, obj_loc, (target & H5G_TARGET_EXISTS), obj_exists) < 0)
            HGOTO_ERROR(H5E_LINK, H5E_TRAVERSE, FAIL, "symbolic link traversal failed");
    } 

    
    if (lnk->type >= H5L_TYPE_UD_MIN && (0 == (target & H5G_TARGET_UDLINK) || !last_comp)) {

        
        if (H5CX_get_nlinks(&nlinks) < 0)
            HGOTO_ERROR(H5E_LINK, H5E_CANTGET, FAIL, "unable to retrieve # of soft / UD links to traverse");

        
        if ((nlinks)-- <= 0)
            HGOTO_ERROR(H5E_LINK, H5E_NLINKS, FAIL, "too many links");

        
        if (H5CX_set_nlinks(nlinks) < 0)
            HGOTO_ERROR(H5E_LINK, H5E_CANTSET, FAIL, "can't update # of soft / UD links to traverse");

        
        if (H5G__traverse_ud(grp_loc, lnk, obj_loc, (target & H5G_TARGET_EXISTS), obj_exists) < 0)
            HGOTO_ERROR(H5E_LINK, H5E_TRAVERSE, FAIL, "user-defined link traversal failed");
    } 

    
    if (H5_addr_defined(obj_loc->oloc->addr) && (0 == (target & H5G_TARGET_MOUNT) || !last_comp)) {
        if (H5F_traverse_mount(obj_loc->oloc ) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "mount point traversal failed");
    } 

    
    if (grp_loc->oloc->holding_file && grp_loc->oloc->file == obj_loc->oloc->file)
        if (H5O_loc_hold_file(obj_loc->oloc) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to hold file open");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__traverse_real(const H5G_loc_t *_loc, const char *name, unsigned target, H5G_traverse_t op, void *op_data)
{
    H5G_loc_t     loc;                    
    H5O_loc_t     grp_oloc;               
    H5G_name_t    grp_path;               
    H5G_loc_t     grp_loc;                
    H5O_loc_t     obj_oloc;               
    H5G_name_t    obj_path;               
    H5G_loc_t     obj_loc;                
    size_t        nchars;                 
    H5O_link_t    lnk;                    
    bool          link_valid    = false;  
    bool          obj_loc_valid = false;  
    H5G_own_loc_t own_loc = H5G_OWN_NONE; 
    bool          group_copy = false;     
    char          comp_buf[1024];         
    char         *comp;                   
    H5WB_t       *wb        = NULL;       
    bool          last_comp = false; 
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(_loc);
    assert(name);
    assert(op);

    
    
    if ('/' == *name) {
        H5G_t *root_grp; 

        
        root_grp = H5G_rootof(_loc->oloc->file);
        assert(root_grp);

        
        loc.oloc = &(root_grp->oloc);
        loc.path = &(root_grp->path);
    } 
    else {
        loc.oloc = _loc->oloc;
        loc.path = _loc->path;
    } 

    
    grp_loc.oloc = &grp_oloc;
    grp_loc.path = &grp_path;
    obj_loc.oloc = &obj_oloc;
    obj_loc.path = &obj_path;

#if defined(H5_USING_MEMCHECKER) || !defined(NDEBUG)
    
    if (H5G_loc_reset(&grp_loc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to reset location");
#endif 

    
    if (H5G_loc_copy(&grp_loc, &loc, H5_COPY_DEEP) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to copy location");
    group_copy = true;

    
    if (H5G_loc_reset(&obj_loc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to reset location");

    
    if (NULL == (wb = H5WB_wrap(comp_buf, sizeof(comp_buf))))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't wrap buffer");

    
    if (NULL == (comp = (char *)H5WB_actual(wb, (strlen(name) + 1))))
        HGOTO_ERROR(H5E_SYM, H5E_NOSPACE, FAIL, "can't get actual buffer");

    
    while ((name = H5G__component(name, &nchars)) && *name) {
        const char *s;             
        bool        lookup_status; 
        bool        obj_exists;    

        
        H5MM_memcpy(comp, name, nchars);
        comp[nchars] = '\0';

        
        if ('.' == comp[0] && !comp[1]) {
            name += nchars;
            continue;
        } 

        
        if (!((s = H5G__component(name + nchars, NULL)) && *s))
            last_comp = true;

        
        if (link_valid) {
            H5O_msg_reset(H5O_LINK_ID, &lnk);
            link_valid = false;
        } 

        
        lookup_status = false;
        if (H5G__obj_lookup(grp_loc.oloc, comp, &lookup_status, &lnk ) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't look up component");
        obj_exists = false;

        
        if (lookup_status) {
            
            assert(lnk.type >= H5L_TYPE_HARD);
            assert(!strcmp(comp, lnk.name));
            link_valid = true;

            
            if (H5G__link_to_loc(&grp_loc, &lnk, &obj_loc) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "cannot initialize object location");
            obj_loc_valid = true;

            
            obj_exists = true;

            
            
            if (H5G__traverse_special(&grp_loc, &lnk, target, last_comp, &obj_loc, &obj_exists) < 0)
                HGOTO_ERROR(H5E_LINK, H5E_TRAVERSE, FAIL, "special link traversal failed");
        } 

        
        if (last_comp) {
            H5O_link_t *cb_lnk; 
            H5G_loc_t  *cb_loc; 

            
            if (lookup_status) {
                cb_lnk = &lnk;
                if (obj_exists)
                    cb_loc = &obj_loc;
                else
                    cb_loc = NULL;
            } 
            else {
                assert(!obj_loc_valid);
                cb_lnk = NULL;
                cb_loc = NULL;
            } 

            
            if ((op)(&grp_loc, comp, cb_lnk, cb_loc, op_data, &own_loc) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CALLBACK, FAIL, "traversal operator failed");

            HGOTO_DONE(SUCCEED);
        } 

        
        if (!lookup_status) {
            
            if (target & H5G_CRT_INTMD_GROUP) {
                const H5O_ginfo_t  def_ginfo = H5G_CRT_GROUP_INFO_DEF; 
                const H5O_linfo_t  def_linfo = H5G_CRT_LINK_INFO_DEF;  
                const H5O_pline_t  def_pline = H5O_CRT_PIPELINE_DEF;   
                H5O_ginfo_t        par_ginfo; 
                H5O_linfo_t        par_linfo; 
                H5O_pline_t        par_pline; 
                H5O_linfo_t        tmp_linfo; 
                htri_t             exists;    
                const H5O_ginfo_t *ginfo;     
                const H5O_linfo_t *linfo;     
                const H5O_pline_t *pline;     
                H5G_obj_create_t   gcrt_info; 
                H5O_obj_create_t  *ocrt_info; 

                
                
                if ((exists = H5O_msg_exists(grp_loc.oloc, H5O_GINFO_ID)) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to read object header");
                if (exists) {
                    
                    if (NULL == H5O_msg_read(grp_loc.oloc, H5O_GINFO_ID, &par_ginfo))
                        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "group info message not present");

                    
                    ginfo = &par_ginfo;
                } 
                else
                    
                    ginfo = &def_ginfo;

                
                
                
                if ((exists = H5G__obj_get_linfo(grp_loc.oloc, &par_linfo)) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to read object header");
                if (exists) {
                    
                    H5MM_memcpy(&tmp_linfo, &def_linfo, sizeof(H5O_linfo_t));
                    tmp_linfo.track_corder = par_linfo.track_corder;
                    tmp_linfo.index_corder = par_linfo.index_corder;
                    linfo                  = &tmp_linfo;
                } 
                else
                    
                    linfo = &def_linfo;

                
                
                if ((exists = H5O_msg_exists(grp_loc.oloc, H5O_PLINE_ID)) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to read object header");
                if (exists) {
                    
                    if (NULL == H5O_msg_read(grp_loc.oloc, H5O_PLINE_ID, &par_pline))
                        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "filter pipeline message not present");

                    
                    pline = &par_pline;
                } 
                else
                    
                    pline = &def_pline;

                
                gcrt_info.gcpl_id = H5P_GROUP_CREATE_DEFAULT;
                
                if ((target & H5G_CRT_OBJ) && (ocrt_info = H5L_OCRT_INFO(op_data)) != NULL) {
                    if (ocrt_info->obj_type == H5O_TYPE_GROUP)
                        gcrt_info.gcpl_id = H5G_OBJ_ID(ocrt_info->crt_info);
                    else if (ocrt_info->obj_type == H5O_TYPE_DATASET)
                        gcrt_info.gcpl_id = H5D_OBJ_ID(ocrt_info->crt_info);
                }

                gcrt_info.cache_type = H5G_NOTHING_CACHED;
                memset(&gcrt_info.cache, 0, sizeof(gcrt_info.cache));
                if (H5G__obj_create_real(grp_oloc.file, ginfo, linfo, pline, &gcrt_info,
                                         obj_loc.oloc ) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create group entry");

                
                if (H5G__loc_insert(&grp_loc, comp, &obj_loc, H5O_TYPE_GROUP, &gcrt_info) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert intermediate group");

                
                if (H5O_dec_rc_by_loc(obj_loc.oloc) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTDEC, FAIL,
                                "unable to decrement refcount on newly created object");

                
                if (H5O_close(obj_loc.oloc, NULL) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to close");

                
                if (grp_loc.oloc->holding_file)
                    if (H5O_loc_hold_file(obj_loc.oloc) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to hold file open");

                
                H5_WARN_CAST_AWAY_CONST_OFF
                if (ginfo != &def_ginfo)
                    if (H5O_msg_reset(H5O_GINFO_ID, (void *)ginfo) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to reset group info message");
                if (linfo != &def_linfo)
                    if (H5O_msg_reset(H5O_LINFO_ID, (void *)linfo) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to reset link info message");
                if (pline != &def_pline)
                    if (H5O_msg_reset(H5O_PLINE_ID, (void *)pline) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to reset I/O pipeline message");
                H5_WARN_CAST_AWAY_CONST_ON
            } 
            else
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "component not found");
        } 

        

        
        H5G_loc_free(&grp_loc);
        H5G_loc_copy(&grp_loc, &obj_loc, H5_COPY_SHALLOW);
        H5G_loc_reset(&obj_loc);
        obj_loc_valid = false;

        
        name += nchars;
    } 

    
    
    assert(group_copy);
    if ((op)(NULL, ".", NULL, &grp_loc, op_data, &own_loc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTNEXT, FAIL, "traversal operator failed");

    
    assert(!(own_loc & H5G_OWN_GRP_LOC));
    if (own_loc & H5G_OWN_OBJ_LOC)
        own_loc |= H5G_OWN_GRP_LOC;

done:
    
    if (ret_value < 0)
        own_loc = H5G_OWN_NONE;

    
    if (obj_loc_valid && !(own_loc & H5G_OWN_OBJ_LOC))
        H5G_loc_free(&obj_loc);
    if (group_copy && !(own_loc & H5G_OWN_GRP_LOC))
        H5G_loc_free(&grp_loc);

    
    if (link_valid)
        if (H5O_msg_reset(H5O_LINK_ID, &lnk) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to reset link message");

    
    if (wb && H5WB_unwrap(wb) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "can't release wrapped buffer");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_traverse(const H5G_loc_t *loc, const char *name, unsigned target, H5G_traverse_t op, void *op_data)
{
    size_t orig_nlinks;         
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (!name || !*name)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "no name given");
    if (!loc)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "no starting location");
    if (!op)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "no operation provided");

    
    if (H5CX_get_nlinks(&orig_nlinks) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to retrieve # of soft / UD links to traverse");

    
    H5_BEGIN_TAG(H5AC__INVALID_TAG)

    
    if (H5G__traverse_real(loc, name, target, op, op_data) < 0)
        HGOTO_ERROR_TAG(H5E_SYM, H5E_NOTFOUND, FAIL, "internal path traversal failed");

    
    H5_END_TAG

    
    if (H5CX_set_nlinks(orig_nlinks) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't reset # of soft / UD links to traverse");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
