/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Opkg.h"      

static void  *H5O__refcount_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                   size_t p_size, const uint8_t *p);
static herr_t H5O__refcount_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                   const void *_mesg);
static void  *H5O__refcount_copy(const void *_mesg, void *_dest);
static size_t H5O__refcount_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__refcount_free(void *_mesg);
static herr_t H5O__refcount_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                          const H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__refcount_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_REFCOUNT[1] = {{
    H5O_REFCOUNT_ID,             
    "refcount",                  
    sizeof(H5O_refcount_t),      
    0,                           
    H5O__refcount_decode,        
    H5O__refcount_encode,        
    H5O__refcount_copy,          
    H5O__refcount_size,          
    NULL,                        
    H5O__refcount_free,          
    NULL,                        
    NULL,                        
    NULL,                        
    NULL,                        
    H5O__refcount_pre_copy_file, 
    NULL,                        
    NULL,                        
    NULL,                        
    NULL,                        
    H5O__refcount_debug          
}};

#define H5O_REFCOUNT_VERSION 0

H5FL_DEFINE_STATIC(H5O_refcount_t);

static void *
H5O__refcount_decode(H5F_t H5_ATTR_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh,
                     unsigned H5_ATTR_UNUSED mesg_flags, unsigned H5_ATTR_UNUSED *ioflags, size_t p_size,
                     const uint8_t *p)
{
    H5O_refcount_t *refcount  = NULL;           
    const uint8_t  *p_end     = p + p_size - 1; 
    void           *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*p++ != H5O_REFCOUNT_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message");

    
    if (NULL == (refcount = H5FL_MALLOC(H5O_refcount_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 4, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT32DECODE(p, *refcount);

    
    ret_value = refcount;

done:
    if (!ret_value && refcount)
        H5FL_FREE(H5O_refcount_t, refcount);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__refcount_encode(H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared,
                     size_t H5_ATTR_UNUSED p_size, uint8_t *p, const void *_mesg)
{
    const H5O_refcount_t *refcount = (const H5O_refcount_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(refcount);

    
    *p++ = H5O_REFCOUNT_VERSION;

    
    UINT32ENCODE(p, *refcount);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__refcount_copy(const void *_mesg, void *_dest)
{
    const H5O_refcount_t *refcount  = (const H5O_refcount_t *)_mesg;
    H5O_refcount_t       *dest      = (H5O_refcount_t *)_dest;
    void                 *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(refcount);
    if (!dest && NULL == (dest = H5FL_MALLOC(H5O_refcount_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *refcount;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__refcount_size(const H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared,
                   const void H5_ATTR_UNUSED *_mesg)
{
    size_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value = 1    
                + 4; 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__refcount_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_refcount_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__refcount_pre_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const void H5_ATTR_UNUSED *native_src,
                            bool *deleted, const H5O_copy_t H5_ATTR_UNUSED *cpy_info,
                            void H5_ATTR_UNUSED *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(deleted);
    assert(cpy_info);

    
    *deleted = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__refcount_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_refcount_t *refcount = (const H5O_refcount_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(refcount);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth, "Number of links:", (unsigned)*refcount);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
