/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5F_FRIEND      
#include "H5MFmodule.h" 
#define H5MF_DEBUGGING  

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Fpkg.h"     
#include "H5MFpkg.h"    

typedef struct {
    H5FS_t *fspace; 
    FILE   *stream; 
    int     indent; 
    int     fwidth; 
} H5MF_debug_iter_ud_t;

static herr_t H5MF__sects_debug_cb(H5FS_section_info_t *_sect, void *_udata);

static herr_t
H5MF__sects_debug_cb(H5FS_section_info_t *_sect, void *_udata)
{
    H5MF_free_section_t  *sect      = (H5MF_free_section_t *)_sect;   
    H5MF_debug_iter_ud_t *udata     = (H5MF_debug_iter_ud_t *)_udata; 
    herr_t                ret_value = SUCCEED;                        

    FUNC_ENTER_PACKAGE

    
    assert(sect);
    assert(udata);

    
    Rfprintf(udata->stream, "%*s%-*s %s\n", udata->indent, "", udata->fwidth, "Section type:",
            (sect->sect_info.type == H5MF_FSPACE_SECT_SIMPLE
                 ? "simple"
                 : (sect->sect_info.type == H5MF_FSPACE_SECT_SMALL
                        ? "small"
                        : (sect->sect_info.type == H5MF_FSPACE_SECT_LARGE ? "large" : "unknown"))));
    Rfprintf(udata->stream, "%*s%-*s %" PRIuHADDR "\n", udata->indent, "", udata->fwidth,
            "Section address:", sect->sect_info.addr);
    Rfprintf(udata->stream, "%*s%-*s %" PRIuHSIZE "\n", udata->indent, "", udata->fwidth,
            "Section size:", sect->sect_info.size);
    Rfprintf(udata->stream, "%*s%-*s %" PRIuHADDR "\n", udata->indent, "", udata->fwidth,
            "End of section:", (haddr_t)((sect->sect_info.addr + sect->sect_info.size) - 1));
    Rfprintf(udata->stream, "%*s%-*s %s\n", udata->indent, "", udata->fwidth,
            "Section state:", (sect->sect_info.state == H5FS_SECT_LIVE ? "live" : "serialized"));

    
    if (H5FS_sect_debug(udata->fspace, _sect, udata->stream, udata->indent + 3, MAX(0, udata->fwidth - 3)) <
        0)
        HGOTO_ERROR(H5E_RESOURCE, H5E_BADITER, FAIL, "can't dump section's debugging info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5MF_sects_debug(H5F_t *f, haddr_t fs_addr, FILE *stream, int indent, int fwidth)
{
    H5F_mem_page_t type;                
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_TAG(H5AC__FREESPACE_TAG, FAIL)

    
    assert(f);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    for (type = H5F_MEM_PAGE_DEFAULT; type < H5F_MEM_PAGE_NTYPES; type++)
        if (H5_addr_eq(f->shared->fs_addr[type], fs_addr)) {
            if (!f->shared->fs_man[type])
                if (H5MF__open_fstype(f, type) < 0)
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, FAIL, "can't initialize file free space");

            if (f->shared->fs_man[type]) {
                H5MF_debug_iter_ud_t udata; 

                
                udata.fspace = f->shared->fs_man[type];
                udata.stream = stream;
                udata.indent = indent;
                udata.fwidth = fwidth;

                
                if (H5FS_sect_iterate(f, f->shared->fs_man[type], H5MF__sects_debug_cb, &udata) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_BADITER, FAIL, "can't iterate over heap's free space");

                
                if (H5FS_close(f, f->shared->fs_man[type]) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't release free space info");
            } 
            break;
        } 

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

#ifdef H5MF_ALLOC_DEBUG_DUMP

herr_t
H5MF__sects_dump(H5F_t *f, FILE *stream)
{
    haddr_t eoa;                 
    int     indent    = 0;       
    int     fwidth    = 50;      
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(H5AC__FREESPACE_TAG)
#ifdef H5MF_ALLOC_DEBUG
    Rfprintf(Rstderr, "%s: Dumping file free space sections\n", __func__);
#endif 

    
    assert(f);
    assert(stream);

    
    if (HADDR_UNDEF == (eoa = H5F_get_eoa(f, H5FD_MEM_DEFAULT)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "driver get_eoa request failed");
#ifdef H5MF_ALLOC_DEBUG
    Rfprintf(Rstderr, "%s: for type = H5FD_MEM_DEFAULT, eoa = %" PRIuHADDR "\n", __func__, eoa);
#endif 

    if (H5F_PAGED_AGGR(f)) {  
        H5F_mem_page_t ptype; 

        for (ptype = H5F_MEM_PAGE_META; ptype < H5F_MEM_PAGE_NTYPES; ptype++) {
            
            Rfprintf(stream, "%*sFile Free Space Info for type = %u:\n", indent, "", (unsigned)ptype);

            
            Rfprintf(stream, "%*sSections:\n", indent + 3, "");

            
            if (f->shared->fs_man[ptype]) {
                H5MF_debug_iter_ud_t udata; 

                
                udata.fspace = f->shared->fs_man[ptype];
                udata.stream = stream;
                udata.indent = indent + 6;
                udata.fwidth = MAX(0, fwidth - 6);

                
                if (H5FS_sect_iterate(f, f->shared->fs_man[ptype], H5MF__sects_debug_cb, &udata) < 0)
                    HGOTO_ERROR(H5E_HEAP, H5E_BADITER, FAIL, "can't iterate over heap's free space");
            } 
            else
                
                Rfprintf(stream, "%*s<none>\n", indent + 6, "");
        }                                  
    }                                      
    else {                                 
        H5FD_mem_t atype;                  
        haddr_t    ma_addr  = HADDR_UNDEF; 
        hsize_t    ma_size  = 0;           
        haddr_t    sda_addr = HADDR_UNDEF; 
        hsize_t    sda_size = 0;           

        
        H5MF__aggr_query(f, &(f->shared->meta_aggr), &ma_addr, &ma_size);
#ifdef H5MF_ALLOC_DEBUG
        Rfprintf(Rstderr,
                "%s: ma_addr = %" PRIuHADDR ", ma_size = %" PRIuHSIZE ", end of ma = %" PRIuHADDR "\n",
                __func__, ma_addr, ma_size, (haddr_t)((ma_addr + ma_size) - 1));
#endif 

        
        H5MF__aggr_query(f, &(f->shared->sdata_aggr), &sda_addr, &sda_size);
#ifdef H5MF_ALLOC_DEBUG
        Rfprintf(Rstderr,
                "%s: sda_addr = %" PRIuHADDR ", sda_size = %" PRIuHSIZE ", end of sda = %" PRIuHADDR "\n",
                __func__, sda_addr, sda_size, (haddr_t)((sda_addr + sda_size) - 1));
#endif 

        
        for (atype = H5FD_MEM_DEFAULT; atype < H5FD_MEM_NTYPES; atype++) {
            
            Rfprintf(stream, "%*sFile Free Space Info for type = %u:\n", indent, "", (unsigned)atype);

            
            if (H5FD_MEM_DEFAULT == f->shared->fs_type_map[atype] || atype == f->shared->fs_type_map[atype]) {
                
                if (HADDR_UNDEF == (eoa = H5F_get_eoa(f, atype)))
                    HGOTO_ERROR(H5E_RESOURCE, H5E_CANTGET, FAIL, "driver get_eoa request failed");
                Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent + 3, "", MAX(0, fwidth - 3), "eoa:", eoa);

                
                Rfprintf(stream, "%*sSections:\n", indent + 3, "");

                
                if (f->shared->fs_man[atype]) {
                    H5MF_debug_iter_ud_t udata; 

                    
                    udata.fspace = f->shared->fs_man[atype];
                    udata.stream = stream;
                    udata.indent = indent + 6;
                    udata.fwidth = MAX(0, fwidth - 6);

                    
                    if (H5FS_sect_iterate(f, f->shared->fs_man[atype], H5MF__sects_debug_cb, &udata) < 0)
                        HGOTO_ERROR(H5E_HEAP, H5E_BADITER, FAIL, "can't iterate over heap's free space");
                }    
                else 
                    Rfprintf(stream, "%*s<none>\n", indent + 6, "");
            } 
            else
                Rfprintf(stream, "%*sMapped to type = %u\n", indent, "",
                        (unsigned)f->shared->fs_type_map[atype]);
        } 
    }     

done:
    Rfprintf(Rstderr, "%s: Done dumping file free space sections\n", __func__);
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 
#endif 
