% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/is_converged.R
\name{is_converged}
\alias{is_converged}
\title{Convergence test for mixed effects models}
\usage{
is_converged(x, tolerance = 0.001, ...)
}
\arguments{
\item{x}{A model object from class \code{merMod}, \code{glmmTMB}, \code{glm}, \code{lavaan} or
\verb{_glm}.}

\item{tolerance}{Indicates up to which value the convergence result is
accepted. The smaller \code{tolerance} is, the stricter the test will be.}

\item{...}{Currently not used.}
}
\value{
\code{TRUE} if convergence is fine and \code{FALSE} if convergence
is suspicious. Additionally, the convergence value is returned as attribute.
}
\description{
\code{is_converged()} provides an alternative convergence
test for \code{merMod}-objects.
}
\section{Convergence and log-likelihood}{

Convergence problems typically arise when the model hasn't converged to a
solution where the log-likelihood has a true maximum. This may result in
unreliable and overly complex (or non-estimable) estimates and standard
errors.
}

\section{Inspect model convergence}{

\strong{lme4} performs a convergence-check (see \code{?lme4::convergence}), however, as
discussed \href{https://github.com/lme4/lme4/issues/120}{here} and suggested by
one of the lme4-authors in \href{https://github.com/lme4/lme4/issues/120#issuecomment-39920269}{this comment},
this check can be too strict. \code{is_converged()} thus provides an alternative
convergence test for \code{merMod}-objects.
}

\section{Resolving convergence issues}{

Convergence issues are not easy to diagnose. The help page on
\code{?lme4::convergence} provides most of the current advice about how to resolve
convergence issues. In general, convergence issues may be addressed by one or
more of the following strategies: 1. Rescale continuous predictors; 2. try a
different optimizer; 3. increase the number of iterations; or, if everything
else fails, 4. simplify the model. Another clue might be large parameter
values, e.g. estimates (on the scale of the linear predictor) larger than 10
in (non-identity link) generalized linear model \emph{might} indicate complete
separation, which can be addressed by regularization, e.g. penalized
regression or Bayesian regression with appropriate priors on the fixed
effects.
}

\section{Convergence versus Singularity}{

Note the different meaning between singularity and convergence: singularity
indicates an issue with the "true" best estimate, i.e. whether the maximum
likelihood estimation for the variance-covariance matrix of the random effects
is positive definite or only semi-definite. Convergence is a question of
whether we can assume that the numerical optimization has worked correctly
or not. A convergence failure means the optimizer (the algorithm) could not
find a stable solution (\emph{Bates et. al 2015}).
}

\examples{
\dontshow{if (require("lme4", quietly = TRUE)) withAutoprint(\{ # examplesIf}
library(lme4)
data(cbpp)
set.seed(1)
cbpp$x <- rnorm(nrow(cbpp))
cbpp$x2 <- runif(nrow(cbpp))

model <- glmer(
  cbind(incidence, size - incidence) ~ period + x + x2 + (1 + x | herd),
  data = cbpp,
  family = binomial()
)

is_converged(model)
\dontshow{\}) # examplesIf}
\dontshow{if (getOption("warn") < 2L && require("glmmTMB")) withAutoprint(\{ # examplesIf}
\donttest{
library(glmmTMB)
model <- glmmTMB(
  Sepal.Length ~ poly(Petal.Width, 4) * poly(Petal.Length, 4) +
    (1 + poly(Petal.Width, 4) | Species),
  data = iris
)

is_converged(model)
}
\dontshow{\}) # examplesIf}
}
\references{
Bates, D., Mächler, M., Bolker, B., and Walker, S. (2015). Fitting Linear
Mixed-Effects Models Using lme4. Journal of Statistical Software, 67(1),
1-48. \doi{10.18637/jss.v067.i01}
}
