% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/op-subset.R
\name{op_subset}
\alias{op_subset}
\alias{op_subset<-}
\alias{op_subset_set}
\title{Subset elements from a tensor}
\usage{
op_subset(x, ...)

op_subset(x, ...) <- value

op_subset_set(x, ..., value)
}
\arguments{
\item{x}{Input tensor.}

\item{...}{Indices specifying elements to extract. Each argument in \code{...} can
be:
\itemize{
\item An integer scalar
\item A 1-d integer or logical vector
\item \code{NULL} or \code{newaxis}
\item The \code{..} symbol
\item A slice expression using \code{:}
}

If only a single argument is supplied to \code{...}, then \code{..1} can also be:
\itemize{
\item A logical array with the same shape as \code{x}
\item An integer matrix where \code{ncol(..1) == op_rank(x)}
}}

\item{value}{new value to replace the selected subset with.}
}
\value{
A tensor containing the subset of elements.
}
\description{
Extract elements from a tensor using common R-style \code{[} indexing idioms. This
function can also be conveniently accessed via the syntax \code{tensor@r[...]}.
}
\details{
While the semantics are similar to R's \code{[}, there are some differences:
}
\section{Differences from R's \code{[}:}{
\itemize{
\item Negative indices follow Python-style indexing, counting from the end of the array.
\item \code{NULL} or \code{newaxis} adds a new dimension (equivalent to \code{op_expand_dims()}).
\item If fewer indices than dimensions (\code{op_rank(x)}) are provided, missing dimensions
are implicitly filled. For example, if \code{x} is a matrix, \code{x[1]} returns the first row.
\item \code{..} or \code{all_dims()} expands to include all unspecified dimensions (see examples).
\item Extended slicing syntax (\code{:}) is supported, including:
\itemize{
\item Strided steps: \code{x@r[start:end:step]}
\item \code{NA} values for \code{start} and \code{end}. \code{NA} for \code{start} defaults to \code{1}, and
\code{NA} for \code{end} defaults to the axis size.
}
\item A logical array matching the shape of \code{x} selects elements in row-wise order.
}
}

\section{Similarities with R's \code{[}:}{
Similarities to R's \code{[} (differences from Python's \code{[}):
\itemize{
\item Positive indices are 1-based.
\item Slices (\code{x[start:end]}) are inclusive of \code{end}.
\item 1-d logical/integer arrays subset along their respective axis.
Multiple vectors provided for different axes return intersected subsets.
\item A single integer matrix with \code{ncol(i) == op_rank(x)} selects elements by
coordinates. Each row in the matrix specifies the location of one value, where each column
corresponds to an axis in the tensor being subsetted.  This means you use a
2-column matrix to subset a matrix, a 3-column matrix to subset a 3d array,
and so on.
}
}

\section{Examples}{
\if{html}{\out{<div class="sourceCode r">}}\preformatted{(x <- op_arange(5L) + 10L)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13 14 15], shape=(5), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Basic example, get first element
op_subset(x, 1)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(11, shape=(), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Use `@r[` syntax
x@r[1]           # same as `op_subset(x, 1)`
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(11, shape=(), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[1:2]         # get the first 2 elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12], shape=(2), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[c(1, 3)]     # first and third element
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 13], shape=(2), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Negative indices
x@r[-1]          # last element
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(15, shape=(), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[-2]          # second to last element
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(14, shape=(), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[c(-1, -2)]   # last and second to last elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([15 14], shape=(2), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[c(-2, -1)]   # second to last and last elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([14 15], shape=(2), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[c(1, -1)]    # first and last elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 15], shape=(2), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Slices
x@r[1:3]          # first 3 elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13], shape=(3), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[NA:3]         # first 3 elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13], shape=(3), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[1:5]          # all elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13 14 15], shape=(5), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[1:-1]         # all elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13 14 15], shape=(5), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[NA:NA]        # all elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13 14 15], shape=(5), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[]             # all elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13 14 15], shape=(5), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[1:-2]         # drop last element
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13 14], shape=(4), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[NA:-2]        # drop last element
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([11 12 13 14], shape=(4), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[2:NA]         # drop first element
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([12 13 14 15], shape=(4), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 2D array examples
xr <- array(1:12, c(3, 4))
x <- op_convert_to_tensor(xr)

# Basic subsetting
x@r[1, ]      # first row
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([ 1  4  7 10], shape=(4), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[1]        # also first row! Missing axes are implicitly inserted
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([ 1  4  7 10], shape=(4), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[-1]       # last row
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([ 3  6  9 12], shape=(4), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[, 2]      # second column
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([4 5 6], shape=(3), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[, 2:2]    # second column, but shape preserved (like [, drop=FALSE])
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[4]
##  [5]
##  [6]], shape=(3, 1), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Subsetting with a boolean array
# Note: extracted elements are selected row-wise, not column-wise
mask <- x >= 6
x@r[mask]             # returns a 1D tensor
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([ 7 10  8 11  6  9 12], shape=(7), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x.r <- as.array(x)
mask.r <- as.array(mask)
# as.array(x)[mask] selects column-wise. Use `aperm()` to reverse search order.
all(aperm(x.r)[aperm(mask.r)] == as.array(x@r[mask]))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# Subsetting with a matrix of index positions
indices <- rbind(c(1, 1), c(2, 2), c(3, 3))
x@r[indices] # get diagonal elements
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor([1 5 9], shape=(3), dtype=int32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x.r[indices] # same as subsetting an R array
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] 1 5 9

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# 3D array examples
# Image: 4x4 pixels, 3 colors (RGB)
# Tensor shape: (img_height, img_width, img_color_channels)
shp <- shape(4, 4, 3)
x <- op_arange(prod(shp)) |> op_reshape(shp)

# Convert to a batch of images by inserting a new axis
# New shape: (batch_size, img_height, img_width, img_color_channels)
x@r[newaxis, , , ] |> op_shape()
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## shape(1, 4, 4, 3)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[newaxis] |> op_shape()  # same as above
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## shape(1, 4, 4, 3)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[NULL] |> op_shape()     # same as above
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## shape(1, 4, 4, 3)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x <- x@r[newaxis]
# Extract color channels
x@r[, , , 1]          # red channel
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[[ 1.  4.  7. 10.]
##   [13. 16. 19. 22.]
##   [25. 28. 31. 34.]
##   [37. 40. 43. 46.]]], shape=(1, 4, 4), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[.., 1]            # red channel, same as above using .. shorthand
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[[ 1.  4.  7. 10.]
##   [13. 16. 19. 22.]
##   [25. 28. 31. 34.]
##   [37. 40. 43. 46.]]], shape=(1, 4, 4), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[.., 2]            # green channel
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[[ 2.  5.  8. 11.]
##   [14. 17. 20. 23.]
##   [26. 29. 32. 35.]
##   [38. 41. 44. 47.]]], shape=(1, 4, 4), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[.., 3]            # blue channel
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## tf.Tensor(
## [[[ 3.  6.  9. 12.]
##   [15. 18. 21. 24.]
##   [27. 30. 33. 36.]
##   [39. 42. 45. 48.]]], shape=(1, 4, 4), dtype=float32)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# .. expands to all unspecified axes.
op_shape(x@r[])
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## shape(1, 4, 4, 3)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{op_shape(x@r[..])
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## shape(1, 4, 4, 3)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{op_shape(x@r[1, ..])
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## shape(4, 4, 3)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{op_shape(x@r[1, .., 1, 1])
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## shape(4)

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{op_shape(x@r[1, 1, 1, .., 1])
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## shape()

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# op_subset<- uses the same semantics, but note that not all tensors
# support modification. E.g., TensorFlow constant tensors cannot be modified,
# while TensorFlow Variables can be.

(x <- tensorflow::tf$Variable(matrix(1, nrow = 2, ncol = 3)))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## <tf.Variable 'Variable:0' shape=(2, 3) dtype=float64, numpy=
## array([[1., 1., 1.],
##        [1., 1., 1.]])>

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{op_subset(x, 1) <- 9
x
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## <tf.Variable 'UnreadVariable' shape=(2, 3) dtype=float64, numpy=
## array([[9., 9., 9.],
##        [1., 1., 1.]])>

}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x@r[1,1] <- 33
x
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## <tf.Variable 'UnreadVariable' shape=(2, 3) dtype=float64, numpy=
## array([[33.,  9.,  9.],
##        [ 1.,  1.,  1.]])>

}\if{html}{\out{</div>}}
}

\seealso{
Other core ops: \cr
\code{\link{op_associative_scan}()} \cr
\code{\link{op_cast}()} \cr
\code{\link{op_cond}()} \cr
\code{\link{op_convert_to_numpy}()} \cr
\code{\link{op_convert_to_tensor}()} \cr
\code{\link{op_custom_gradient}()} \cr
\code{\link{op_dtype}()} \cr
\code{\link{op_fori_loop}()} \cr
\code{\link{op_is_tensor}()} \cr
\code{\link{op_map}()} \cr
\code{\link{op_rearrange}()} \cr
\code{\link{op_scan}()} \cr
\code{\link{op_scatter}()} \cr
\code{\link{op_scatter_update}()} \cr
\code{\link{op_searchsorted}()} \cr
\code{\link{op_shape}()} \cr
\code{\link{op_slice}()} \cr
\code{\link{op_slice_update}()} \cr
\code{\link{op_stop_gradient}()} \cr
\code{\link{op_switch}()} \cr
\code{\link{op_unstack}()} \cr
\code{\link{op_vectorized_map}()} \cr
\code{\link{op_while_loop}()} \cr

Other ops: \cr
\code{\link{op_abs}()} \cr
\code{\link{op_add}()} \cr
\code{\link{op_all}()} \cr
\code{\link{op_angle}()} \cr
\code{\link{op_any}()} \cr
\code{\link{op_append}()} \cr
\code{\link{op_arange}()} \cr
\code{\link{op_arccos}()} \cr
\code{\link{op_arccosh}()} \cr
\code{\link{op_arcsin}()} \cr
\code{\link{op_arcsinh}()} \cr
\code{\link{op_arctan}()} \cr
\code{\link{op_arctan2}()} \cr
\code{\link{op_arctanh}()} \cr
\code{\link{op_argmax}()} \cr
\code{\link{op_argmin}()} \cr
\code{\link{op_argpartition}()} \cr
\code{\link{op_argsort}()} \cr
\code{\link{op_array}()} \cr
\code{\link{op_associative_scan}()} \cr
\code{\link{op_average}()} \cr
\code{\link{op_average_pool}()} \cr
\code{\link{op_bartlett}()} \cr
\code{\link{op_batch_normalization}()} \cr
\code{\link{op_binary_crossentropy}()} \cr
\code{\link{op_bincount}()} \cr
\code{\link{op_bitwise_and}()} \cr
\code{\link{op_bitwise_invert}()} \cr
\code{\link{op_bitwise_left_shift}()} \cr
\code{\link{op_bitwise_not}()} \cr
\code{\link{op_bitwise_or}()} \cr
\code{\link{op_bitwise_right_shift}()} \cr
\code{\link{op_bitwise_xor}()} \cr
\code{\link{op_blackman}()} \cr
\code{\link{op_broadcast_to}()} \cr
\code{\link{op_cast}()} \cr
\code{\link{op_categorical_crossentropy}()} \cr
\code{\link{op_cbrt}()} \cr
\code{\link{op_ceil}()} \cr
\code{\link{op_celu}()} \cr
\code{\link{op_cholesky}()} \cr
\code{\link{op_clip}()} \cr
\code{\link{op_concatenate}()} \cr
\code{\link{op_cond}()} \cr
\code{\link{op_conj}()} \cr
\code{\link{op_conv}()} \cr
\code{\link{op_conv_transpose}()} \cr
\code{\link{op_convert_to_numpy}()} \cr
\code{\link{op_convert_to_tensor}()} \cr
\code{\link{op_copy}()} \cr
\code{\link{op_corrcoef}()} \cr
\code{\link{op_correlate}()} \cr
\code{\link{op_cos}()} \cr
\code{\link{op_cosh}()} \cr
\code{\link{op_count_nonzero}()} \cr
\code{\link{op_cross}()} \cr
\code{\link{op_ctc_decode}()} \cr
\code{\link{op_ctc_loss}()} \cr
\code{\link{op_cumprod}()} \cr
\code{\link{op_cumsum}()} \cr
\code{\link{op_custom_gradient}()} \cr
\code{\link{op_deg2rad}()} \cr
\code{\link{op_depthwise_conv}()} \cr
\code{\link{op_det}()} \cr
\code{\link{op_diag}()} \cr
\code{\link{op_diagflat}()} \cr
\code{\link{op_diagonal}()} \cr
\code{\link{op_diff}()} \cr
\code{\link{op_digitize}()} \cr
\code{\link{op_divide}()} \cr
\code{\link{op_divide_no_nan}()} \cr
\code{\link{op_dot}()} \cr
\code{\link{op_dot_product_attention}()} \cr
\code{\link{op_dtype}()} \cr
\code{\link{op_eig}()} \cr
\code{\link{op_eigh}()} \cr
\code{\link{op_einsum}()} \cr
\code{\link{op_elu}()} \cr
\code{\link{op_empty}()} \cr
\code{\link{op_equal}()} \cr
\code{\link{op_erf}()} \cr
\code{\link{op_erfinv}()} \cr
\code{\link{op_exp}()} \cr
\code{\link{op_exp2}()} \cr
\code{\link{op_expand_dims}()} \cr
\code{\link{op_expm1}()} \cr
\code{\link{op_extract_sequences}()} \cr
\code{\link{op_eye}()} \cr
\code{\link{op_fft}()} \cr
\code{\link{op_fft2}()} \cr
\code{\link{op_flip}()} \cr
\code{\link{op_floor}()} \cr
\code{\link{op_floor_divide}()} \cr
\code{\link{op_fori_loop}()} \cr
\code{\link{op_full}()} \cr
\code{\link{op_full_like}()} \cr
\code{\link{op_gelu}()} \cr
\code{\link{op_get_item}()} \cr
\code{\link{op_glu}()} \cr
\code{\link{op_greater}()} \cr
\code{\link{op_greater_equal}()} \cr
\code{\link{op_hamming}()} \cr
\code{\link{op_hanning}()} \cr
\code{\link{op_hard_shrink}()} \cr
\code{\link{op_hard_sigmoid}()} \cr
\code{\link{op_hard_silu}()} \cr
\code{\link{op_hard_tanh}()} \cr
\code{\link{op_heaviside}()} \cr
\code{\link{op_histogram}()} \cr
\code{\link{op_hstack}()} \cr
\code{\link{op_identity}()} \cr
\code{\link{op_ifft2}()} \cr
\code{\link{op_imag}()} \cr
\code{\link{op_image_affine_transform}()} \cr
\code{\link{op_image_crop}()} \cr
\code{\link{op_image_extract_patches}()} \cr
\code{\link{op_image_gaussian_blur}()} \cr
\code{\link{op_image_hsv_to_rgb}()} \cr
\code{\link{op_image_map_coordinates}()} \cr
\code{\link{op_image_pad}()} \cr
\code{\link{op_image_perspective_transform}()} \cr
\code{\link{op_image_resize}()} \cr
\code{\link{op_image_rgb_to_grayscale}()} \cr
\code{\link{op_image_rgb_to_hsv}()} \cr
\code{\link{op_in_top_k}()} \cr
\code{\link{op_inner}()} \cr
\code{\link{op_inv}()} \cr
\code{\link{op_irfft}()} \cr
\code{\link{op_is_tensor}()} \cr
\code{\link{op_isclose}()} \cr
\code{\link{op_isfinite}()} \cr
\code{\link{op_isinf}()} \cr
\code{\link{op_isnan}()} \cr
\code{\link{op_istft}()} \cr
\code{\link{op_kaiser}()} \cr
\code{\link{op_layer_normalization}()} \cr
\code{\link{op_leaky_relu}()} \cr
\code{\link{op_left_shift}()} \cr
\code{\link{op_less}()} \cr
\code{\link{op_less_equal}()} \cr
\code{\link{op_linspace}()} \cr
\code{\link{op_log}()} \cr
\code{\link{op_log10}()} \cr
\code{\link{op_log1p}()} \cr
\code{\link{op_log2}()} \cr
\code{\link{op_log_sigmoid}()} \cr
\code{\link{op_log_softmax}()} \cr
\code{\link{op_logaddexp}()} \cr
\code{\link{op_logdet}()} \cr
\code{\link{op_logical_and}()} \cr
\code{\link{op_logical_not}()} \cr
\code{\link{op_logical_or}()} \cr
\code{\link{op_logical_xor}()} \cr
\code{\link{op_logspace}()} \cr
\code{\link{op_logsumexp}()} \cr
\code{\link{op_lstsq}()} \cr
\code{\link{op_lu_factor}()} \cr
\code{\link{op_map}()} \cr
\code{\link{op_matmul}()} \cr
\code{\link{op_max}()} \cr
\code{\link{op_max_pool}()} \cr
\code{\link{op_maximum}()} \cr
\code{\link{op_mean}()} \cr
\code{\link{op_median}()} \cr
\code{\link{op_meshgrid}()} \cr
\code{\link{op_min}()} \cr
\code{\link{op_minimum}()} \cr
\code{\link{op_mod}()} \cr
\code{\link{op_moments}()} \cr
\code{\link{op_moveaxis}()} \cr
\code{\link{op_multi_hot}()} \cr
\code{\link{op_multiply}()} \cr
\code{\link{op_nan_to_num}()} \cr
\code{\link{op_ndim}()} \cr
\code{\link{op_negative}()} \cr
\code{\link{op_nonzero}()} \cr
\code{\link{op_norm}()} \cr
\code{\link{op_normalize}()} \cr
\code{\link{op_not_equal}()} \cr
\code{\link{op_one_hot}()} \cr
\code{\link{op_ones}()} \cr
\code{\link{op_ones_like}()} \cr
\code{\link{op_outer}()} \cr
\code{\link{op_pad}()} \cr
\code{\link{op_polar}()} \cr
\code{\link{op_power}()} \cr
\code{\link{op_prod}()} \cr
\code{\link{op_psnr}()} \cr
\code{\link{op_qr}()} \cr
\code{\link{op_quantile}()} \cr
\code{\link{op_ravel}()} \cr
\code{\link{op_real}()} \cr
\code{\link{op_rearrange}()} \cr
\code{\link{op_reciprocal}()} \cr
\code{\link{op_relu}()} \cr
\code{\link{op_relu6}()} \cr
\code{\link{op_repeat}()} \cr
\code{\link{op_reshape}()} \cr
\code{\link{op_rfft}()} \cr
\code{\link{op_right_shift}()} \cr
\code{\link{op_rms_normalization}()} \cr
\code{\link{op_roll}()} \cr
\code{\link{op_rot90}()} \cr
\code{\link{op_round}()} \cr
\code{\link{op_rsqrt}()} \cr
\code{\link{op_saturate_cast}()} \cr
\code{\link{op_scan}()} \cr
\code{\link{op_scatter}()} \cr
\code{\link{op_scatter_update}()} \cr
\code{\link{op_searchsorted}()} \cr
\code{\link{op_segment_max}()} \cr
\code{\link{op_segment_sum}()} \cr
\code{\link{op_select}()} \cr
\code{\link{op_selu}()} \cr
\code{\link{op_separable_conv}()} \cr
\code{\link{op_shape}()} \cr
\code{\link{op_sigmoid}()} \cr
\code{\link{op_sign}()} \cr
\code{\link{op_signbit}()} \cr
\code{\link{op_silu}()} \cr
\code{\link{op_sin}()} \cr
\code{\link{op_sinh}()} \cr
\code{\link{op_size}()} \cr
\code{\link{op_slice}()} \cr
\code{\link{op_slice_update}()} \cr
\code{\link{op_slogdet}()} \cr
\code{\link{op_soft_shrink}()} \cr
\code{\link{op_softmax}()} \cr
\code{\link{op_softplus}()} \cr
\code{\link{op_softsign}()} \cr
\code{\link{op_solve}()} \cr
\code{\link{op_solve_triangular}()} \cr
\code{\link{op_sort}()} \cr
\code{\link{op_sparse_categorical_crossentropy}()} \cr
\code{\link{op_sparse_plus}()} \cr
\code{\link{op_sparse_sigmoid}()} \cr
\code{\link{op_sparsemax}()} \cr
\code{\link{op_split}()} \cr
\code{\link{op_sqrt}()} \cr
\code{\link{op_square}()} \cr
\code{\link{op_squareplus}()} \cr
\code{\link{op_squeeze}()} \cr
\code{\link{op_stack}()} \cr
\code{\link{op_std}()} \cr
\code{\link{op_stft}()} \cr
\code{\link{op_stop_gradient}()} \cr
\code{\link{op_subtract}()} \cr
\code{\link{op_sum}()} \cr
\code{\link{op_svd}()} \cr
\code{\link{op_swapaxes}()} \cr
\code{\link{op_switch}()} \cr
\code{\link{op_take}()} \cr
\code{\link{op_take_along_axis}()} \cr
\code{\link{op_tan}()} \cr
\code{\link{op_tanh}()} \cr
\code{\link{op_tanh_shrink}()} \cr
\code{\link{op_tensordot}()} \cr
\code{\link{op_threshold}()} \cr
\code{\link{op_tile}()} \cr
\code{\link{op_top_k}()} \cr
\code{\link{op_trace}()} \cr
\code{\link{op_transpose}()} \cr
\code{\link{op_tri}()} \cr
\code{\link{op_tril}()} \cr
\code{\link{op_triu}()} \cr
\code{\link{op_trunc}()} \cr
\code{\link{op_unravel_index}()} \cr
\code{\link{op_unstack}()} \cr
\code{\link{op_var}()} \cr
\code{\link{op_vdot}()} \cr
\code{\link{op_vectorize}()} \cr
\code{\link{op_vectorized_map}()} \cr
\code{\link{op_view_as_complex}()} \cr
\code{\link{op_view_as_real}()} \cr
\code{\link{op_vstack}()} \cr
\code{\link{op_where}()} \cr
\code{\link{op_while_loop}()} \cr
\code{\link{op_zeros}()} \cr
\code{\link{op_zeros_like}()} \cr

Other core ops: \cr
\code{\link{op_associative_scan}()} \cr
\code{\link{op_cast}()} \cr
\code{\link{op_cond}()} \cr
\code{\link{op_convert_to_numpy}()} \cr
\code{\link{op_convert_to_tensor}()} \cr
\code{\link{op_custom_gradient}()} \cr
\code{\link{op_dtype}()} \cr
\code{\link{op_fori_loop}()} \cr
\code{\link{op_is_tensor}()} \cr
\code{\link{op_map}()} \cr
\code{\link{op_rearrange}()} \cr
\code{\link{op_scan}()} \cr
\code{\link{op_scatter}()} \cr
\code{\link{op_scatter_update}()} \cr
\code{\link{op_searchsorted}()} \cr
\code{\link{op_shape}()} \cr
\code{\link{op_slice}()} \cr
\code{\link{op_slice_update}()} \cr
\code{\link{op_stop_gradient}()} \cr
\code{\link{op_switch}()} \cr
\code{\link{op_unstack}()} \cr
\code{\link{op_vectorized_map}()} \cr
\code{\link{op_while_loop}()} \cr

Other ops: \cr
\code{\link{op_abs}()} \cr
\code{\link{op_add}()} \cr
\code{\link{op_all}()} \cr
\code{\link{op_angle}()} \cr
\code{\link{op_any}()} \cr
\code{\link{op_append}()} \cr
\code{\link{op_arange}()} \cr
\code{\link{op_arccos}()} \cr
\code{\link{op_arccosh}()} \cr
\code{\link{op_arcsin}()} \cr
\code{\link{op_arcsinh}()} \cr
\code{\link{op_arctan}()} \cr
\code{\link{op_arctan2}()} \cr
\code{\link{op_arctanh}()} \cr
\code{\link{op_argmax}()} \cr
\code{\link{op_argmin}()} \cr
\code{\link{op_argpartition}()} \cr
\code{\link{op_argsort}()} \cr
\code{\link{op_array}()} \cr
\code{\link{op_associative_scan}()} \cr
\code{\link{op_average}()} \cr
\code{\link{op_average_pool}()} \cr
\code{\link{op_bartlett}()} \cr
\code{\link{op_batch_normalization}()} \cr
\code{\link{op_binary_crossentropy}()} \cr
\code{\link{op_bincount}()} \cr
\code{\link{op_bitwise_and}()} \cr
\code{\link{op_bitwise_invert}()} \cr
\code{\link{op_bitwise_left_shift}()} \cr
\code{\link{op_bitwise_not}()} \cr
\code{\link{op_bitwise_or}()} \cr
\code{\link{op_bitwise_right_shift}()} \cr
\code{\link{op_bitwise_xor}()} \cr
\code{\link{op_blackman}()} \cr
\code{\link{op_broadcast_to}()} \cr
\code{\link{op_cast}()} \cr
\code{\link{op_categorical_crossentropy}()} \cr
\code{\link{op_cbrt}()} \cr
\code{\link{op_ceil}()} \cr
\code{\link{op_celu}()} \cr
\code{\link{op_cholesky}()} \cr
\code{\link{op_clip}()} \cr
\code{\link{op_concatenate}()} \cr
\code{\link{op_cond}()} \cr
\code{\link{op_conj}()} \cr
\code{\link{op_conv}()} \cr
\code{\link{op_conv_transpose}()} \cr
\code{\link{op_convert_to_numpy}()} \cr
\code{\link{op_convert_to_tensor}()} \cr
\code{\link{op_copy}()} \cr
\code{\link{op_corrcoef}()} \cr
\code{\link{op_correlate}()} \cr
\code{\link{op_cos}()} \cr
\code{\link{op_cosh}()} \cr
\code{\link{op_count_nonzero}()} \cr
\code{\link{op_cross}()} \cr
\code{\link{op_ctc_decode}()} \cr
\code{\link{op_ctc_loss}()} \cr
\code{\link{op_cumprod}()} \cr
\code{\link{op_cumsum}()} \cr
\code{\link{op_custom_gradient}()} \cr
\code{\link{op_deg2rad}()} \cr
\code{\link{op_depthwise_conv}()} \cr
\code{\link{op_det}()} \cr
\code{\link{op_diag}()} \cr
\code{\link{op_diagflat}()} \cr
\code{\link{op_diagonal}()} \cr
\code{\link{op_diff}()} \cr
\code{\link{op_digitize}()} \cr
\code{\link{op_divide}()} \cr
\code{\link{op_divide_no_nan}()} \cr
\code{\link{op_dot}()} \cr
\code{\link{op_dot_product_attention}()} \cr
\code{\link{op_dtype}()} \cr
\code{\link{op_eig}()} \cr
\code{\link{op_eigh}()} \cr
\code{\link{op_einsum}()} \cr
\code{\link{op_elu}()} \cr
\code{\link{op_empty}()} \cr
\code{\link{op_equal}()} \cr
\code{\link{op_erf}()} \cr
\code{\link{op_erfinv}()} \cr
\code{\link{op_exp}()} \cr
\code{\link{op_exp2}()} \cr
\code{\link{op_expand_dims}()} \cr
\code{\link{op_expm1}()} \cr
\code{\link{op_extract_sequences}()} \cr
\code{\link{op_eye}()} \cr
\code{\link{op_fft}()} \cr
\code{\link{op_fft2}()} \cr
\code{\link{op_flip}()} \cr
\code{\link{op_floor}()} \cr
\code{\link{op_floor_divide}()} \cr
\code{\link{op_fori_loop}()} \cr
\code{\link{op_full}()} \cr
\code{\link{op_full_like}()} \cr
\code{\link{op_gelu}()} \cr
\code{\link{op_get_item}()} \cr
\code{\link{op_glu}()} \cr
\code{\link{op_greater}()} \cr
\code{\link{op_greater_equal}()} \cr
\code{\link{op_hamming}()} \cr
\code{\link{op_hanning}()} \cr
\code{\link{op_hard_shrink}()} \cr
\code{\link{op_hard_sigmoid}()} \cr
\code{\link{op_hard_silu}()} \cr
\code{\link{op_hard_tanh}()} \cr
\code{\link{op_heaviside}()} \cr
\code{\link{op_histogram}()} \cr
\code{\link{op_hstack}()} \cr
\code{\link{op_identity}()} \cr
\code{\link{op_ifft2}()} \cr
\code{\link{op_imag}()} \cr
\code{\link{op_image_affine_transform}()} \cr
\code{\link{op_image_crop}()} \cr
\code{\link{op_image_extract_patches}()} \cr
\code{\link{op_image_gaussian_blur}()} \cr
\code{\link{op_image_hsv_to_rgb}()} \cr
\code{\link{op_image_map_coordinates}()} \cr
\code{\link{op_image_pad}()} \cr
\code{\link{op_image_perspective_transform}()} \cr
\code{\link{op_image_resize}()} \cr
\code{\link{op_image_rgb_to_grayscale}()} \cr
\code{\link{op_image_rgb_to_hsv}()} \cr
\code{\link{op_in_top_k}()} \cr
\code{\link{op_inner}()} \cr
\code{\link{op_inv}()} \cr
\code{\link{op_irfft}()} \cr
\code{\link{op_is_tensor}()} \cr
\code{\link{op_isclose}()} \cr
\code{\link{op_isfinite}()} \cr
\code{\link{op_isinf}()} \cr
\code{\link{op_isnan}()} \cr
\code{\link{op_istft}()} \cr
\code{\link{op_kaiser}()} \cr
\code{\link{op_layer_normalization}()} \cr
\code{\link{op_leaky_relu}()} \cr
\code{\link{op_left_shift}()} \cr
\code{\link{op_less}()} \cr
\code{\link{op_less_equal}()} \cr
\code{\link{op_linspace}()} \cr
\code{\link{op_log}()} \cr
\code{\link{op_log10}()} \cr
\code{\link{op_log1p}()} \cr
\code{\link{op_log2}()} \cr
\code{\link{op_log_sigmoid}()} \cr
\code{\link{op_log_softmax}()} \cr
\code{\link{op_logaddexp}()} \cr
\code{\link{op_logdet}()} \cr
\code{\link{op_logical_and}()} \cr
\code{\link{op_logical_not}()} \cr
\code{\link{op_logical_or}()} \cr
\code{\link{op_logical_xor}()} \cr
\code{\link{op_logspace}()} \cr
\code{\link{op_logsumexp}()} \cr
\code{\link{op_lstsq}()} \cr
\code{\link{op_lu_factor}()} \cr
\code{\link{op_map}()} \cr
\code{\link{op_matmul}()} \cr
\code{\link{op_max}()} \cr
\code{\link{op_max_pool}()} \cr
\code{\link{op_maximum}()} \cr
\code{\link{op_mean}()} \cr
\code{\link{op_median}()} \cr
\code{\link{op_meshgrid}()} \cr
\code{\link{op_min}()} \cr
\code{\link{op_minimum}()} \cr
\code{\link{op_mod}()} \cr
\code{\link{op_moments}()} \cr
\code{\link{op_moveaxis}()} \cr
\code{\link{op_multi_hot}()} \cr
\code{\link{op_multiply}()} \cr
\code{\link{op_nan_to_num}()} \cr
\code{\link{op_ndim}()} \cr
\code{\link{op_negative}()} \cr
\code{\link{op_nonzero}()} \cr
\code{\link{op_norm}()} \cr
\code{\link{op_normalize}()} \cr
\code{\link{op_not_equal}()} \cr
\code{\link{op_one_hot}()} \cr
\code{\link{op_ones}()} \cr
\code{\link{op_ones_like}()} \cr
\code{\link{op_outer}()} \cr
\code{\link{op_pad}()} \cr
\code{\link{op_polar}()} \cr
\code{\link{op_power}()} \cr
\code{\link{op_prod}()} \cr
\code{\link{op_psnr}()} \cr
\code{\link{op_qr}()} \cr
\code{\link{op_quantile}()} \cr
\code{\link{op_ravel}()} \cr
\code{\link{op_real}()} \cr
\code{\link{op_rearrange}()} \cr
\code{\link{op_reciprocal}()} \cr
\code{\link{op_relu}()} \cr
\code{\link{op_relu6}()} \cr
\code{\link{op_repeat}()} \cr
\code{\link{op_reshape}()} \cr
\code{\link{op_rfft}()} \cr
\code{\link{op_right_shift}()} \cr
\code{\link{op_rms_normalization}()} \cr
\code{\link{op_roll}()} \cr
\code{\link{op_rot90}()} \cr
\code{\link{op_round}()} \cr
\code{\link{op_rsqrt}()} \cr
\code{\link{op_saturate_cast}()} \cr
\code{\link{op_scan}()} \cr
\code{\link{op_scatter}()} \cr
\code{\link{op_scatter_update}()} \cr
\code{\link{op_searchsorted}()} \cr
\code{\link{op_segment_max}()} \cr
\code{\link{op_segment_sum}()} \cr
\code{\link{op_select}()} \cr
\code{\link{op_selu}()} \cr
\code{\link{op_separable_conv}()} \cr
\code{\link{op_shape}()} \cr
\code{\link{op_sigmoid}()} \cr
\code{\link{op_sign}()} \cr
\code{\link{op_signbit}()} \cr
\code{\link{op_silu}()} \cr
\code{\link{op_sin}()} \cr
\code{\link{op_sinh}()} \cr
\code{\link{op_size}()} \cr
\code{\link{op_slice}()} \cr
\code{\link{op_slice_update}()} \cr
\code{\link{op_slogdet}()} \cr
\code{\link{op_soft_shrink}()} \cr
\code{\link{op_softmax}()} \cr
\code{\link{op_softplus}()} \cr
\code{\link{op_softsign}()} \cr
\code{\link{op_solve}()} \cr
\code{\link{op_solve_triangular}()} \cr
\code{\link{op_sort}()} \cr
\code{\link{op_sparse_categorical_crossentropy}()} \cr
\code{\link{op_sparse_plus}()} \cr
\code{\link{op_sparse_sigmoid}()} \cr
\code{\link{op_sparsemax}()} \cr
\code{\link{op_split}()} \cr
\code{\link{op_sqrt}()} \cr
\code{\link{op_square}()} \cr
\code{\link{op_squareplus}()} \cr
\code{\link{op_squeeze}()} \cr
\code{\link{op_stack}()} \cr
\code{\link{op_std}()} \cr
\code{\link{op_stft}()} \cr
\code{\link{op_stop_gradient}()} \cr
\code{\link{op_subtract}()} \cr
\code{\link{op_sum}()} \cr
\code{\link{op_svd}()} \cr
\code{\link{op_swapaxes}()} \cr
\code{\link{op_switch}()} \cr
\code{\link{op_take}()} \cr
\code{\link{op_take_along_axis}()} \cr
\code{\link{op_tan}()} \cr
\code{\link{op_tanh}()} \cr
\code{\link{op_tanh_shrink}()} \cr
\code{\link{op_tensordot}()} \cr
\code{\link{op_threshold}()} \cr
\code{\link{op_tile}()} \cr
\code{\link{op_top_k}()} \cr
\code{\link{op_trace}()} \cr
\code{\link{op_transpose}()} \cr
\code{\link{op_tri}()} \cr
\code{\link{op_tril}()} \cr
\code{\link{op_triu}()} \cr
\code{\link{op_trunc}()} \cr
\code{\link{op_unravel_index}()} \cr
\code{\link{op_unstack}()} \cr
\code{\link{op_var}()} \cr
\code{\link{op_vdot}()} \cr
\code{\link{op_vectorize}()} \cr
\code{\link{op_vectorized_map}()} \cr
\code{\link{op_view_as_complex}()} \cr
\code{\link{op_view_as_real}()} \cr
\code{\link{op_vstack}()} \cr
\code{\link{op_where}()} \cr
\code{\link{op_while_loop}()} \cr
\code{\link{op_zeros}()} \cr
\code{\link{op_zeros_like}()} \cr

Other core ops: \cr
\code{\link{op_associative_scan}()} \cr
\code{\link{op_cast}()} \cr
\code{\link{op_cond}()} \cr
\code{\link{op_convert_to_numpy}()} \cr
\code{\link{op_convert_to_tensor}()} \cr
\code{\link{op_custom_gradient}()} \cr
\code{\link{op_dtype}()} \cr
\code{\link{op_fori_loop}()} \cr
\code{\link{op_is_tensor}()} \cr
\code{\link{op_map}()} \cr
\code{\link{op_rearrange}()} \cr
\code{\link{op_scan}()} \cr
\code{\link{op_scatter}()} \cr
\code{\link{op_scatter_update}()} \cr
\code{\link{op_searchsorted}()} \cr
\code{\link{op_shape}()} \cr
\code{\link{op_slice}()} \cr
\code{\link{op_slice_update}()} \cr
\code{\link{op_stop_gradient}()} \cr
\code{\link{op_switch}()} \cr
\code{\link{op_unstack}()} \cr
\code{\link{op_vectorized_map}()} \cr
\code{\link{op_while_loop}()} \cr

Other ops: \cr
\code{\link{op_abs}()} \cr
\code{\link{op_add}()} \cr
\code{\link{op_all}()} \cr
\code{\link{op_angle}()} \cr
\code{\link{op_any}()} \cr
\code{\link{op_append}()} \cr
\code{\link{op_arange}()} \cr
\code{\link{op_arccos}()} \cr
\code{\link{op_arccosh}()} \cr
\code{\link{op_arcsin}()} \cr
\code{\link{op_arcsinh}()} \cr
\code{\link{op_arctan}()} \cr
\code{\link{op_arctan2}()} \cr
\code{\link{op_arctanh}()} \cr
\code{\link{op_argmax}()} \cr
\code{\link{op_argmin}()} \cr
\code{\link{op_argpartition}()} \cr
\code{\link{op_argsort}()} \cr
\code{\link{op_array}()} \cr
\code{\link{op_associative_scan}()} \cr
\code{\link{op_average}()} \cr
\code{\link{op_average_pool}()} \cr
\code{\link{op_bartlett}()} \cr
\code{\link{op_batch_normalization}()} \cr
\code{\link{op_binary_crossentropy}()} \cr
\code{\link{op_bincount}()} \cr
\code{\link{op_bitwise_and}()} \cr
\code{\link{op_bitwise_invert}()} \cr
\code{\link{op_bitwise_left_shift}()} \cr
\code{\link{op_bitwise_not}()} \cr
\code{\link{op_bitwise_or}()} \cr
\code{\link{op_bitwise_right_shift}()} \cr
\code{\link{op_bitwise_xor}()} \cr
\code{\link{op_blackman}()} \cr
\code{\link{op_broadcast_to}()} \cr
\code{\link{op_cast}()} \cr
\code{\link{op_categorical_crossentropy}()} \cr
\code{\link{op_cbrt}()} \cr
\code{\link{op_ceil}()} \cr
\code{\link{op_celu}()} \cr
\code{\link{op_cholesky}()} \cr
\code{\link{op_clip}()} \cr
\code{\link{op_concatenate}()} \cr
\code{\link{op_cond}()} \cr
\code{\link{op_conj}()} \cr
\code{\link{op_conv}()} \cr
\code{\link{op_conv_transpose}()} \cr
\code{\link{op_convert_to_numpy}()} \cr
\code{\link{op_convert_to_tensor}()} \cr
\code{\link{op_copy}()} \cr
\code{\link{op_corrcoef}()} \cr
\code{\link{op_correlate}()} \cr
\code{\link{op_cos}()} \cr
\code{\link{op_cosh}()} \cr
\code{\link{op_count_nonzero}()} \cr
\code{\link{op_cross}()} \cr
\code{\link{op_ctc_decode}()} \cr
\code{\link{op_ctc_loss}()} \cr
\code{\link{op_cumprod}()} \cr
\code{\link{op_cumsum}()} \cr
\code{\link{op_custom_gradient}()} \cr
\code{\link{op_deg2rad}()} \cr
\code{\link{op_depthwise_conv}()} \cr
\code{\link{op_det}()} \cr
\code{\link{op_diag}()} \cr
\code{\link{op_diagflat}()} \cr
\code{\link{op_diagonal}()} \cr
\code{\link{op_diff}()} \cr
\code{\link{op_digitize}()} \cr
\code{\link{op_divide}()} \cr
\code{\link{op_divide_no_nan}()} \cr
\code{\link{op_dot}()} \cr
\code{\link{op_dot_product_attention}()} \cr
\code{\link{op_dtype}()} \cr
\code{\link{op_eig}()} \cr
\code{\link{op_eigh}()} \cr
\code{\link{op_einsum}()} \cr
\code{\link{op_elu}()} \cr
\code{\link{op_empty}()} \cr
\code{\link{op_equal}()} \cr
\code{\link{op_erf}()} \cr
\code{\link{op_erfinv}()} \cr
\code{\link{op_exp}()} \cr
\code{\link{op_exp2}()} \cr
\code{\link{op_expand_dims}()} \cr
\code{\link{op_expm1}()} \cr
\code{\link{op_extract_sequences}()} \cr
\code{\link{op_eye}()} \cr
\code{\link{op_fft}()} \cr
\code{\link{op_fft2}()} \cr
\code{\link{op_flip}()} \cr
\code{\link{op_floor}()} \cr
\code{\link{op_floor_divide}()} \cr
\code{\link{op_fori_loop}()} \cr
\code{\link{op_full}()} \cr
\code{\link{op_full_like}()} \cr
\code{\link{op_gelu}()} \cr
\code{\link{op_get_item}()} \cr
\code{\link{op_glu}()} \cr
\code{\link{op_greater}()} \cr
\code{\link{op_greater_equal}()} \cr
\code{\link{op_hamming}()} \cr
\code{\link{op_hanning}()} \cr
\code{\link{op_hard_shrink}()} \cr
\code{\link{op_hard_sigmoid}()} \cr
\code{\link{op_hard_silu}()} \cr
\code{\link{op_hard_tanh}()} \cr
\code{\link{op_heaviside}()} \cr
\code{\link{op_histogram}()} \cr
\code{\link{op_hstack}()} \cr
\code{\link{op_identity}()} \cr
\code{\link{op_ifft2}()} \cr
\code{\link{op_imag}()} \cr
\code{\link{op_image_affine_transform}()} \cr
\code{\link{op_image_crop}()} \cr
\code{\link{op_image_extract_patches}()} \cr
\code{\link{op_image_gaussian_blur}()} \cr
\code{\link{op_image_hsv_to_rgb}()} \cr
\code{\link{op_image_map_coordinates}()} \cr
\code{\link{op_image_pad}()} \cr
\code{\link{op_image_perspective_transform}()} \cr
\code{\link{op_image_resize}()} \cr
\code{\link{op_image_rgb_to_grayscale}()} \cr
\code{\link{op_image_rgb_to_hsv}()} \cr
\code{\link{op_in_top_k}()} \cr
\code{\link{op_inner}()} \cr
\code{\link{op_inv}()} \cr
\code{\link{op_irfft}()} \cr
\code{\link{op_is_tensor}()} \cr
\code{\link{op_isclose}()} \cr
\code{\link{op_isfinite}()} \cr
\code{\link{op_isinf}()} \cr
\code{\link{op_isnan}()} \cr
\code{\link{op_istft}()} \cr
\code{\link{op_kaiser}()} \cr
\code{\link{op_layer_normalization}()} \cr
\code{\link{op_leaky_relu}()} \cr
\code{\link{op_left_shift}()} \cr
\code{\link{op_less}()} \cr
\code{\link{op_less_equal}()} \cr
\code{\link{op_linspace}()} \cr
\code{\link{op_log}()} \cr
\code{\link{op_log10}()} \cr
\code{\link{op_log1p}()} \cr
\code{\link{op_log2}()} \cr
\code{\link{op_log_sigmoid}()} \cr
\code{\link{op_log_softmax}()} \cr
\code{\link{op_logaddexp}()} \cr
\code{\link{op_logdet}()} \cr
\code{\link{op_logical_and}()} \cr
\code{\link{op_logical_not}()} \cr
\code{\link{op_logical_or}()} \cr
\code{\link{op_logical_xor}()} \cr
\code{\link{op_logspace}()} \cr
\code{\link{op_logsumexp}()} \cr
\code{\link{op_lstsq}()} \cr
\code{\link{op_lu_factor}()} \cr
\code{\link{op_map}()} \cr
\code{\link{op_matmul}()} \cr
\code{\link{op_max}()} \cr
\code{\link{op_max_pool}()} \cr
\code{\link{op_maximum}()} \cr
\code{\link{op_mean}()} \cr
\code{\link{op_median}()} \cr
\code{\link{op_meshgrid}()} \cr
\code{\link{op_min}()} \cr
\code{\link{op_minimum}()} \cr
\code{\link{op_mod}()} \cr
\code{\link{op_moments}()} \cr
\code{\link{op_moveaxis}()} \cr
\code{\link{op_multi_hot}()} \cr
\code{\link{op_multiply}()} \cr
\code{\link{op_nan_to_num}()} \cr
\code{\link{op_ndim}()} \cr
\code{\link{op_negative}()} \cr
\code{\link{op_nonzero}()} \cr
\code{\link{op_norm}()} \cr
\code{\link{op_normalize}()} \cr
\code{\link{op_not_equal}()} \cr
\code{\link{op_one_hot}()} \cr
\code{\link{op_ones}()} \cr
\code{\link{op_ones_like}()} \cr
\code{\link{op_outer}()} \cr
\code{\link{op_pad}()} \cr
\code{\link{op_polar}()} \cr
\code{\link{op_power}()} \cr
\code{\link{op_prod}()} \cr
\code{\link{op_psnr}()} \cr
\code{\link{op_qr}()} \cr
\code{\link{op_quantile}()} \cr
\code{\link{op_ravel}()} \cr
\code{\link{op_real}()} \cr
\code{\link{op_rearrange}()} \cr
\code{\link{op_reciprocal}()} \cr
\code{\link{op_relu}()} \cr
\code{\link{op_relu6}()} \cr
\code{\link{op_repeat}()} \cr
\code{\link{op_reshape}()} \cr
\code{\link{op_rfft}()} \cr
\code{\link{op_right_shift}()} \cr
\code{\link{op_rms_normalization}()} \cr
\code{\link{op_roll}()} \cr
\code{\link{op_rot90}()} \cr
\code{\link{op_round}()} \cr
\code{\link{op_rsqrt}()} \cr
\code{\link{op_saturate_cast}()} \cr
\code{\link{op_scan}()} \cr
\code{\link{op_scatter}()} \cr
\code{\link{op_scatter_update}()} \cr
\code{\link{op_searchsorted}()} \cr
\code{\link{op_segment_max}()} \cr
\code{\link{op_segment_sum}()} \cr
\code{\link{op_select}()} \cr
\code{\link{op_selu}()} \cr
\code{\link{op_separable_conv}()} \cr
\code{\link{op_shape}()} \cr
\code{\link{op_sigmoid}()} \cr
\code{\link{op_sign}()} \cr
\code{\link{op_signbit}()} \cr
\code{\link{op_silu}()} \cr
\code{\link{op_sin}()} \cr
\code{\link{op_sinh}()} \cr
\code{\link{op_size}()} \cr
\code{\link{op_slice}()} \cr
\code{\link{op_slice_update}()} \cr
\code{\link{op_slogdet}()} \cr
\code{\link{op_soft_shrink}()} \cr
\code{\link{op_softmax}()} \cr
\code{\link{op_softplus}()} \cr
\code{\link{op_softsign}()} \cr
\code{\link{op_solve}()} \cr
\code{\link{op_solve_triangular}()} \cr
\code{\link{op_sort}()} \cr
\code{\link{op_sparse_categorical_crossentropy}()} \cr
\code{\link{op_sparse_plus}()} \cr
\code{\link{op_sparse_sigmoid}()} \cr
\code{\link{op_sparsemax}()} \cr
\code{\link{op_split}()} \cr
\code{\link{op_sqrt}()} \cr
\code{\link{op_square}()} \cr
\code{\link{op_squareplus}()} \cr
\code{\link{op_squeeze}()} \cr
\code{\link{op_stack}()} \cr
\code{\link{op_std}()} \cr
\code{\link{op_stft}()} \cr
\code{\link{op_stop_gradient}()} \cr
\code{\link{op_subtract}()} \cr
\code{\link{op_sum}()} \cr
\code{\link{op_svd}()} \cr
\code{\link{op_swapaxes}()} \cr
\code{\link{op_switch}()} \cr
\code{\link{op_take}()} \cr
\code{\link{op_take_along_axis}()} \cr
\code{\link{op_tan}()} \cr
\code{\link{op_tanh}()} \cr
\code{\link{op_tanh_shrink}()} \cr
\code{\link{op_tensordot}()} \cr
\code{\link{op_threshold}()} \cr
\code{\link{op_tile}()} \cr
\code{\link{op_top_k}()} \cr
\code{\link{op_trace}()} \cr
\code{\link{op_transpose}()} \cr
\code{\link{op_tri}()} \cr
\code{\link{op_tril}()} \cr
\code{\link{op_triu}()} \cr
\code{\link{op_trunc}()} \cr
\code{\link{op_unravel_index}()} \cr
\code{\link{op_unstack}()} \cr
\code{\link{op_var}()} \cr
\code{\link{op_vdot}()} \cr
\code{\link{op_vectorize}()} \cr
\code{\link{op_vectorized_map}()} \cr
\code{\link{op_view_as_complex}()} \cr
\code{\link{op_view_as_real}()} \cr
\code{\link{op_vstack}()} \cr
\code{\link{op_where}()} \cr
\code{\link{op_while_loop}()} \cr
\code{\link{op_zeros}()} \cr
\code{\link{op_zeros_like}()} \cr
}
\concept{core ops}
\concept{ops}
