% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mpa.R
\name{mpa}
\alias{mpa}
\title{Marine Predators Algorithm (MPA)}
\usage{
mpa(SearchAgents_no, Max_iter, lb, ub, dim, fobj, logFile = NULL, ...)
}
\arguments{
\item{SearchAgents_no}{Number of search agents (predators). At least 2.
Typical values range from 20 to 50.}

\item{Max_iter}{Maximum number of iterations. At least 3.
Typical values range from 100 to 500 depending on problem complexity.}

\item{lb}{Lower bounds for each dimension. Can be a single value (applied to
all dimensions) or a vector of length \code{dim}.}

\item{ub}{Upper bounds for each dimension. Can be a single value (applied to
all dimensions) or a vector of length \code{dim}.}

\item{dim}{Number of dimensions (decision variables) in the optimization problem.}

\item{fobj}{Objective function to minimize. Must accept a numeric vector of
length \code{dim} and return a single numeric value.}

\item{logFile}{A path for logging (text file). Defaulted to \code{NULL}. If
\code{NULL}, no logging is performed.}

\item{...}{Additional arguments. Currently supports \code{prefix} for log
message prefixing. See Details.}
}
\value{
An object of class \code{mpa_result}, which is a list containing:
  \item{Top_predator_fit}{Best fitness value found (numeric scalar)}
  \item{Top_predator_pos}{Best position found (numeric vector of length \code{dim})}
  \item{Convergence_curve}{Convergence curve over iterations (numeric vector
    of length \code{Max_iter})}
}
\description{
Implementation of the Marine Predators Algorithm (MPA) in R. MPA
  is a nature-inspired optimization algorithm that follows the rules
  governing optimal foraging strategy and encounter rate policy between
  predator and prey in marine ecosystems.
}
\details{
This is a \strong{minimization} algorithm. To maximize a function, negate its
output (e.g., use \code{function(x) -f(x)} instead of \code{f}).
}
\section{Algorithm Phases}{

The MPA algorithm operates in three distinct phases based on the iteration count:

\describe{
  \item{Phase 1 (iterations 0 to Max_iter/3)}{
    \strong{High velocity ratio} - The prey moves faster than the predator.
    Exploration is emphasized using Brownian motion. This phase promotes
    global search across the solution space.
  }
  \item{Phase 2 (iterations Max_iter/3 to 2*Max_iter/3)}{
    \strong{Unit velocity ratio} - Predator and prey move at similar speeds.
    The population is split: half uses Brownian motion (exploitation),
    half uses Levy flight (exploration). This balances exploration and exploitation.
  }
  \item{Phase 3 (iterations 2*Max_iter/3 to Max_iter)}{
    \strong{Low velocity ratio} - The predator moves faster than the prey.
    Levy flight is used for all agents, focusing on exploitation around
    the best solution found.
  }
}
}

\section{Internal Parameters}{

The algorithm uses two internal parameters that are not exposed:
\describe{
  \item{FADs}{Fish Aggregating Devices effect parameter, set to 0.2.
    Controls the probability of applying the FADs effect which helps
    escape local optima.}
  \item{P}{Prey movement probability, set to 0.5. Controls the step size
    scaling factor during position updates.}
}
}

\section{Memory Mechanism}{

MPA implements a memory mechanism (Marine Memory) that preserves the best
positions found by each agent. If a new position has worse fitness than the
previous one, the agent reverts to its previous position.
}

\section{Additional Arguments}{

The \code{...} parameter currently accepts:
\describe{
  \item{prefix}{A character string to prefix log messages. Defaults to
    empty string if not provided.}
}
}

\examples{
# Basic usage with the Sphere function (F01)
result <- mpa(
  SearchAgents_no = 25, Max_iter = 100, lb = -100, ub = 100,
  dim = 30, fobj = F01
)
print(result)

# Using different bounds per dimension
result2 <- mpa(
  SearchAgents_no = 20, Max_iter = 50,
  lb = c(-5, 0), ub = c(10, 15),
  dim = 2, fobj = F17
)

# Maximization example (negate the objective function)
maximize_f <- function(x) -sum(x^2)
result3 <- mpa(
  SearchAgents_no = 20, Max_iter = 50,
  lb = -10, ub = 10, dim = 5,
  fobj = function(x) -maximize_f(x)
)
# The actual maximum value is -result3$Top_predator_fit

}
\references{
Faramarzi, A., Heidarinejad, M., Mirjalili, S., & Gandomi, A. H.
(2020). Marine Predators Algorithm: A Nature-inspired Metaheuristic. Expert
Systems with Applications, 152, 113377.
\doi{10.1016/j.eswa.2020.113377}
}
\seealso{
[get_function_details()] for benchmark functions,
[levy()] for Levy flight implementation,
[initialize_population()] for population initialization.
}
