% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_icd_codes.R
\name{get_icd_codes}
\alias{get_icd_codes}
\title{Get ICD Codes}
\usage{
get_icd_codes(with.descriptions = FALSE, with.hierarchy = FALSE)
}
\arguments{
\item{with.descriptions}{Logical scalar, if \code{TRUE} include the
description of the codes.}

\item{with.hierarchy}{Logical scalar, if \code{TRUE} include the ICD
hierarchy.}
}
\value{
a \code{data.frame}

The default return has the following columns:
\itemize{
\item \code{icdv}: Integer vector indicating if the code is from ICD-9 or ICD-10
\item \code{dx}: Integer vector.  1 if the code is a diagnostic, (ICD-9-CM, ICD-10-CM, WHO, CDC Mortality), or 0 if the code is procedural (ICD-9-PCS, ICD-10-PCS)
\item \code{full_code}: Character vector with the ICD code and any relevant decimal point
\item \code{code}: Character vector with the compact ICD code omitting any relevant decimal point
\item \code{src}: Character vector reporting the source of the information.  See Details.
\item \code{known_start}: Integer vector reporting the first known year of use.  See Details.
\item \code{known_end}: Integer vector reporting the last known year of use.  See Details.
\item \code{assignable_start}: Integer vector reporting the first known year the code was assignable.  See Details.
\item \code{assignable_end}: Integer vector reporting the last known year the code was assignable.  See Details.
}

When \code{with.descriptions = TRUE} there are the following additional columns:
\itemize{
\item \code{desc}: Character vector of descriptions.  For \code{cms} codes descriptions from CMS are used preferentially over CDC.
\item \code{desc_start}: Integer vector of the first year the description was used.
\item \code{desc_end}: Integer vector of the last year the description was used.
}

When \code{with.hierarchy = TRUE} there are the following additional columns:
\itemize{
\item \code{chapter}
\item \code{subchapter}
\item \code{category}
\item \code{subcategory}
\item \code{subclassification}
\item \code{subsubclassification}
\item \code{extension}
}
}
\description{
Retrieve a copy of the internal lookup table for all known ICD codes.
}
\details{
\subsection{Sources}{

There are three sources of ICD codes.
\itemize{
\item \code{cms}: Codes from the ICD-9-CM, ICD-9-PCS, ICD-10-CM, and ICD-10-PCS standards.
\item \code{who}: Codes from World Health Organization.
\item \code{cdc}: Codes from CDC Mortality coding standard.
}
}

\subsection{Fiscal and Calendar Years}{

When reporting years there is a mix of fiscal and calendar years.

Fiscal years are the United States Federal Government fiscal years, running
from October 1 to September 30. For example, fiscal year 2013 started October
1 2012 and ended on September 30 2013.

Calendar years run January 1 to December 31.

Within the ICD data there are columns
\code{known_start}, \code{known_end}, \code{assignable_start}, \code{assignable_end},
\code{desc_start} and \code{desc_end}.  For ICD codes with \code{src == "cms"}, these are
fiscal years.  For codes with \code{src == "cdc"} or \code{src == "who"} these are
calendar years.

\code{known_start} is the first fiscal or calendar year (depending on source) that
the medicalcoder package as definitive source data for.  ICD-9-CM started in
the United States in fiscal year 1980.  The CDC extracts included in
medicalcoder span fiscal years 1997--2012; the CMS ICD-9-CM/PCS extracts
start in fiscal year 2006 and run through fiscal year 2015.  As such 1997 is
the earliest "known start" for ICD-9 within medicalcoder.

\code{known_end} is the last fiscal or calendar year (depending on source)
for which we have definitive source data for.  For ICD-9-CM and ICD-9-PCS,
CMS provides data through fiscal year 2015, while the CDC extracts stop at
fiscal year 2012.  For ICD-10-CM and ICD-10-PCS, which are active, it is just
the last year of known data.  ICD-10 from the WHO ends in 2019.
}

\subsection{Header and Assignable Codes}{

"Assignable" indicates that the code is the most granular for the source.
Ideally codes are reported with the greatest level of detail but that is not
always the case.  Also, the greatest level of detail can differ between
sources.
Example: C86 is a header code for \code{cms} and \code{who} because codes C86.0, C86.1,
C86.2, C86.3, C86.4, C86.5, and C86.6 all exist in both standards.  No code
with a fifth digit exists in the \code{who} so all these four digit codes are
'assignable.'  In the \code{cms} standard, C86.0 was assignable through fiscal
year 2024. In fiscal year 2025 codes C86.00 and C86.01 were added making
C86.0 a header code and C86.00 and C86.01 assignable codes.
}
}
\examples{
icd_codes <- get_icd_codes()
str(icd_codes)

# Explore the change in the assignable year for C86 code between CMS and
# WHO
subset(get_icd_codes(), grepl("^C86$", full_code))
subset(get_icd_codes(), grepl("^C86\\\\.\\\\d$", full_code))
subset(get_icd_codes(), grepl("^C86\\\\.0(\\\\d|$)", full_code))

is_icd("C86", headerok = FALSE) # FALSE
is_icd("C86", headerok = TRUE)  # TRUE
is_icd("C86", headerok = TRUE, src = "cdc") # Not a CDC mortality code

lookup_icd_codes("^C86\\\\.0\\\\d*", regex = TRUE)
}
\seealso{
\code{\link[=is_icd]{is_icd()}}, \code{\link[=lookup_icd_codes]{lookup_icd_codes()}},
\code{vignette(topic = "icd", package = "medicalcoder")}
}
\concept{icd tools}
