#' MetaDeconfound
#'
#' MetaDeconfound checks all feature <-> covariate combinations for
#' counfounding effects of covariates on feature <-> effect correlation
#'
#' @param featureMat a data frame with row(sample ID)
#' and column(feature such as metabolite or microbial OTU )
#' names, listing features for all samples
#' @param metaMat a data frame with row(sample ID) and
#' column(meta data such as age,BMI and all possible confounders)
#' names listing metadata for all samples. The first column should be a binary
#' status/outcome variable (e.g. case=1 and control=0).
#' All other binary variables need to be in 0/1 syntax as well.
#' @param nnodes number of nodes/cores to be used for parallel processing
#' @param adjustMethod multiple testing p-value correction using one of the
#' methods of \link[stats]{p.adjust.methods}
#' @param adjustLevel 1 = correction for number of features (default);
#' 2 = correction for number of features AND number of metavariables;
#' 3 = correction for number of features AND number of variables in mediationMat
#' (default when mediationMat is supplied)
#' @param robustCutoff minimal number of sample size for each covariate
#' in order to have sufficient power for association testing
#' @param QCutoff significance cutoff for q-value, DEFAULT = 0.1
#' @param DCutoff effect size cutoff
#' (either Cliff's Delta or Spearman's rho), DEFAULT = 0
#' @param PHS_cutoff PostHoc Significance cutoff
#' @param logfile name of optional logging file.
#' @param logLevel logging verbosity, possible levels:
#' TRACE, DEBUG, INFO, WARN, ERROR, FATAL, OFF,  DEFAULT = INFO
#' @param startStop vector of optional strings controlling which
#' parts of the pipeline should be executed.
#' ("naiveStop": only naive associations will be computed, no confounder analysis is done)
#' @param QValues optional data.frame containing pre-computed multiple-testing corrected p-values for naive associations
#' @param DValues optional data.frame containing pre-computed effect sizes for naive associations
#' @param minQValues pessimistic qvalues, can be generated by
#' \link[metadeconfoundR]{ImportLongPrior}.
#' This dataframe of QValues is used to incorporate prior knowledge of
#' potential associations between individual features and metadata by supplying
#' QValues < QCutoff for these associations. All significant associations thus
#' reported will be treated as potentially confounding influences.
#' @param deconfT vector of metavariable names *always* to be included as potential confounder
#' @param deconfF vector of metavariable names *never* to be included as potential confounder
#' @param doConfs optional parameter for additional computation of confidence
#' interval of linear models in the deconfounding step. level:
#' (0 = no , 1 = logging, 2 = strict (default))
#' @param doRanks optional vector of metavariable names, that should be rank
#' transformed when building linear models in the doconfounding step
#' @param randomVar optional vector of metavariable names to be treated as
#' random effect variables. These variables will not be tested for naive
#' associations and will not be included as potential confounders,
#' but will be added as random effects "+ (1|variable)" into any models being built.
#' Any associations reducible to the supplied random effect(s) will be labeled
#'  as "NS". Note: Ps, Qs, Ds are computed independently and thereby not changed
#'  through inclusion of random effects.
#' @param fixedVar optional vector of metavariable names to be treated as
#' fixed effect variables. These variabels will not be tested for naive
#' associations and will not be included as potential confounders,
#' but will be added as fixed effects "+ variable" into any models being built.
#' Any associations reducible to the supplied fixed effect(s) will be labeled
#' as "NS". Note: Ps, Qs, Ds are computed independently and thereby not changed
#' through inclusion of fixed effects.
#' @param robustCutoffRho optional robustness cutoff for continuous variables
#' @param typeCategorical optional character vector of metavariable names to
#' always be treated as categorical
#' @param typeContinuous optional character vector of metavariable names to
#' always be treated as continuous
#' @param logistic optional logical parameter; DEFAULT = FALSE;
#' Set TRUE to treat supplied features as binary instead of continuous
#' @param rawCounts optional logical parameter; DEFAULT = FALSE;
#' Set TRUE to treat supplied features as not normalized/rarefied counts;
#' metadeconfoundR will compute total read count per sample and include this
#' information in the modelling steps. WARNING: naive associations in
#' first part of metadeconfoundR are computed on TSS-transformed version of input data.
#' @param returnLong DEFAULT = FALSE; Set TRUE to get output in one long
#' format data.frame instead of list of four wide format data.frames
#' @param collectMods DEFAULT = FALSE; Set TRUE to collect all model objects
#' generated by Metadeconfound and return them in a nested list alongside the
#' standard Ps/Qs/Ds/status output.
#' @param noConfConfs DEFAULT = TRUE; Set FALSE if confounders that are
#' themselves confounded by other variables should be kept in the list of
#' confounders in the status label of affected associations.
#' @param mediationMat optional additional data frame for mediation analysis
#' with row(sample ID) and column(feature such as metabolite or microbial
#' OTU ) names. Features supplied here will be tested for associations in
#' featureMat, and confounder controlled for variables in metaMat.
#' @param ... for additional arguments used internally (development/debugging)
#' @return list with elements (or data.frame with columns, when returnLong = TRUE) Ds = effectsize,
#' Ps = uncorrected p-value for naive association,
#' Qs = multiple testing corrected p-value/fdr,
#' and status = confounding status for all
#' feature <=> covariate combinations with following categories:
#' (NS = not significant, OK_sd = strictly deconfounded, OK_nc = no covariates,
#' OK_d = doubtful, AD = ambiguously deconfounded, C: followed by comma
#' separated covariate names = confounded by listed covariates)\cr
#'
#' Can be plotted using \link[metadeconfoundR]{BuildHeatmap}.
#' @details for more details and explanations please see the vignette.
#' @examples
#'data(reduced_feature)
#'data(metaMatMetformin)
#'\donttest{
#'example_output <- MetaDeconfound(featureMat = reduced_feature,
#'                                   metaMat = metaMatMetformin,
#'                                   logLevel = "ERROR")
#'}
#'
#' @import logger
#' @importFrom reshape2 melt
#' @importFrom methods is
#' @importFrom stats na.omit
#' @export
#'


MetaDeconfound <- function(featureMat,
                           metaMat,
                           nnodes = 1,
                           adjustMethod = "fdr",
                           adjustLevel = NULL,
                           robustCutoff = 5,
                           QCutoff = 0.1,
                           DCutoff = 0,
                           PHS_cutoff = 0.05,
                           logfile = NULL,
                           logLevel = "INFO",# new TB20240602
                           startStop = NA,
                           QValues = NA,
                           DValues = NA,
                           minQValues = NULL,
                           deconfT = NULL,
                           deconfF = NULL,
                           doConfs = 2,
                           doRanks = NA,
                           randomVar = NA,
                           fixedVar = NA,# new TB20230727
                           robustCutoffRho = NULL,# new SKF20200221
                           typeCategorical = NULL,# new SKF20200221
                           typeContinuous = NULL,# new SKF20200221
                           logistic = FALSE,# new SKF20201017
                           rawCounts = FALSE,# new TB20220202
                           returnLong = FALSE,# new TB20210409
                           collectMods = FALSE,# new TB20220208
                           mediationMat = NULL,# new TB20241121
                           noConfConfs= TRUE, # new TB20250827
                           ...) {

  logger::log_threshold(level = logLevel, namespace = "metadeconfoundR")
  logger::log_formatter(logger::formatter_paste, namespace = "metadeconfoundR")

  if (is.null(logfile)) {
    #logger::log_appender(logger::appender_console)
    logger::log_appender(logger::appender_stdout, namespace = "metadeconfoundR")
  } else {
    logger::log_appender(logger::appender_file(file = logfile), namespace = "metadeconfoundR")
  }

  ###
  ###
  ### logging start and initial sanity checks on input parameters
  ###
  ###

  logger::log_info(namespace = "metadeconfoundR", '\n\t###\n\t###\n\tDeconfounding run started')
  if ("naiveStop" %in% startStop) {
    logger::log_warn(namespace = "metadeconfoundR", 'Detected "naiveStop" in "startStop" paramter. Will only return naive associations.')
  }

  if (missing(metaMat)) {
    logger::log_error(namespace = "metadeconfoundR", 'Necessary argument "metaMat" missing.')
    stop('Error - Necessary argument "metaMat" missing.')
  }
  if (missing(featureMat)) {
    logger::log_error(namespace = "metadeconfoundR", 'Necessary argument "featureMat" missing.')
    stop('Error - Necessary argument "featureMat" missing.')
  }

  if (is(featureMat, "tbl") | is(metaMat, "tbl") | is(mediationMat, "tbl")) {
    logger::log_warn(namespace = "metadeconfoundR", "Tibbles detected in input data frames. This might lead to unexpected behaviors. Please convert to base data.frame class!")
  }

  if (is.null(logfile) && (doConfs > 0)) {
    logger::log_debug(namespace = "metadeconfoundR", '"doConfs" parameter is set to > 0 but "logfile" is not specified. Can not log warnings for confidence intervals spanning 0.')
  }

  if (nrow(metaMat) != nrow(featureMat)) {
    logger::log_error(namespace = "metadeconfoundR", "featureMat and metaMat don't have same number of rows.")
    stop("featureMat and metaMat don't have same number of rows.")
  }
  if (any(order(rownames(metaMat)) != order(rownames(featureMat)))) {
    logger::log_error(namespace = "metadeconfoundR", "rownames of featureMat and metaMat don't have same order.")
    stop("Rownames of featureMat and metaMat don't have same order.
         (order(rownames(metaMat)) != order(rownames(featureMat)))")
  }

  if (!is.null(mediationMat)) {
    if (nrow(metaMat) != nrow(mediationMat)) {
      logger::log_error(namespace = "metadeconfoundR", "mediationMat and metaMat don't have same number of rows.")
      stop("mediationMat and metaMat don't have same number of rows.")
    }
    if (any(order(rownames(metaMat)) != order(rownames(mediationMat)))) {
      logger::log_error(namespace = "metadeconfoundR", "rownames of mediationMat and metaMat don't have same order.")
      stop("Rownames of mediationMat and metaMat don't have same order.
         (order(rownames(metaMat)) != order(rownames(mediationMat)))")
    }
    # merge meta and mediation DFs and set all elements of mediationMat to deconfF
    metaMat <- cbind(metaMat, mediationMat)
    deconfF <- unique(c(deconfF, colnames(mediationMat)))
  }

  # check proper naming of rows and columns
  faultyColnamesMeta <- colnames(metaMat)[which(colnames(metaMat) != make.names(colnames(metaMat)))]
  faultyColnamesFeat <- colnames(featureMat)[which(colnames(featureMat) != make.names(colnames(featureMat)))]
  faultyRownamesMeta <-rownames(metaMat)[which(rownames(metaMat) != make.names(rownames(metaMat)))]
  if (length(c(faultyColnamesMeta, faultyColnamesFeat, faultyRownamesMeta)) > 0) {
    logger::log_warn(namespace = "metadeconfoundR",
    paste0(
        "\n\tUnallowed characters detected in rownames and/or colnames of featureMat and/or metaMat!\n",
        "\tmetadeconfoundR will try to remove these characters using the make.names() function."
        ))
    colnames(metaMat) <- make.names(colnames(metaMat), unique = T)
    colnames(featureMat) <- make.names(colnames(featureMat), unique = T)
    rownames(metaMat) <- make.names(rownames(metaMat), unique = T)
    rownames(featureMat) <- make.names(rownames(featureMat), unique = T)
  }

  allNumColls <- sapply(featureMat, FUN = function(x) is(x, "numeric"))
  if (!all(allNumColls)) {
    logger::log_error(namespace = "metadeconfoundR", "Non-numeric columns detected in featureMat.")
    stop("Non-numeric columns detected in featureMat.")
  }

  if (!is.null(deconfT) | !is.null(deconfF)) {

    if ((sum(deconfT %in% colnames(metaMat)) < length(deconfT)) |
        (sum(deconfF %in% colnames(metaMat)) < length(deconfF))) {
      logger::log_error(namespace = "metadeconfoundR", "Elements of deconfT/deconfF are not present in colnames of metaMat.")
      logger::log_info(namespace = "metadeconfoundR", "Check identical spelling of variable
                      names in deconfT/deconfF and metaMat")
      stop("Elements of deconfT/deconfF are not present in colnames of metaMat.")
    } else if (sum(deconfT %in% deconfF) > 0) {
      logger::log_error(namespace = "metadeconfoundR", "Some elements of deconfT and deconfF seem to be identical.")
      stop("Some elements of deconfT and deconfF seem to be identical.")
    }
  }



  if (is.null(QValues) || is.null(DValues)) {
    logger::log_error(namespace = "metadeconfoundR", "QValues and/or DValues argument is supplied but seems to be empty (NULL).")
    stop("QValues and/or DValues argument is supplied but seems to be empty (NULL).")
  }


  if (rawCounts == TRUE) {
    if (logistic == TRUE) {
      logger::log_error(namespace = "metadeconfoundR", "rawCounts and logistic can not be both set to TRUE!")
      stop("rawCounts and logistic can not be both set to TRUE!")
    }
    logger::log_warn(namespace = "metadeconfoundR", 'Raw count mode is enabled!For computation of naive associations only, raw counts are being normalized by dividing each sample by its total count! ')
  }

  if (is(randomVar, "list")) {
    logger::log_error(namespace = "metadeconfoundR", "randomVar does not need to be supplied as list anymore, please change to new syntax.")
    stop("randomVar does not need to be supplied as list anymore, please change to new syntax.")
  }

  # make sure only continuous OR binary features are applied for logistic = F/T.
  if (!logistic) {
    if (any(apply(
          featureMat,
          2,
          FUN = function (x)
            VarType(na.exclude(x), "varName", NULL, NULL)
          ) == "binary")) {
      logger::log_warn(namespace = "metadeconfoundR", 'There appear to be binary features in featurMat. Remove from featureMat, or set logistic = T.')
    }
  }
  else {
    if (any(apply(
          featureMat,
          2,
          FUN = function (x)
            VarType(na.exclude(x), "varName", NULL, NULL)
          ) == "continuous")) {
      logger::log_warn(namespace = "metadeconfoundR", 'There appear to be continuous features in featurMat. Remove from featureMat, or set logistic = F.')
    }
  }

  .MetaDeconfound(
    featureMat = featureMat,
    metaMat = metaMat,
    nnodes = nnodes,
    adjustMethod = adjustMethod,
    adjustLevel = adjustLevel,
    robustCutoff = robustCutoff,
    QCutoff = QCutoff,
    DCutoff = DCutoff,
    PHS_cutoff = PHS_cutoff,
    logfile = logfile,
    logLevel = logfile,
    startStop = startStop,
    QValues = QValues,
    DValues = DValues,
    minQValues = minQValues,
    deconfT = deconfT,
    deconfF = deconfF,
    doConfs = doConfs,
    doRanks = doRanks,
    randomVar = randomVar,
    fixedVar = fixedVar,
    robustCutoffRho = robustCutoffRho,
    typeCategorical = typeCategorical,
    typeContinuous = typeContinuous,
    logistic = logistic,
    rawCounts = rawCounts,
    returnLong = returnLong,
    collectMods = collectMods,
    mediationMat = mediationMat,
    noConfConfs = noConfConfs,
    ...
  )
}

.MetaDeconfound <- function(featureMat,
                            metaMat,
                            nnodes = 1,
                            robustCutoff = 5,
                            adjustMethod = "fdr",
                            adjustLevel = NULL,
                            QCutoff = 0.1,
                            DCutoff = 0,
                            PHS_cutoff = 0.05,
                            logfile = NULL,
                            logLevel = "INFO",
                            startStop = NA,
                            QValues = NA,
                            DValues = NA,
                            minQValues=NULL,
                            deconfT = NULL,
                            deconfF = NULL,
                            doConfs = 2,
                            doRanks = NA,
                            randomVar = NA,
                            fixedVar = NA,
                            robustCutoffRho = NULL,
                            typeCategorical = NULL,
                            typeContinuous = NULL,
                            logistic = FALSE,
                            rawCounts = FALSE,
                            returnLong = FALSE,
                            collectMods = FALSE,
                            mediationMat = NULL,
                            noConfConfs = TRUE,
                            nAGQ = 1
                            ) {

  if (nnodes < 2) {
    logger::log_info(namespace = "metadeconfoundR", 'Computing in serial mode. Set nnodes > 1 do to switch to faster parallel processing.')
    nnodes <- 1
  }

  if (length(randomVar) > 1) {
    if (nAGQ > 1) {
      nAGQ <- 1
      logger::log_warn(namespace = "metadeconfoundR", "nAGQ was set to 1. Can not be > 1 if more than one random variable is added to a glmer.")
    }
  }

  if (collectMods == TRUE) {
    logger::log_warn(namespace = "metadeconfoundR", "collectMods was set to TRUE, model building step is run with nnodes = 1.")
  }

  samples <- row.names (featureMat)
  features <- colnames (featureMat)
  noFeatures <- length (features)

  if (is.null (robustCutoffRho)) { robustCutoffRho <- robustCutoff } # new SKF20200221

  RVnames <- NA
  covariates <- colnames (metaMat) # each covariate + the status category
  if (!is.na(randomVar[[1]])) { # list input parameter is split for further use within pipeline
    RVnames <- randomVar


    logger::log_info(namespace = "metadeconfoundR", paste0("The following parameters will be added to all linear models as random effects: '", randomVar, "'"))
    logger::log_info(namespace = "metadeconfoundR", paste(randomVar))
    logger::log_warn(namespace = "metadeconfoundR", paste0("the following random effect covariates will be excluded as potential donfounders: ", paste0(RVnames, collapse = ", ")))
    logger::log_warn(namespace = "metadeconfoundR", "naive associations reducible to these efects will get the status label 'NS', while output elements Ps, Qs, and Ds will be unchanged.")
  }

  if (!is.na(fixedVar[[1]])) {
    RVnames <- na.omit(c(RVnames, fixedVar))


    logger::log_info(namespace = "metadeconfoundR", paste0("The following parameters will be added to all linear models as fixed effects: '", fixedVar, "'"))
    logger::log_info(namespace = "metadeconfoundR", paste(fixedVar))
    logger::log_warn(namespace = "metadeconfoundR", paste0("the following fixed effect covariates will be excluded as potential donfounders: ", paste0(RVnames, collapse = ", ")))
    logger::log_warn(namespace = "metadeconfoundR", "naive associations reducible to these efects will get the status label 'NS', while output elements Ps, Qs, and Ds will be unchanged.")
  }

  noCovariates <- length (covariates)

  logger::log_info(namespace = "metadeconfoundR", paste0("Checking robustness of data for covariates"))

  isRobust <- CheckSufficientPower(metaMat = metaMat,
                                   covariates = covariates,
                                   noCovariates = noCovariates,
                                   nnodes = nnodes,
                                   robustCutoff = robustCutoff,
                                   robustCutoffRho = robustCutoffRho, # new SKF20200221
                                   typeCategorical = typeCategorical, # new SKF20200221
                                   typeContinuous = typeContinuous, # new SKF20200221
                                   RVnames = RVnames,
                                   startStop = startStop,
                                   deconfF = deconfF) # new TB20220704

  logger::log_debug(namespace = "metadeconfoundR", paste(
    "CheckSufficientPower -- (dim(isRobust[[2]]):",
    paste(dim(isRobust[[2]]), collapse = ", ")))

  if (!all(isRobust[[1]])) {
    logger::log_info(namespace = "metadeconfoundR", paste0((length(isRobust[[1]]) - sum(isRobust[[1]])), " covariates where marked as too sparse and won't be considered in further analysis due to lack of sufficient data: ",
                           sub(pattern = ", ", replacement = "", x = paste0(", ", names(isRobust[[1]][isRobust[[1]] == 0]), collapse = ""))))
  }



  if (is.na(QValues[[1]]) | is.na(DValues[[1]])) {
    # if no external Qs and Ds are supplied, compute them!
    logger::log_info(namespace = "metadeconfoundR", "Computation of naive associations started.")

    # if adjustLevel is not supplied set to 1.
      # Should mediationMat be supplied, set to 3 instead.
    if (is.null(adjustLevel)) {
      adjustLevel <- 1
      if (!is.null(mediationMat)) {
        adjustLevel <- 3
        logger::log_warn(namespace = "metadeconfoundR", "adjustLevel not specified, setting to 3:
                                 multiple testing p-value correction over the number of
                                 features AND number of varables in mediationMat.")
      }
    }

    if (adjustLevel == 3 & is.null(mediationMat)) {
      logger::log_error(namespace = "metadeconfoundR", "adjustLevel == 3 not possible without supplying mediationMat.")
      stop("adjustLevel == 3 not possible without supplying mediationMat.")
    }

    naiveAssociation <- NaiveAssociation(
      featureMat = featureMat,
      samples = samples,
      features = features,
      noFeatures = noFeatures,
      metaMat = metaMat,
      covariates = covariates,
      noCovariates = noCovariates,
      isRobust = isRobust,
      typeCategorical = typeCategorical,# new SKF20200221
      typeContinuous = typeContinuous,# new SKF20200221
      logistic = logistic, # new SKF20201017
      adjustMethod = adjustMethod,
      adjustLevel = adjustLevel,
      nnodes = nnodes,
      rawCounts = rawCounts,# new TB20221129
      mediationMat
    )

    if ("naiveStop" %in% startStop) {
      logger::log_warn(namespace = "metadeconfoundR", paste('Process stopped before computing confounding status because "startStop" parameter contained "naiveStop". '))

      if (!returnLong) {
        return(list(Ps = naiveAssociation$Ps,
                    Qs = naiveAssociation$Qs,
                    Ds = naiveAssociation$Ds
                    ))
      }

      long_out <- reshape2::melt(naiveAssociation$Ps, varnames = c("feature", "metaVariable"), value.name = "Ps")
      long_out$Qs <- reshape2::melt(naiveAssociation$Qs)[, 3]
      long_out$Ds <- reshape2::melt(naiveAssociation$Ds)[, 3]
      return(long_out)
    }
  } else { # if precomputed Qs and Ds are supplied as arguments
    naiveAssociation <- list(Ps = QValues, Qs = QValues, Ds = DValues)
    logger::log_warn(namespace = "metadeconfoundR", paste("Naive p-values are NOT computed, but simply are a copy of the supplied adjusted p-values (QValues)."))
    logger::log_warn(namespace = "metadeconfoundR", paste('Confonding status is computed based on Q-values and effect sizes supplied via "QValue" and "DValue" parameters. '))
  }

  if (collectMods == TRUE) {
    nnodes <- 1
    logger::log_warn(namespace = "metadeconfoundR", paste('collectMods == TRUE --> setting nnodes = 1 for model building phase.'))
  }


    reducibilityStatus <- CheckReducibility(featureMat = featureMat,
                                            metaMat = metaMat,
                                            noFeatures = noFeatures,
                                            noCovariates = noCovariates,
                                            features = features,
                                            covariates = covariates,
                                            Qs = naiveAssociation$Qs,
                                            Ds = naiveAssociation$Ds,
                                            minQValues= minQValues,
                                            nnodes = nnodes,
                                            QCutoff = QCutoff,
                                            DCutoff = DCutoff,
                                            PHS_cutoff = PHS_cutoff,
                                            deconfT = deconfT,
                                            deconfF = deconfF,
                                            doConfs = doConfs,
                                            doRanks = doRanks,
                                            randomVar = randomVar,
                                            fixedVar = fixedVar, # new TB20230727
                                            RVnames = RVnames,
                                            isRobust = isRobust,
                                            logistic = logistic, # new SKF20201017,
                                            rawCounts = rawCounts, # new TB20220202
                                            nAGQ = nAGQ, # new TB 20221201
                                            collectMods = collectMods, # new TB20220208
                                            noConfConfs = noConfConfs, # new TB20250827
                                            )
  #}

  if (collectMods) {
    collectedMods <- reducibilityStatus[[2]]
    reducibilityStatus <-reducibilityStatus[[1]]
  }

  logger::log_info(namespace = "metadeconfoundR", "MetadecondoundR run completed successfully!")

  if (!returnLong) {
    returnList <- list(Ps = naiveAssociation$Ps,
                       Qs = naiveAssociation$Qs,
                       Ds = naiveAssociation$Ds,
                       status=reducibilityStatus)
    if (collectMods) {
      returnList$collectedMods <- collectedMods
    }
    return(returnList)
  }


  long_out <- reshape2::melt(naiveAssociation$Ps, varnames = c("feature", "metaVariable"), value.name = "Ps")
  long_out$Qs <- reshape2::melt(naiveAssociation$Qs)[, 3]
  long_out$Ds <- reshape2::melt(naiveAssociation$Ds)[, 3]
  long_out$status <- reshape2::melt(reducibilityStatus)[, 3]

  if (!is.null(mediationMat)) {
    long_out$groupingVar <- "metadata"
    long_out$groupingVar[long_out$metaVariable %in% colnames(mediationMat)] <- "secondOmicsSpace"
    long_out$groupingVar <- factor(long_out$groupingVar, levels = c("secondOmicsSpace", "metadata"))
  }
  #long_out_signif <- subset(x = long_out, (status != "NS") & !is.na(status))
  if (collectMods) {
    long_out <- list(stdOutput = long_out,
                     collectedMods = collectedMods)
    logger::log_warn(namespace = "metadeconfoundR", 'Collected models are part ouf output. Only use output[["stdOutput"]] as BuildHeatmap input!')
  }
  return(long_out)

}
