#' Create the path elements for AEMET API
#'
#' Path vectors for AEMET API to use with httr2
#'
#' @param api_options Option list as generated by \link{\code{aemet_options}}
#'
#' @noRd
.create_aemet_path <- function(api_options) {

  # we need the resolution to create the corresponding path
  resolution <- api_options$resolution
  # we need to transform the dates to the character string specific format for the AEMET path.
  # We will use a stamp function:
  aemet_stamp <- lubridate::stamp("2020-12-25T00:00:00UTC", orders = "YOmdHMS", quiet = TRUE)

  # current day
  if (resolution == 'current_day') {
    return(c('opendata', 'api', 'observacion', 'convencional', 'todas'))
  }

  # daily
  if (resolution == 'daily') {
    return(
      c(
        'opendata', 'api', 'valores', 'climatologicos', 'diarios', 'datos',
        'fechaini', aemet_stamp(api_options$start_date),
        'fechafin', aemet_stamp(api_options$end_date),
        'todasestaciones'
      )
    )
  }

  # monthly
  # monthly API does not work for now
  if (resolution %in% c('monthly', 'yearly')) {
    # stop if stations is null. For monthly API, one and only one station must be provided
    if (length(api_options$stations) < 1) {
      cli::cli_abort(c(
        "AEMET API for monthly/yearly aggregated values needs one station provided"
      ))
    }

    # issue a warning if more than one station is provided
    if (length(api_options$stations) > 1) {
      cli::cli_warn(c(
        "AEMET API for monthly/yearly aggregated values only accepts one station per query.\n",
        "Only the first station provided ({.val {api_options$stations[1]}}) will be used."
      ))
    }

    return(
      c(
        'opendata', 'api', 'valores', 'climatologicos', 'mensualesanuales', 'datos',
        'anioini', lubridate::year(api_options$start_date), 'aniofin', lubridate::year(api_options$end_date),
        'estacion', api_options$stations[1]
      )
    )
  }

  # not recognised resolution
  cli::cli_abort(c(
    "{.arg {api_options$resolution}} is not a valid temporal resolution for AEMET. Please see aemet_options help for more information"
  ))
}

#' Create request for AEMET API
#' 
#' Create the request for aemet based on api_options (for key) and path
#' 
#' This function creates the request, deal with errors and retries if needed
#' and access the data and metadata of the response.
#' 
#' @param path character vector with the path as obtained by
#'   \link{\code{.create_aemet_path}}
#' @param api_options Option list as generated by \link{\code{aemet_options}}
#' 
#' @return a list with two elements, "data" and "metadata". Both are tibbles.
#' 
#' @noRd 
.create_aemet_request <- function(path, api_options) {

  aemet_request <- httr2::request("https://opendata.aemet.es") |>
    httr2::req_url_path_append(path) |>
    httr2::req_headers_redacted(api_key = api_options$api_key) |>
    httr2::req_headers("Cache-Control" = "no-cache") |>
    httr2::req_user_agent(
      "meteospain R package (https://emf.creaf.cat/software/meteospain/)"
    ) |>
    httr2::req_error(
      body = \(resp) {
        message <- httr2::resp_body_string(resp)

        # known errors for better messaging
        if (httr2::resp_content_type(resp) == "text/html") {
          message <- httr2::resp_body_html(resp) |>
            rvest::html_element("body") |>
            rvest::html_text()
          return(message)
        }

        # text plain but NOT after all retries and still 429
        if (httr2::resp_content_type(resp) == "text/plain" && httr2::resp_status(resp) != 429L) {
          message <- httr2::resp_body_string(resp) |>
            jsonlite::fromJSON() |>
            _$descripcion
          return(message)
        }

        if (httr2::resp_content_type(resp) == "application/json") {
          message <- httr2::resp_body_json(resp, simplifyVector = TRUE)
          return(message)
        }

        message
      }
    ) |>
    httr2::req_retry(
      max_tries = 4,
      retry_on_failure = TRUE,
      is_transient = \(resp) {
        httr2::resp_status(resp) %in% c(429, 500, 503)
      },
      backoff = \(resp) {
        61
      },
      after = \(resp) {
        # if error is 500, with 429 inside as html, then wait 61 seconds
        if (
          httr2::resp_status(resp) %in% c(500) &&
            httr2::resp_content_type(resp) == "text/html" &&
            stringr::str_detect(
              rvest::html_text(rvest::html_element(httr2::resp_body_html(resp), "h1")),
              "429 Too Many Requests"
            )
        ) {
          return(61)
        }

        # default, left backoff decide (NA)
        NA
      }
    )

  # intermediate response. If everything is ok, estado is 200 and there are
  # "datos" and "metadatos" fields in the json response. If not (i.e. when
  # dates are outside the range), the intermediate response is still 200, but
  # doesn't have "datos" and "metadatos" fields and the estado is different
  intermediate_response <- httr2::req_perform(aemet_request)

  if (httr2::resp_body_json(intermediate_response)$estado != 200L) {
    cli::cli_abort(c(
      "x" = httr2::resp_body_json(intermediate_response)$estado,
      "i" = httr2::resp_body_json(intermediate_response)$descripcion
    ))
  }

  aemet_data_response <- aemet_request |>
    httr2::req_url(httr2::resp_body_json(intermediate_response)$datos) |>
    httr2::req_perform() |>
    httr2::resp_body_string() |>
    jsonlite::fromJSON() |>
    dplyr::as_tibble()

  aemet_metadata_response <- aemet_request |>
    httr2::req_url(httr2::resp_body_json(intermediate_response)$metadatos) |>
    httr2::req_perform() |>
    httr2::resp_body_string() |>
    jsonlite::fromJSON() |>
    dplyr::as_tibble()

  return(list(data = aemet_data_response, metadata = aemet_metadata_response))
}

#' Get info for the aemet stations
#'
#' Get info for the aemet stations
#'
#' @noRd

.get_info_aemet <- function(api_options) {
  # path
  path_resolution <- c(
    'opendata', 'api', 'valores', 'climatologicos', 'inventarioestaciones', 'todasestaciones'
  )
  # cache
  cache_ref <- rlang::hash(c(path_resolution, api_options$api_key))

  # get data from cache or from API if new
  info_aemet <- .get_cached_result(cache_ref, {

    # create the request, get the response and transform the data
    .create_aemet_request(path_resolution, api_options)$data |>
      dplyr::as_tibble() |>
      # add service name, to identify the data if joining with other services
      dplyr::mutate(service = 'aemet') |>
      dplyr::select(
        "service", station_id = "indicativo", station_name = "nombre",
        station_province = "provincia", altitude = "altitud", latitude = "latitud",
        longitude = "longitud"
      ) |>
      # latitude and longitude are in strings with the cardinal letter. We need to transform that to numeric
      # and negative when S or W.
      dplyr::mutate(
        altitude = as.numeric(stringr::str_replace_all(.data$altitude, ',', '.')),
        altitude = units::set_units(.data$altitude, "m"),
        latitude = .aemet_coords_generator(.data$latitude),
        longitude = .aemet_coords_generator(.data$longitude)
      ) |>
      sf::st_as_sf(coords = c('longitude', 'latitude'), crs = 4326)

  })

  return(info_aemet)
}

#' Get data from AEMET
#'
#' Get data from AEMET service
#'
#' For current_day and daily, there is no need of supply the stations_id in the query,
#' as the data is not so big. So, in case of stations provided, we can filter later, after getting
#' the data. This also has the advantage of using only one query, reducing the probability of reaching
#' the API limit per minute or total.
#'
#' @param api_options Option list as generated by \link{\code{aemet_options}}
#'
#' @noRd
.get_data_aemet <- function(api_options) {
  # create api path
  path_resolution <- .create_aemet_path(api_options)

  # cache
  cache_ref <- rlang::hash(c(path_resolution, api_options$api_key))

  # if resolution less than daily, remove the cache
  if (api_options$resolution == "current_day") {
    apis_cache$remove(cache_ref)
  }

  # get data from cache or from API if new
  data_aemet <- .get_cached_result(cache_ref, {

    # In this case, we need the stations info also, so we get it:
    stations_info <- .get_info_aemet(api_options)

    # Resolution specific carpentry 
    # Now, current day and daily have differences, in the names of the variables and also
    # in the need to join the stations data to offer coords. We can branch the code with ifs, repeating the
    # common steps in the data carpentry or we can create the specific functions and have only one common pipe.
    resolution_specific_carpentry <- switch(
      api_options$resolution,
      'current_day' = .aemet_current_day_carpentry,
      'daily' = .aemet_daily_carpentry,
      'monthly' = .aemet_monthly_yearly_carpentry,
      'yearly' = .aemet_monthly_yearly_carpentry
    )

    # and now data: create the request, get the response and transform the data
    stations_data <- .create_aemet_request(path_resolution, api_options)

    res <- stations_data$data |>
      # apply the resolution-specific transformations
      resolution_specific_carpentry(stations_info, resolution = api_options$resolution) |>
      # arrange data
      dplyr::arrange(.data$timestamp, .data$station_id) |>
      # reorder variables to be consistent among all services
      relocate_vars() |>
      # ensure we have an sf
      sf::st_as_sf()

    # Copyright message -------------------------------------------------------------------------------------
    cli::cli_inform(c(
      i = copyright_style(unique(stations_data$metadata$copyright)),
      legal_note_style(unique(stations_data$metadata$notaLegal))
    ))

    # Return ------------------------------------------------------------------------------------------------
    res
  })

  # Filter expression for stations ------------------------------------------------------------------------
  # In case stations were supplied, we need also to filter them
  filter_expression <- TRUE
  # update filter if there is stations supplied, but not for monthly. In monthly only one
  # station must be used, so the filtering is unnecesary
  if (!rlang::is_null(api_options$stations)) {
    filter_expression <- switch(
      api_options$resolution,
      'current_day' = rlang::expr(.data$station_id %in% api_options$stations),
      'daily' = rlang::expr(.data$station_id %in% api_options$stations),
      'monthly' = TRUE,
      'yearly' = TRUE
    )
  }

  # return the res filtered by stations if provided
  data_aemet_fil <- data_aemet |>
    # remove unwanted stations
    dplyr::filter(!! filter_expression)

  # Check if any stations were returned -------------------------------------------------------------------
  if ((!is.null(api_options$stations)) & nrow(data_aemet_fil) < 1) {
    cli::cli_abort(c(
      x = "Station(s) provided have no data for the dates selected.",
      "Available stations with data for the actual query are:",
      glue::glue_collapse(unique(data_aemet$station_id), sep = ', ', last = ' and ')
    ))
  }

  return(data_aemet_fil)
}


# resolution_specific_carpentry -------------------------------------------------------------------------
.aemet_current_day_carpentry <- function(data, stations_info, ...) {
  data |>
    dplyr::select(dplyr::any_of(c(
      timestamp = "fint", station_id = "idema", station_name = "ubi",
      altitude = "alt",
      precipitation = "prec",
      max_wind_speed = "vmax",
      wind_direction = "dv",
      wind_speed = "vv",
      max_wind_direction = "dmax",
      relative_humidity = "hr",
      insolation = "inso",
      atmospheric_pressure = "pres",
      temperature_soil = "ts",
      temperature_soil_20 = "tss20cm",
      temperature_soil_5 = "tss5cm",
      temperature = "ta",
      temperature_dew_point = "tpr",
      min_temperature = "tamin",
      max_temperature = "tamax",
      snow_cover = "nieve",
      longitude = "lon", latitude = "lat"
    ))) |>
    # create any variable missing
    .create_missing_vars(
      var_names = c(
        "precipitation", "max_wind_speed", "wind_direction", "wind_speed",
        "max_wind_direction", "relative_humidity", "insolation",
        "atmospheric_pressure", "temperature_soil", "temperature_soil_20",
        "temperature_soil_5", "temperature", "temperature_dew_point",
        "min_temperature", "max_temperature", "snow_cover"
      )
    ) |>
    # units
    dplyr::mutate(
      service = 'aemet',
      timestamp = lubridate::as_datetime(.data$timestamp),
      altitude = units::set_units(.data$altitude, "m"),
      precipitation = units::set_units(.data$precipitation, "L/m^2"),
      max_wind_speed = units::set_units(.data$max_wind_speed, "m/s"),
      wind_direction = units::set_units(.data$wind_direction, "degree"),
      wind_speed = units::set_units(.data$wind_speed, "m/s"),
      max_wind_direction = units::set_units(.data$max_wind_direction, "degree"),
      relative_humidity = units::set_units(.data$relative_humidity, "%"),
      insolation = units::set_units(.data$insolation, "hours"),
      atmospheric_pressure = units::set_units(.data$atmospheric_pressure, "hPa"),
      temperature_soil = units::set_units(.data$temperature_soil, "degree_C"),
      temperature_soil_20 = units::set_units(.data$temperature_soil_20, "degree_C"),
      temperature_soil_5 = units::set_units(.data$temperature_soil_5, "degree_C"),
      temperature = units::set_units(.data$temperature, "degree_C"),
      temperature_dew_point = units::set_units(.data$temperature_dew_point, "degree_C"),
      min_temperature = units::set_units(.data$min_temperature, "degree_C"),
      max_temperature = units::set_units(.data$max_temperature, "degree_C"),
      snow_cover = units::set_units(.data$snow_cover, "cm")
    ) |>
    dplyr::left_join(stations_info, by = c('service', 'station_id', 'station_name', 'altitude')) |>
    sf::st_as_sf(coords = c('longitude', 'latitude'), crs = 4326)
}

.aemet_daily_carpentry <- function(data, stations_info, ...) {
  data |>
    dplyr::select(dplyr::any_of(c(
      timestamp = "fecha",
      station_id = "indicativo", station_name = "nombre", station_province = "provincia",
      mean_temperature = "tmed",
      precipitation = "prec",
      min_temperature = "tmin",
      max_temperature = "tmax",
      wind_direction = "dir",
      mean_wind_speed = "velmedia",
      max_wind_speed = "racha",
      insolation = "sol",
      max_atmospheric_pressure = "presmax",
      min_atmospheric_pressure = "presmin",
      mean_relative_humidity = "hrMedia",
      max_relative_humidity = "hrMax",
      min_relative_humidity = "hrMin"
    ))) |>
    # create any variable missing
    .create_missing_vars(
      var_names = c(
        "mean_temperature", "precipitation", "min_temperature", "max_temperature",
        "wind_direction", "mean_wind_speed", "max_wind_speed", "insolation",
        "max_atmospheric_pressure", "min_atmospheric_pressure", "mean_relative_humidity",
        "max_relative_humidity", "min_relative_humidity"
      )
    ) |>
    # variables are characters, with "," as decimal point, so....
    dplyr::mutate(
      service = 'aemet',
      timestamp = lubridate::as_datetime(.data$timestamp),
      mean_temperature = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_temperature, ",", "."))),
      precipitation = suppressWarnings(as.numeric(stringr::str_replace_all(.data$precipitation, ",", "."))),
      min_temperature = suppressWarnings(as.numeric(stringr::str_replace_all(.data$min_temperature, ",", "."))),
      max_temperature = suppressWarnings(as.numeric(stringr::str_replace_all(.data$max_temperature, ",", "."))),
      wind_direction = suppressWarnings(as.numeric(stringr::str_replace_all(.data$wind_direction, ",", "."))),
      mean_wind_speed = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_wind_speed, ",", "."))),
      max_wind_speed = suppressWarnings(as.numeric(stringr::str_replace_all(.data$max_wind_speed, ",", "."))),
      insolation = suppressWarnings(as.numeric(stringr::str_replace_all(.data$insolation, ",", "."))),
      max_atmospheric_pressure = suppressWarnings(
        as.numeric(stringr::str_replace_all(.data$max_atmospheric_pressure, ",", "."))
      ),
      min_atmospheric_pressure = suppressWarnings(
        as.numeric(stringr::str_replace_all(.data$min_atmospheric_pressure, ",", "."))
      ),
      mean_relative_humidity = suppressWarnings(
        as.numeric(stringr::str_replace_all(.data$mean_relative_humidity, ",", "."))
      ),
      max_relative_humidity = suppressWarnings(
        as.numeric(stringr::str_replace_all(.data$max_relative_humidity, ",", "."))
      ),
      min_relative_humidity = suppressWarnings(
        as.numeric(stringr::str_replace_all(.data$min_relative_humidity, ",", "."))
      ),
      # and set the units also
      mean_temperature = units::set_units(.data$mean_temperature, "degree_C"),
      precipitation = units::set_units(.data$precipitation, "L/m^2"),
      min_temperature = units::set_units(.data$min_temperature, "degree_C"),
      max_temperature = units::set_units(.data$max_temperature, "degree_C"),
      wind_direction = units::set_units(.data$wind_direction, "degree"),
      mean_wind_speed = units::set_units(.data$mean_wind_speed, "m/s"),
      max_wind_speed = units::set_units(.data$max_wind_speed, "m/s"),
      insolation = units::set_units(.data$insolation, "hours"),
      max_atmospheric_pressure = units::set_units(.data$max_atmospheric_pressure, "hPa"),
      min_atmospheric_pressure = units::set_units(.data$min_atmospheric_pressure, "hPa"),
      mean_relative_humidity = units::set_units(.data$mean_relative_humidity, "%"),
      max_relative_humidity = units::set_units(.data$max_relative_humidity, "%"),
      min_relative_humidity = units::set_units(.data$min_relative_humidity, "%")
    ) |>
    dplyr::left_join(stations_info, by = c('service', 'station_id', 'station_name', 'station_province'))
}

.aemet_monthly_yearly_carpentry <- function(data, stations_info, resolution) {

  # resolution depending negate argument
  negate_filter <- FALSE
  if (resolution == "monthly") {
    negate_filter <- TRUE
  }
  # data carpentry
  data |>
    dplyr::select(dplyr::any_of(c(
      timestamp = "fecha",
      station_id = "indicativo", station_name = "nombre", station_province = "provincia",
      # temperatures
      mean_temperature = "tm_mes",
      min_temperature_mean = "tm_min",
      max_temperature_mean = "tm_max",
      min_temperature_absolute = "ta_min",
      max_temperature_absolute = "ta_max",
      min_temperature_max = "ts_min",
      max_temperature_min = "ti_max",
      temperature_days_30 = "nt_30",
      temperature_days_0 = "nt_00",
      mean_temperature_soil_10 = "ts_10",
      mean_temperature_soil_20 = "ts_20",
      mean_temperature_soil_50 = "ts_50",
      # rh
      mean_relative_humidity = "hr",
      vapour_pressure = "e",
      evaporation_total = "evap",
      # precipitation
      total_precipitation = "p_mes",
      rain_days = "n_llu",
      rain_days_01 = "np_001",
      rain_days_1 = "np_010",
      rain_days_10 = "np_100",
      rain_days_30 = "np_300",
      snow_days = "n_nie",
      hail_days = "n_gra",
      storm_days = "n_tor",
      fog_days = "n_fog",
      clear_days = "n_des",
      cloudy_days = "n_nub",
      cover_days = "n_cub",
      # wind
      mean_wind_speed = "w_med",
      # radiation
      mean_insolation = "inso",
      mean_global_radiation = "glo",
      insolation_perc = "p_sol",
      # pressure
      mean_atmospheric_pressure = "q_med",
      max_atmospheric_pressure = "q_max",
      min_atmospheric_pressure = "q_min"
    ))) |>
    # remove yearly or monthly values, depending on resolution
    dplyr::filter(
      stringr::str_detect(.data$timestamp, "-13", negate = negate_filter)
    ) |>
    # remove any "-13" for yearly values (if monthly, this step dont do anything), for
    # the timestamp parsing to work
    dplyr::mutate(
      timestamp = stringr::str_remove(.data$timestamp, "-13")
    ) |>
    # create any variable missing
    .create_missing_vars(
      var_names = c(
        "mean_temperature", "min_temperature_mean", "max_temperature_mean",
        "min_temperature_absolute", "max_temperature_absolute", "min_temperature_max",
        "max_temperature_min", "temperature_days_30", "temperature_days_0",
        "mean_temperature_soil_10", "mean_temperature_soil_20", "mean_temperature_soil_50",
        "mean_relative_humidity", "vapour_pressure", "evaporation_total",
        "total_precipitation", "rain_days", "rain_days_01", "rain_days_1", "rain_days_10",
        "rain_days_30", "snow_days", "hail_days", "storm_days", "fog_days", "clear_days",
        "cloudy_days", "cover_days", "mean_wind_speed", "mean_insolation",
        "mean_global_radiation", "insolation_perc", "mean_atmospheric_pressure",
        "max_atmospheric_pressure", "min_atmospheric_pressure"
      )
    ) |>
    # timestamp has to be parsed, "ym" for monthly values, "y" for yearly, and
    # variables are characters, with "," as decimal point, so....
    dplyr::mutate(
      service = 'aemet',
      timestamp = lubridate::parse_date_time(.data$timestamp, orders = c("ym", "y")),
      mean_temperature = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_temperature, ",", "."))),
      min_temperature_mean = suppressWarnings(as.numeric(stringr::str_replace_all(.data$min_temperature_mean, ",", "."))),
      max_temperature_mean = suppressWarnings(as.numeric(stringr::str_replace_all(.data$max_temperature_mean, ",", "."))),
      min_temperature_absolute = suppressWarnings(as.numeric(stringr::str_replace_all(.data$min_temperature_absolute, ",", "."))),
      max_temperature_absolute = suppressWarnings(as.numeric(stringr::str_replace_all(.data$max_temperature_absolute, ",", "."))),
      min_temperature_max = suppressWarnings(as.numeric(stringr::str_replace_all(.data$min_temperature_max, ",", "."))),
      max_temperature_min = suppressWarnings(as.numeric(stringr::str_replace_all(.data$max_temperature_min, ",", "."))),
      temperature_days_30 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$temperature_days_30, ",", "."))),
      temperature_days_0 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$temperature_days_0, ",", "."))),
      mean_temperature_soil_10 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_temperature_soil_10, ",", "."))),
      mean_temperature_soil_20 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_temperature_soil_20, ",", "."))),
      mean_temperature_soil_50 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_temperature_soil_50, ",", "."))),
      mean_relative_humidity = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_relative_humidity, ",", "."))),
      vapour_pressure = suppressWarnings(as.numeric(stringr::str_replace_all(.data$vapour_pressure, ",", ".")) / 10),
      evaporation_total = suppressWarnings(as.numeric(stringr::str_replace_all(.data$evaporation_total, ",", "."))),
      total_precipitation = suppressWarnings(as.numeric(stringr::str_replace_all(.data$total_precipitation, ",", "."))),
      rain_days = suppressWarnings(as.numeric(stringr::str_replace_all(.data$rain_days, ",", "."))),
      rain_days_01 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$rain_days_01, ",", "."))),
      rain_days_1 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$rain_days_1, ",", "."))),
      rain_days_10 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$rain_days_10, ",", "."))),
      rain_days_30 = suppressWarnings(as.numeric(stringr::str_replace_all(.data$rain_days_30, ",", "."))),
      snow_days = suppressWarnings(as.numeric(stringr::str_replace_all(.data$snow_days, ",", "."))),
      hail_days = suppressWarnings(as.numeric(stringr::str_replace_all(.data$hail_days, ",", "."))),
      storm_days = suppressWarnings(as.numeric(stringr::str_replace_all(.data$storm_days, ",", "."))),
      fog_days = suppressWarnings(as.numeric(stringr::str_replace_all(.data$fog_days, ",", "."))),
      clear_days = suppressWarnings(as.numeric(stringr::str_replace_all(.data$clear_days, ",", "."))),
      cloudy_days = suppressWarnings(as.numeric(stringr::str_replace_all(.data$cloudy_days, ",", "."))),
      cover_days = suppressWarnings(as.numeric(stringr::str_replace_all(.data$cover_days, ",", "."))),
      mean_wind_speed = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_wind_speed, ",", "."))),
      mean_insolation = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_insolation, ",", "."))),
      mean_global_radiation = 10 * suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_global_radiation, ",", "."))),
      insolation_perc = suppressWarnings(as.numeric(stringr::str_replace_all(.data$insolation_perc, ",", "."))),
      mean_atmospheric_pressure = suppressWarnings(as.numeric(stringr::str_replace_all(.data$mean_atmospheric_pressure, ",", "."))),
      max_atmospheric_pressure = suppressWarnings(as.numeric(stringr::str_replace_all(.data$max_atmospheric_pressure, ",", "."))),
      min_atmospheric_pressure = suppressWarnings(as.numeric(stringr::str_replace_all(.data$min_atmospheric_pressure, ",", "."))),
      # and set the units also
      mean_temperature = units::set_units(.data$mean_temperature, "degree_C"),
      min_temperature_mean = units::set_units(.data$min_temperature_mean, "degree_C"),
      max_temperature_mean = units::set_units(.data$max_temperature_mean, "degree_C"),
      min_temperature_absolute = units::set_units(.data$min_temperature_absolute, "degree_C"),
      max_temperature_absolute = units::set_units(.data$max_temperature_absolute, "degree_C"),
      min_temperature_max = units::set_units(.data$min_temperature_max, "degree_C"),
      max_temperature_min = units::set_units(.data$max_temperature_min, "degree_C"),
      temperature_days_30 = units::set_units(.data$temperature_days_30, "days"),
      temperature_days_0 = units::set_units(.data$temperature_days_0, "days"),
      mean_temperature_soil_10 = units::set_units(.data$mean_temperature_soil_10, "degree_C"),
      mean_temperature_soil_20 = units::set_units(.data$mean_temperature_soil_20, "degree_C"),
      mean_temperature_soil_50 = units::set_units(.data$mean_temperature_soil_50, "degree_C"),
      mean_relative_humidity = units::set_units(.data$mean_relative_humidity, "%"),
      vapour_pressure = units::set_units(.data$vapour_pressure, "hPa"),
      evaporation_total = units::set_units(.data$evaporation_total, "degree_C"),
      total_precipitation = units::set_units(.data$total_precipitation, "degree_C"),
      rain_days = units::set_units(.data$rain_days, "days"),
      rain_days_01 = units::set_units(.data$rain_days_01, "days"),
      rain_days_1 = units::set_units(.data$rain_days_1, "days"),
      rain_days_10 = units::set_units(.data$rain_days_10, "days"),
      rain_days_30 = units::set_units(.data$rain_days_30, "days"),
      snow_days = units::set_units(.data$snow_days, "days"),
      hail_days = units::set_units(.data$hail_days, "days"),
      storm_days = units::set_units(.data$storm_days, "days"),
      fog_days = units::set_units(.data$fog_days, "days"),
      clear_days = units::set_units(.data$clear_days, "days"),
      cloudy_days = units::set_units(.data$cloudy_days, "days"),
      cover_days = units::set_units(.data$cover_days, "days"),
      mean_wind_speed = units::set_units(.data$mean_wind_speed, "km/h"),
      mean_insolation = units::set_units(.data$mean_insolation, "hours"),
      mean_global_radiation = units::set_units(.data$mean_global_radiation, "kJ/m^2"),
      insolation_perc = units::set_units(.data$insolation_perc, "%"),
      mean_atmospheric_pressure = units::set_units(.data$mean_atmospheric_pressure, "hPa"),
      max_atmospheric_pressure = units::set_units(.data$max_atmospheric_pressure, "hPa"),
      min_atmospheric_pressure = units::set_units(.data$min_atmospheric_pressure, "hPa")
    ) |>
    dplyr::left_join(stations_info, by = c('service', 'station_id'))
}

