% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ps_trim.R
\name{ps_trim}
\alias{ps_trim}
\title{Trim Propensity Scores}
\usage{
ps_trim(
  ps,
  method = c("ps", "adaptive", "pctl", "pref", "cr", "optimal"),
  lower = NULL,
  upper = NULL,
  .exposure = NULL,
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)
}
\arguments{
\item{ps}{A numeric vector of propensity scores in (0, 1) for binary
exposures, or a matrix / data frame where each column gives the propensity
score for one level of a categorical exposure.}

\item{method}{Trimming method. One of:
\itemize{
\item \strong{\code{"ps"}} (default): Fixed threshold. Observations with propensity
scores outside \verb{[lower, upper]} are trimmed. For categorical exposures,
observations where \emph{any} column falls below \code{lower} (the symmetric
threshold delta) are trimmed.
\item \strong{\code{"adaptive"}}: Data-driven threshold that minimizes the asymptotic
variance of the IPW estimator (Crump et al., 2009). The \code{lower} and
\code{upper} arguments are ignored.
\item \strong{\code{"pctl"}}: Quantile-based. Observations outside the \verb{[lower, upper]}
quantiles of the propensity score distribution are trimmed. Defaults:
\code{lower = 0.05}, \code{upper = 0.95}.
\item \strong{\code{"pref"}}: Preference score trimming. Transforms propensity scores
to the preference scale (Walker et al., 2013) and trims outside
\verb{[lower, upper]}. Requires \code{.exposure}. Binary exposures only. Defaults:
\code{lower = 0.3}, \code{upper = 0.7}.
\item \strong{\code{"cr"}}: Common range (clinical equipoise). Trims to the overlap
region of the propensity score distributions across exposure groups.
Requires \code{.exposure}. Binary exposures only. The \code{lower} and \code{upper}
arguments are ignored.
\item \strong{\code{"optimal"}}: Multi-category optimal trimming (Yang et al., 2016).
Categorical exposures only. Requires \code{.exposure}.
}

For categorical exposures, only \code{"ps"} and \code{"optimal"} are supported.}

\item{lower, upper}{Numeric thresholds whose interpretation depends on
\code{method}:
\itemize{
\item \code{"ps"}: absolute propensity score bounds (defaults: 0.1, 0.9). For
categorical exposures, only \code{lower} is used as the symmetric threshold.
\item \code{"pctl"}: quantile probabilities (defaults: 0.05, 0.95).
\item \code{"pref"}: preference score bounds (defaults: 0.3, 0.7).
\item \code{"adaptive"}, \code{"cr"}, \code{"optimal"}: ignored (thresholds are data-driven).
}}

\item{.exposure}{An exposure variable. Required for \code{"pref"}, \code{"cr"} (binary
vector), and \code{"optimal"} (factor or character). Not required for other
methods.}

\item{.focal_level}{The value of \code{.exposure} representing the focal
(treated) group. For binary exposures, defaults to the higher value.
Required for \code{wt_att()} and \code{wt_atu()} with categorical exposures.}

\item{.reference_level}{The value of \code{.exposure} representing the reference
(control) group. Automatically detected if not supplied.}

\item{...}{Additional arguments passed to methods.}

\item{.treated}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{.focal_level} instead.}

\item{.untreated}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{.reference_level} instead.}
}
\value{
A \strong{\code{ps_trim}} object (a numeric vector with class \code{"ps_trim"}, or a
matrix with class \code{"ps_trim_matrix"}). Trimmed observations are \code{NA}.
Metadata is stored in the \code{"ps_trim_meta"} attribute and can be accessed
with \code{\link[=ps_trim_meta]{ps_trim_meta()}}. Key fields include:
\itemize{
\item \code{method}: the trimming method used
\item \code{keep_idx}: integer indices of retained observations
\item \code{trimmed_idx}: integer indices of trimmed (NA) observations
\item Method-specific fields such as \code{cutoff} (adaptive), \code{q_lower}/\code{q_upper}
(pctl), \code{cr_lower}/\code{cr_upper} (cr), \code{delta} (categorical ps),
or \code{lambda} (optimal)
}
}
\description{
Trim observations with extreme propensity scores by replacing them with \code{NA},
effectively removing those units from downstream analyses. The returned object
has the same length (or dimensions) as the input, with trimmed entries set to
\code{NA}. After trimming, refit the propensity score model on the retained
observations with \code{\link[=ps_refit]{ps_refit()}}.
}
\details{
\subsection{How trimming works}{

Trimming identifies observations with extreme (near 0 or 1) propensity
scores and sets them to \code{NA}. These observations are excluded from
subsequent weight calculations and effect estimation. The goal is to
remove units that lack sufficient overlap between exposure groups, which
would otherwise receive extreme weights and destabilize estimates.
}

\subsection{Choosing a method}{
\itemize{
\item Use \code{"ps"} when you have a specific threshold in mind or want a simple
default.
\item Use \code{"adaptive"} for a principled, data-driven cutoff that targets
variance reduction.
\item Use \code{"pctl"} to trim a fixed percentage of extreme values from each tail.
\item Use \code{"pref"} when you want to restrict to the region of clinical
equipoise based on the preference score.
\item Use \code{"cr"} to restrict to the common support region where both exposure
groups have observed propensity scores.
\item Use \code{"optimal"} for multi-category (3+) exposures; this is the only
data-driven method available for categorical treatments.
}
}

\subsection{Typical workflow}{
\enumerate{
\item Fit a propensity score model
\item Apply \code{ps_trim()} to flag extreme values
\item Call \code{\link[=ps_refit]{ps_refit()}} to re-estimate propensity scores on the retained sample
\item Compute weights with \code{\link[=wt_ate]{wt_ate()}} or another weight function
}
}

\subsection{Object behavior}{

Arithmetic operations on \code{ps_trim} objects return plain numeric vectors,
since transformed propensity scores (e.g., \code{1/ps}) are no longer propensity
scores. Trimmed values propagate as \code{NA} in calculations; use \code{na.rm = TRUE}
where appropriate.

When combining \code{ps_trim} objects with \code{\link[=c]{c()}}, trimming parameters must match.
Mismatched parameters trigger a warning and return a numeric vector.

Use \code{\link[=ps_trim_meta]{ps_trim_meta()}} to inspect the trimming metadata, including the method,
cutoffs, and which observations were retained or trimmed.
}
}
\examples{
set.seed(2)
n <- 300
x <- rnorm(n)
z <- rbinom(n, 1, plogis(1.3 * x))
fit <- glm(z ~ x, family = binomial)
ps <- predict(fit, type = "response")

# Fixed threshold trimming (default)
trimmed <- ps_trim(ps, method = "ps", lower = 0.1, upper = 0.9)
trimmed

# How many observations were trimmed?
sum(is_unit_trimmed(trimmed))

# Data-driven adaptive trimming
ps_trim(ps, method = "adaptive")

# Quantile-based trimming at 5th and 95th percentiles
ps_trim(ps, method = "pctl")

# Refit after trimming, then compute weights
trimmed <- ps_trim(ps, method = "adaptive")
refitted <- ps_refit(trimmed, fit)
wt_ate(refitted, .exposure = z)

}
\references{
Crump, R. K., Hotz, V. J., Imbens, G. W., & Mitnik, O. A. (2009). Dealing
with limited overlap in estimation of average treatment effects.
\emph{Biometrika}, 96(1), 187--199.

Walker, A. M., Patrick, A. R., Lauer, M. S., et al. (2013). A tool for
assessing the feasibility of comparative effectiveness research.
\emph{Comparative Effectiveness Research}, 3, 11--20.

Yang, S., Imbens, G. W., Cui, Z., Faries, D. E., & Kadziola, Z. (2016).
Propensity score matching and subclassification in observational studies
with multi-level treatments. \emph{Biometrics}, 72(4), 1055--1065.
}
\seealso{
\code{\link[=ps_trunc]{ps_trunc()}} for bounding (winsorizing) instead of discarding,
\code{\link[=ps_refit]{ps_refit()}} to re-estimate propensity scores after trimming,
\code{\link[=ps_calibrate]{ps_calibrate()}} for calibration-based adjustment,
\code{\link[=ps_trim_meta]{ps_trim_meta()}} to inspect trimming metadata,
\code{\link[=is_ps_trimmed]{is_ps_trimmed()}} and \code{\link[=is_unit_trimmed]{is_unit_trimmed()}} for logical queries.
}
