% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_cissvae.R
\name{run_cissvae}
\alias{run_cissvae}
\title{Run the CISS-VAE pipeline for missing data imputation}
\usage{
run_cissvae(
  data,
  index_col = NULL,
  val_proportion = 0.1,
  replacement_value = 0,
  columns_ignore = NULL,
  imputable_matrix = NULL,
  binary_feature_mask = NULL,
  print_dataset = TRUE,
  clusters = NULL,
  n_clusters = NULL,
  seed = 42,
  missingness_proportion_matrix = NULL,
  scale_features = FALSE,
  k_neighbors = 15L,
  leiden_resolution = 0.5,
  leiden_objective = "CPM",
  hidden_dims = c(150, 120, 60),
  latent_dim = 15,
  layer_order_enc = c("unshared", "unshared", "unshared"),
  layer_order_dec = c("shared", "shared", "shared"),
  latent_shared = FALSE,
  output_shared = FALSE,
  batch_size = 4000,
  epochs = 500,
  initial_lr = 0.01,
  decay_factor = 0.999,
  weight_decay = 0.001,
  beta = 0.001,
  device = NULL,
  max_loops = 100,
  patience = 2,
  epochs_per_loop = NULL,
  initial_lr_refit = NULL,
  decay_factor_refit = NULL,
  beta_refit = NULL,
  verbose = FALSE,
  return_model = TRUE,
  return_clusters = FALSE,
  return_silhouettes = FALSE,
  return_history = FALSE,
  return_dataset = FALSE,
  return_validation_dataset = FALSE,
  debug = FALSE
)
}
\arguments{
\item{data}{A data.frame or matrix (samples × features) containing the data to impute.
May contain \code{NA} values which will be imputed.}

\item{index_col}{Character. Name of column in \code{data} to treat as sample identifier.
This column will be removed before training and re-attached to results. Default \code{NULL}.}

\item{val_proportion}{Numeric. Fraction of non-missing entries to hold out for
validation during training. Must be between 0 and 1. Default \code{0.1}.}

\item{replacement_value}{Numeric. Fill value for masked entries during training.
Default \code{0.0}.}

\item{columns_ignore}{Character or integer vector. Columns to exclude from validation set.
Can specify by name or index. Default \code{NULL}.}

\item{imputable_matrix}{Logical matrix indicating entries allowed to be imputed.}

\item{binary_feature_mask}{Logical vector marking which columns are binary.}

\item{print_dataset}{Logical. If \code{TRUE}, prints dataset summary information during
processing. Default \code{TRUE}.}

\item{clusters}{Optional vector or single-column data.frame of precomputed cluster
labels for samples. If \code{NULL}, clustering will be performed automatically. Default \code{NULL}.}

\item{n_clusters}{Integer. Number of clusters for KMeans clustering when \code{clusters}
is \code{NULL}. Number of clusters for KMeans clustering when 'clusters' is NULL. If \code{NULL},
will use \href{https://leidenalg.readthedocs.io/en/stable/intro.html}{Leiden} for clustering.
Default \code{NULL}.}

\item{seed}{Integer. Random seed for reproducible results. Default \code{42}.}

\item{missingness_proportion_matrix}{Optional pre-computed missingness proportion
matrix for biomarker-based clustering. If provided, clustering will be based on
these proportions. Default \code{NULL}.}

\item{scale_features}{Logical. Whether to scale features when using missingness
proportion matrix clustering. Default \code{FALSE}.}

\item{k_neighbors}{Integer. Number of nearest neighbors for Leiden clustering. Defaults to 15.}

\item{leiden_resolution}{Float. Resolution parameter for Leiden clustering. Defaults to 0.5.}

\item{leiden_objective}{Character. Objective function for Leiden clustering. One of ("CPM", "RB", "Modularity")}

\item{hidden_dims}{Integer vector. Sizes of hidden layers in encoder/decoder.
Length determines number of hidden layers. Default \code{c(150, 120, 60)}.}

\item{latent_dim}{Integer. Dimension of latent space representation. Default \code{15}.}

\item{layer_order_enc}{Character vector. Sharing pattern for encoder layers.
Each element should be "shared" or "unshared". Length must match \code{length(hidden_dims)}.
Default \code{c("unshared", "unshared", "unshared")}.}

\item{layer_order_dec}{Character vector. Sharing pattern for decoder layers.
Each element should be "shared" or "unshared". Length must match \code{length(hidden_dims)}.
Default \code{c("shared", "shared", "shared")}.}

\item{latent_shared}{Logical. Whether latent space weights are shared across clusters.
Default \code{FALSE}.}

\item{output_shared}{Logical. Whether output layer weights are shared across clusters.
Default \code{FALSE}.}

\item{batch_size}{Integer. Mini-batch size for training. Larger values may improve
training stability but require more memory. Default \code{4000}.}

\item{epochs}{Integer. Number of epochs for initial training phase. Default \code{500}.}

\item{initial_lr}{Numeric. Initial learning rate for optimizer. Default \code{0.01}.}

\item{decay_factor}{Numeric. Exponential decay factor for learning rate scheduling.
Must be between 0 and 1. Default \code{0.999}.}

\item{weight_decay}{Weight decay (L2 penalty) used in Adam optimizer.}

\item{beta}{Numeric. Weight for KL divergence term in VAE loss function.
Controls regularization strength. Default \code{0.001}.}

\item{device}{Character. Device specification for computation ("cpu" or "cuda").
If \code{NULL}, automatically selects best available device. Default \code{NULL}.}

\item{max_loops}{Integer. Maximum number of impute-refit loops to perform.
Default \code{100}.}

\item{patience}{Integer. Early stopping patience for refit loops. Training stops
if validation loss doesn't improve for this many consecutive loops. Default \code{2}.}

\item{epochs_per_loop}{Integer. Number of epochs per refit loop. If \code{NULL},
uses same value as \code{epochs}. Default \code{NULL}.}

\item{initial_lr_refit}{Numeric. Learning rate for refit loops. If \code{NULL},
uses same value as \code{initial_lr}. Default \code{NULL}.}

\item{decay_factor_refit}{Numeric. Decay factor for refit loops. If \code{NULL},
uses same value as \code{decay_factor}. Default \code{NULL}.}

\item{beta_refit}{Numeric. KL weight for refit loops. If \code{NULL},
uses same value as \code{beta}. Default \code{NULL}.}

\item{verbose}{Logical. If \code{TRUE}, prints detailed progress information during
training. Default \code{FALSE}.}

\item{return_model}{Logical. If \code{TRUE}, returns the trained Python VAE model object.
Default \code{TRUE}.}

\item{return_clusters}{Logical. If TRUE returns cluster vector}

\item{return_silhouettes}{Logical. If \code{TRUE}, returns silhouette scores for
cluster quality assessment. Default \code{FALSE}.}

\item{return_history}{Logical. If \code{TRUE}, returns training history as a data.frame
containing loss values and metrics over epochs. Default \code{FALSE}.}

\item{return_dataset}{Logical. If \code{TRUE}, returns the ClusterDataset object used
during training (contains validation data, masks, etc.). Default \code{FALSE}.}

\item{return_validation_dataset}{Logical. If \code{TRUE} returns validation dataset}

\item{debug}{Logical; if TRUE, additional metadata is returned for debugging.}
}
\value{
A list containing imputed data and optional additional outputs:
\describe{
\item{imputed_dataset}{data.frame of imputed data with same dimensions as input.
Missing values are filled with model predictions. If \code{index_col} was
provided, it is re-attached as the first column.}
\item{model}{(if \code{return_model=TRUE}) Python CISSVAE model object.
Can be used for further analysis or predictions.}
\item{cluster_dataset}{(if \code{return_dataset=TRUE}) Python ClusterDataset object
containing validation data, masks, normalization parameters, and cluster labels.
Can be used with performance_by_cluster() and other analysis functions.}
\item{clusters}{(if \code{return_clusters=TRUE}) Returns vector of cluster assignments}
\item{silhouettes}{(if \code{return_silhouettes=TRUE}) Numeric silhouette
score measuring cluster separation quality.}
\item{training_history}{(if \code{return_history=TRUE}) data.frame containing training
history with columns for epoch, losses, and validation metrics.}
\item{val_data}{(if \code{return_validation_dataset=TRUE}) data.frame containing values held
aside for validation.}
\item{val_imputed}{(if \code{return_validation_dataset=TRUE}) data.frame containing imputed values of set held
aside for validation.}
}
}
\description{
This function wraps the Python \code{run_cissvae} function from the \code{ciss_vae} package,
providing a complete pipeline for missing data imputation using a Cluster-Informed
Shared and Specific Variational Autoencoder (CISS-VAE). The function handles data
preprocessing, model training, and returns imputed data along with optional
model artifacts.

The CISS-VAE architecture uses cluster information to learn both shared and
cluster-specific representations, enabling more accurate imputation by leveraging
patterns within and across different data subgroups.
}
\details{
The CISS-VAE method works in two main phases:
\enumerate{
\item \strong{Initial Training}: The model is trained on the original data with validation
holdout to learn initial representations and imputation patterns.
\item \strong{Impute-Refit Loops}: The model iteratively imputes missing values and
retrains on the updated dataset until convergence or maximum loops reached.
}

The architecture uses both shared and cluster-specific layers to capture:
\itemize{
\item \strong{Shared patterns}: Common relationships across all clusters
\item \strong{Specific patterns}: Unique relationships within each cluster
}
}
\section{Requirements}{

This function requires the Python \code{ciss_vae} package to be installed and
accessible via \code{reticulate}.
}

\section{Performance tips}{

\itemize{
\item If Leiden clustering yields too many clusters, consider increasing \code{k_neighbors} or reducing \code{leiden_resolution}.
\item Use GPU computation when available for faster training on large datasets. Use \code{check_devices()} to see what devices are available.
\item Adjust \code{batch_size} based on available memory (larger is faster but uses more memory).
\item Set \code{verbose = TRUE} to monitor training progress.
}
}

\examples{
\donttest{
## Requires a working Python environment via reticulate
## Examples are wrapped in try() to avoid failures on CRAN check systems
library(rCISSVAE)

data(df_missing)
data(clusters)

try({
dat = run_cissvae(
 data = df_missing,
 index_col = "index",
 val_proportion = 0.1, ## pass a vector for different proportions by cluster
 columns_ignore = c("Age", "Salary", "ZipCode10001", "ZipCode20002", "ZipCode30003"), 
 clusters = clusters$clusters, ## we have precomputed cluster labels so we pass them here
 epochs = 5,
 return_silhouettes = FALSE,
 return_history = TRUE,  # Get detailed training history
 verbose = FALSE,
 return_model = TRUE, ## Allows for plotting model schematic
 device = "cpu",  # Explicit device selection
 layer_order_enc = c("unshared", "shared", "unshared"),
 layer_order_dec = c("shared", "unshared", "shared")
)
})
}
}
\seealso{
\code{\link{create_missingness_prop_matrix}} for creating missingness proportion matrices
\code{\link{performance_by_cluster}} for analyzing model performance using the returned dataset
}
