% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misc.utilities.R
\name{match_names}
\alias{match_names}
\title{Construct a named vector with semantics useful for parameter vectors}
\usage{
match_names(v, names, default = NULL, partial = TRUE, errname = NULL)
}
\arguments{
\item{v}{a vector}

\item{names}{a character vector of element names}

\item{default}{value to be used for elements of \code{names} not found in \code{v}}

\item{partial}{whether partial matching is allowed}

\item{errname}{optional, name to be reported in any error messages;
defaults to \code{deparse1(substitute(v))}}
}
\value{
A named vector with names \code{names} (in that order). See
Details.
}
\description{
This is a helper function that constructs a named vector with names
in \code{names} with values taken from \code{v} and optionally \code{default},
performing various checks. It supersedes \code{\link[=vector.namesmatch]{vector.namesmatch()}}.
}
\details{
If \code{v} is not named, it is required to be the same length as
\code{names} and is simply given the corresponding names. If it is
named, nonempty names are matched to the corresponding elements of
\code{names}, with partial matching supported.

Default values can be specified by the caller in \code{default} or by
the end-user by adding an element with an empty (\code{""}) name in
addition to the others. If given, the latter overrides the former.

Duplicated names in \code{v} or \code{names} are resolved sequentially,
though note the example below for caveat about partial matching.

Zero-length \code{v} is handled as follows:
\itemize{
\item If length of \code{names} is empty, return \code{v} unchanged.
\item If it is not and \code{default} is not \code{NULL}, return the \code{default} vector.
\item Otherwise, raise an error.
}

An informative error is raised under any of the following conditions:
\itemize{
\item \code{v} is not named but has length that differs from that of \code{names}.
\item More than one element of \code{v} has an empty name.
\item Not all elements of \code{names} are matched by an element of \code{v}, and
no default is specified.
\item Not all elements of \code{v} are used up for elements of \code{names}.
\item There is ambiguity that \code{\link[=pmatch]{pmatch()}} cannot resolve.
}
}
\note{
At this time, passing \code{partial=FALSE} will use a crude
sentinel to prevent partial matching, which in some, extremely
improbable, circumstances might not work.
}
\examples{

# Unnamed:
test <- as.numeric(1:3)
stopifnot(identical(
  match_names(test, c('a', 'c', 'b')),
  c(a = 1, c = 2, b = 3)
))

# Named, reordered:
test <- c(c = 1, b = 2, a = 3)
stopifnot(identical(
  match_names(test, c('a', 'c', 'b')),
  c(a = 3, c = 1, b = 2)
))

# Default value specified by default= assigned to a
test <- c(c = 1, b = 2)
stopifnot(identical(
  match_names(test, c('a', 'c', 'b'), NA),
  c(a = NA, c = 1, b = 2)
))

# Default value specified in v assigned to a and b:
test <- c(c = 1, 2)
stopifnot(identical(
  match_names(test, c('a', 'c', 'b')),
  c(a = 2, c = 1, b = 2)
))

# Partial matching
test <- c(c = 1, 2)
stopifnot(identical(
  match_names(test, c('a', 'cab', 'b')),
  c(a = 2, cab = 1, b = 2)
))

# Multiple matching
test <- c(c = 1, 2, c = 3)
stopifnot(identical(
  match_names(test, c('a', 'c', 'c')),
  c(a = 2, c = 1, c = 3)
))

# Partial + multiple matching caveat: exact match will match first.
test <- c(c = 1, a = 2, ca = 3)
stopifnot(identical(
  match_names(test, c('a', 'ca', 'ca')),
  c(a = 2, ca = 3, ca = 1)
))

}
