% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/round.R
\name{rounding}
\alias{rounding}
\alias{trunc_t}
\alias{floor_t}
\alias{ceiling_t}
\alias{round_t}
\title{Rounding Time Indices}
\usage{
trunc_t(x, unit)

floor_t(x, unit)

ceiling_t(x, unit, ceiling = c("default", "following", "last"))

round_t(x, unit, ceiling = c("default", "following", "last"))
}
\arguments{
\item{x}{an object of \code{tind} class (or an R object coercible to it).}

\item{unit}{a character string determining unit (expected by \code{trunc_t}),
a number, an object of \code{tdiff} class, or a character string
with a number and unit name.}

\item{ceiling}{(optional) a character string determining the behaviour
of \code{ceiling_t} (and \code{round_t}), see Details.}
}
\value{
An object of \code{tind} class of the same type and length as \code{x}
except for \code{trunc_t}, for which the type of the result is determined
based on \code{unit} argument.
}
\description{
Time indices can be rounded to different time units (depending on the type
of time index at hand, see Details).

\code{trunc_t} rounds the indices down to a given \code{unit}
with change of index type where applicable.

\code{floor_t} rounds the indices down to a (multiple of a) unit.

\code{ceiling_t} rounds the indices up to a (multiple of a) unit.

\code{round_t} rounds the indices to the closest multiple of a unit, i.e.
the result of \code{floor_t} or \code{ceiling_t}, whichever is closer.
}
\details{
\strong{Units and Unit Multiples}

For \code{trunc_t}, \code{unit} has to be a character string determining
resolution / type to which indices should be truncated. For the remaining
functions, \code{unit} argument can be a number (the default unit for index
type will be used), a character string with unit name, an object
of \code{tdiff} class, or a character string with a number and unit name.

The following unit names are accepted:
\describe{
  \item{\code{"y"}, \code{"year"}, \code{"years"}}{years,}
  \item{\code{"q"}, \code{"quarter"}, \code{"quarters"}}{quarters,}
  \item{\code{"m"}, \code{"mon"}, \code{"month"}, \code{"months"}}{months,}
  \item{\code{"w"}, \code{"week"}, \code{"weeks"}}{weeks,}
  \item{\code{"d"}, \code{"day"}, \code{"days"}}{days,}
  \item{\code{"h"}, \code{"hour"}, \code{"hours"}}{hours,}
  \item{\code{"min"}, \code{"minute"}, \code{"minutes"}}{minutes,}
  \item{\code{"s"}, \code{"sec"}, \code{"second"}, \code{"seconds"}}{seconds.}
}
The default unit for date-time and time of day indices is a second.

The list of admissible multiples of units can be found in the documentation
of \code{\link{resolution_t}} method.

For time indices of types \code{"i"} and \code{"n"} (integer and numeric indices)
unit can be any finite, positive integer / numeric value.

In case of a tie (\code{x - floor_t(x, *)} equal to \code{ceiling_t(x, *) - x}),
\code{round_t} returns the result of \code{ceiling_t}.

\strong{Controlling behaviour of \code{ceiling_t} (and \code{round_t})}

For non-instant time indices, i.e indices that actually represent periods of time
(days weeks, months, etc.) \code{ceiling_t} rounds to the first index in a period
by default. For instance, when rounding dates to months, the first day of a month
will be unchanged and other days will be rounded to the first day in the
\emph{following} month. This behaviour can be altered by setting \code{ceiling}
argument. If set to \code{"following"}, the first index (day in our example)
in the following period will be returned. If set to \code{"last"}, the last
index (day in our example) in the period will be returned. See Examples.
}
\note{
Methods \code{floor}, \code{ceiling}, \code{round}, and \code{trunc} are
not implemented for \code{tind} class due to generics' argument list
limitations.
}
\examples{
(d <- as.tind("2024-08-27"))
floor_t(d, "w")
trunc_t(d, "w")
ceiling_t(d, "w")
round_t(d, "w")
floor_t(d, "m")
trunc_t(d, "m")
ceiling_t(d, "m")
round_t(d, "m")
floor_t(d, "3m")
ceiling_t(d, "3m")
round_t(d, "3m")

(dt <- as.tind("2024-08-27 13:51:52"))
floor_t(dt, 10)
floor_t(dt, "10s")
ceiling_t(dt, "10s")
round_t(dt, "10s")
floor_t(dt, "min")
trunc_t(dt, "min")
ceiling_t(dt, "min")
round_t(dt, "min")
floor_t(dt, "h")
trunc_t(dt, "h")
ceiling_t(dt, "h")
round_t(dt, "h")
floor_t(dt, "2h")
ceiling_t(dt, "2h")
round_t(dt, "2h")
floor_t(dt, "d")
trunc_t(dt, "d")
ceiling_t(dt, "d")
round_t(dt, "d")

# corner cases - DST change (02:00 missing)
(dt <- date_time("2025-03-30", H = c(0:1, 3:6)))
floor_t(dt, "2h")
ceiling_t(dt, "2h")

# adjusting behaviour of ceiling_t for non-instant time indices
# a short sequence of dates covering two months
(ds <- as.tind("2023-01-01") + -2:2)
# default behaviour
ceiling_t(ds, "2m")
ceiling_t(ds, "2m", ceiling = "default")
# round up to the first day in the following 2-month period
ceiling_t(ds, "2m", ceiling = "following")
# round up to the last day in the current 2-month period
ceiling_t(ds, "2m", ceiling = "last")
# a corner case, note that we will get the next day as a result
ceiling_t(today(), "1d", ceiling = "following")

}
\seealso{
\code{\link{resolution_t}} method.
}
