% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/algorithms.R
\name{grid_search}
\alias{grid_search}
\title{Carry out the grid search algorithm with a zoom.}
\usage{
grid_search(
  FUN,
  grid,
  MoreArgs = NULL,
  zoom = 0,
  decay = 0.5,
  num = 1,
  parallel = FALSE,
  cores = NULL,
  silent = TRUE
)
}
\arguments{
\item{FUN}{the target function to be minimized.}

\item{grid}{an object of class \code{GRID} created by \code{\link{build_grid}}.}

\item{MoreArgs}{a named list of additional arguments to \code{FUN}, see \code{\link{mapply}}.}

\item{zoom}{number of (additional) zoom-in layers, \code{0} by default.}

\item{decay}{a number in \eqn{(0,1)} controlling the decay of subgrid sizes.}

\item{num}{number of points to return at each grid search, \code{1} by default.}

\item{parallel}{a logical; if \code{TRUE}, parallel computation is used.}

\item{cores}{an integer specifying the requested number of workers when \code{parallel = TRUE}.
If \code{NULL}, the function uses \code{2} workers by default (subject to \code{future::availableCores()}).}

\item{silent}{a logical indicating whether progress information is printed.}
}
\value{
a list with components:
\item{par}{the approximate global minimizer}
\item{points}{all candidate points found by the grid search with zoom-in layers}
}
\description{
This function carries out the grid search algorithm with a zoom.
}
\details{
The target function \code{FUN} to be minimized is a scalar real-valued function.
Maximization can be achieved by multiplying \code{-1} to the original function
and then passing the new function to \code{FUN}.

The \code{grid} must be created by \code{\link{build_grid}}.

Any other invariant arguments to \code{FUN} can be specified in \code{MoreArgs}
using a named list, see \code{\link{mapply}}.

The common grid search first builds a grid within a bounded region, evaluates \code{FUN}
at each grid point, and returns the \code{num} points that yield the smallest values.

\code{zoom = 0} implies no zoom-in (a single grid search). Any integer \code{zoom > 0}
applies additional zoom-in layers. With \code{zoom > 0}, the algorithm performs
\deqn{ n^{0} + n^{1} + n^{2} + \cdots + n^{z} }
grid searches, where \eqn{n} is \code{num} and \eqn{z} is \code{zoom}.
Consequently, the total number of returned points is
\deqn{ n^{1} + n^{2} + n^{3} + \cdots + n^{z+1} }.

At each zoom-in layer, the algorithm builds subgrids around the best points found
in the previous layer. To limit the computational burden, the subgrid size is reduced
by the decay rate \code{decay}. For each parameter, the number of points in the subgrid is
\code{max(Int(decay * N), 3)}, where \code{N} is the number of points in the original grid
for that parameter.

Parallel computation can be enabled by setting \code{parallel = TRUE}. In that case,
the function uses the \pkg{future} framework with \code{future::multisession}
(cross-platform). The number of workers is determined as follows:
\enumerate{
  \item Let \eqn{n} be the value returned by \code{future::availableCores()}.
  \item Let \eqn{m} be the user input \code{cores}. If \code{cores = NULL}, set \eqn{m = 2}.
  \item The number of workers is \eqn{\min(m, n)}.
}
If \code{parallel = TRUE}, the packages \pkg{future} and \pkg{future.apply} must be installed.

The boolean \code{silent} controls whether progress information is printed to the console.
}
\examples{
# Rastrigin function
ndim = 2
nA = 10
Rastrigin <- function(vx) nA * ndim + sum(vx * vx - nA * cos(2 * pi * vx))

# Build a grid
bin = c(from = -5.12, to = 5.12, by = .5)
grid = build_grid(bin, bin)

# Serial computation
ret0 = grid_search(Rastrigin, grid, silent = FALSE)
ret0$par

\donttest{
# Finer grid
bin = c(from = -5.12, to = 5.12, by = .1)
grid = build_grid(bin, bin)

# Serial computation
ret1 = grid_search(Rastrigin, grid, silent = FALSE)
ret1$par

# Parallel computation (requires future and future.apply)
ret2 = grid_search(Rastrigin, grid, num = 2, parallel = TRUE, cores = 2, silent = FALSE)
ret2$par

# Grid search with zoom-in layers
ret3 = grid_search(Rastrigin, grid, zoom = 2, num = 2, parallel = TRUE, cores = 2, silent = FALSE)
ret3$par
}

}
\seealso{
\code{\link{build_grid}}, \code{\link{grid_search_check}}
}
\author{
Yukai Yang, \email{yukai.yang@statistik.uu.se}
}
\keyword{algorithms}
