/*
 * Decompiled with CFR 0.152.
 */
package generic.expressions;

import generic.expressions.ExpressionElement;
import generic.expressions.ExpressionException;
import generic.expressions.ExpressionGrouper;
import generic.expressions.ExpressionOperator;
import generic.expressions.ExpressionValue;
import generic.expressions.LongExpressionValue;
import ghidra.util.NumericUtilities;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.function.Function;
import java.util.stream.Collectors;

public class ExpressionEvaluator {
    private static final String TOKEN_CHARS = "+-*/()<>|^&~ =!";
    private boolean assumeHex = false;
    private Function<String, ExpressionValue> evaluator;

    public static Long evaluateToLong(String input) {
        return ExpressionEvaluator.evaluateToLong(input, false);
    }

    public static Long evaluateToLong(String input, boolean assumeHex) {
        ExpressionEvaluator evaluator = new ExpressionEvaluator(assumeHex);
        try {
            return evaluator.parseAsLong(input);
        }
        catch (ExpressionException e) {
            return null;
        }
    }

    public ExpressionEvaluator() {
        this(false);
    }

    public ExpressionEvaluator(boolean assumeHex) {
        this(assumeHex, s -> null);
    }

    public ExpressionEvaluator(Function<String, ExpressionValue> evaluator) {
        this(false, evaluator);
    }

    public ExpressionEvaluator(boolean assumeHex, Function<String, ExpressionValue> evaluator) {
        this.assumeHex = assumeHex;
        this.evaluator = Objects.requireNonNull(evaluator);
    }

    public long parseAsLong(String input) throws ExpressionException {
        ExpressionValue expressionValue = this.parse(input);
        if (expressionValue instanceof LongExpressionValue) {
            LongExpressionValue longValue = (LongExpressionValue)expressionValue;
            return longValue.getLongValue();
        }
        throw new ExpressionException("Expression did not evalute to a long! Got a " + String.valueOf(expressionValue.getClass()) + " instead.");
    }

    public void setAssumeHex(boolean b) {
        this.assumeHex = b;
    }

    protected ExpressionValue parse(String input) throws ExpressionException {
        return this.parse(input, null);
    }

    protected ExpressionValue parse(String input, ExpressionValue initial) throws ExpressionException {
        ArrayList<ExpressionElement> list = new ArrayList<ExpressionElement>();
        if (initial != null) {
            list.add(initial);
        }
        this.parseToList(input, list);
        return this.eval(list);
    }

    private void parseToList(String input, List<ExpressionElement> list) throws ExpressionException {
        LookAheadTokenizer parser = new LookAheadTokenizer(this, input);
        while (parser.hasMoreTokens()) {
            String token = parser.getCurrentToken();
            if (token.isBlank()) {
                parser.advance(1);
                continue;
            }
            if (this.processGroupToken(list, token)) {
                parser.advance(1);
                continue;
            }
            if (this.processOperator(list, token, parser.getNextToken())) {
                ExpressionOperator op = this.getLastOperator(list);
                parser.advance(op.size());
                continue;
            }
            if (this.processNumber(list, token)) {
                parser.advance(1);
                continue;
            }
            if (this.processSymbol(list, token)) {
                parser.advance(1);
                continue;
            }
            throw new ExpressionException("Could not evaluate token \"" + token + "\"");
        }
        if (list.isEmpty()) {
            throw new ExpressionException("Expression is empty. Nothing to parse!");
        }
    }

    private ExpressionValue eval(List<ExpressionElement> list) throws ExpressionException {
        this.processGroups(list);
        this.processUnaryOperators(list);
        this.processBinaryOperators(list);
        if (list.size() != 1) {
            String result = list.stream().map(Object::toString).collect(Collectors.joining(" "));
            throw new ExpressionException("Parse failed! Stopped at \"" + result + "\"");
        }
        ExpressionElement element = list.get(0);
        if (element instanceof ExpressionValue) {
            ExpressionValue ev = (ExpressionValue)element;
            return ev;
        }
        throw new ExpressionException("Parse failed to evaluate to a value! Stopped at " + String.valueOf(element));
    }

    private void processBinaryOperators(List<ExpressionElement> list) throws ExpressionException {
        List<Set<ExpressionOperator>> ops = ExpressionOperator.getBinaryOperatorsByPrecedence();
        for (Set<ExpressionOperator> set : ops) {
            this.processBinaryOperators(list, set);
        }
    }

    private void processBinaryOperators(List<ExpressionElement> list, Set<ExpressionOperator> operators) throws ExpressionException {
        int operatorIndex = this.findValidBinaryOperator(list, operators, 1);
        while (operatorIndex >= 0) {
            ExpressionOperator operator = (ExpressionOperator)list.get(operatorIndex);
            ExpressionValue value1 = (ExpressionValue)list.get(operatorIndex - 1);
            ExpressionValue value2 = (ExpressionValue)list.get(operatorIndex + 1);
            ExpressionValue newValue = value1.applyBinaryOperator(operator, value2);
            list.set(operatorIndex - 1, newValue);
            list.subList(operatorIndex, operatorIndex + 2).clear();
            operatorIndex = this.findValidBinaryOperator(list, operators, operatorIndex);
        }
    }

    private int findValidBinaryOperator(List<ExpressionElement> list, Set<ExpressionOperator> operators, int startIndex) {
        for (int i = startIndex; i < list.size() - 1; ++i) {
            if (!operators.contains(list.get(i)) || !(list.get(i - 1) instanceof ExpressionValue & list.get(i + 1) instanceof ExpressionValue)) continue;
            return i;
        }
        return -1;
    }

    private void processUnaryOperators(List<ExpressionElement> list) throws ExpressionException {
        int unaryOperatorIndex = this.findValidUnaryOperator(list);
        while (unaryOperatorIndex >= 0) {
            ExpressionOperator operator = (ExpressionOperator)list.get(unaryOperatorIndex);
            ExpressionValue value = (ExpressionValue)list.get(unaryOperatorIndex + 1);
            ExpressionValue newValue = value.applyUnaryOperator(operator);
            list.remove(unaryOperatorIndex);
            list.set(unaryOperatorIndex, newValue);
            unaryOperatorIndex = this.findValidUnaryOperator(list);
        }
    }

    private int findValidUnaryOperator(List<ExpressionElement> list) {
        for (int i = 0; i < list.size() - 1; ++i) {
            ExpressionOperator op;
            ExpressionElement expressionElement = list.get(i);
            if (!(expressionElement instanceof ExpressionOperator) || !(op = (ExpressionOperator)expressionElement).isUnary() || !(list.get(i + 1) instanceof ExpressionValue)) continue;
            return i;
        }
        return -1;
    }

    private void processGroups(List<ExpressionElement> list) throws ExpressionException {
        int groupStart = this.findGroupStart(list);
        while (groupStart >= 0) {
            int groupEndIndex = this.findGroupEnd(list, groupStart);
            if (groupEndIndex < 0) {
                throw new ExpressionException("Missing end parenthesis!");
            }
            ExpressionValue value = this.eval(list.subList(groupStart + 1, groupEndIndex));
            list.set(groupStart, value);
            list.subList(groupStart + 1, groupStart + 3).clear();
            groupStart = this.findGroupStart(list);
        }
    }

    private int findGroupStart(List<ExpressionElement> list) {
        for (int i = 0; i < list.size(); ++i) {
            if (list.get(i) != ExpressionGrouper.LEFT_PAREN) continue;
            return i;
        }
        return -1;
    }

    private int findGroupEnd(List<ExpressionElement> list, int groupStart) {
        int depth = 1;
        for (int i = groupStart + 1; i < list.size(); ++i) {
            ExpressionElement obj = list.get(i);
            if (obj == ExpressionGrouper.LEFT_PAREN) {
                ++depth;
                continue;
            }
            if (obj != ExpressionGrouper.RIGHT_PAREN || --depth != 0) continue;
            return i;
        }
        return -1;
    }

    private ExpressionOperator getLastOperator(List<ExpressionElement> list) {
        ExpressionElement lastElement = list.get(list.size() - 1);
        return (ExpressionOperator)lastElement;
    }

    private boolean processSymbol(List<ExpressionElement> list, String token) {
        ExpressionValue value = this.evaluateSymbol(token);
        if (value != null) {
            list.add(value);
            return true;
        }
        return false;
    }

    protected ExpressionValue evaluateSymbol(String token) {
        return this.evaluator.apply(token);
    }

    private boolean processOperator(List<ExpressionElement> list, String token, String nextToken) {
        boolean preferBinary = this.shouldPreferBinaryOp(list);
        ExpressionOperator op = ExpressionOperator.getOperator(token, nextToken, preferBinary);
        if (op != null) {
            list.add(op);
            return true;
        }
        return false;
    }

    private boolean processGroupToken(List<ExpressionElement> list, String token) {
        if (token.equals("(")) {
            list.add(ExpressionGrouper.LEFT_PAREN);
            return true;
        }
        if (token.equals(")")) {
            list.add(ExpressionGrouper.RIGHT_PAREN);
            return true;
        }
        return false;
    }

    private boolean shouldPreferBinaryOp(List<ExpressionElement> list) {
        if (list.isEmpty()) {
            return false;
        }
        ExpressionElement lastElement = list.get(list.size() - 1);
        if (lastElement instanceof ExpressionValue) {
            return true;
        }
        if (lastElement instanceof ExpressionOperator) {
            return false;
        }
        if (lastElement == ExpressionGrouper.LEFT_PAREN) {
            return false;
        }
        return lastElement == ExpressionGrouper.RIGHT_PAREN;
    }

    private boolean processNumber(List<ExpressionElement> list, String token) {
        int radix = 10;
        if (this.assumeHex && this.processAsHexNumber(list, token)) {
            return true;
        }
        if ((token = this.toLowerAndRemoveEndNumberDecorators(token)).startsWith("0x")) {
            radix = 16;
            token = token.substring(2);
        }
        try {
            long value = radix == 10 ? NumericUtilities.parseLong(token) : NumericUtilities.parseHexLong(token);
            list.add(new LongExpressionValue(value));
            return true;
        }
        catch (Exception e) {
            return false;
        }
    }

    private String toLowerAndRemoveEndNumberDecorators(String token) {
        if ((token = token.toLowerCase()).endsWith("ull") || token.endsWith("llu")) {
            token = token.substring(0, token.length() - 3);
        } else if (token.endsWith("ul") || token.endsWith("lu") || token.endsWith("ll")) {
            token = token.substring(0, token.length() - 2);
        } else if (token.endsWith("l") || token.endsWith("u")) {
            token = token.substring(0, token.length() - 1);
        }
        return token;
    }

    private boolean processAsHexNumber(List<ExpressionElement> list, String token) {
        try {
            long value = NumericUtilities.parseHexLong(token);
            list.add(new LongExpressionValue(value));
            return true;
        }
        catch (NumberFormatException numberFormatException) {
            return false;
        }
    }

    private class LookAheadTokenizer {
        private StringTokenizer tokenizer;
        private String currentToken;
        private String nextToken;

        LookAheadTokenizer(ExpressionEvaluator expressionEvaluator, String input) {
            this.tokenizer = new StringTokenizer(input, ExpressionEvaluator.TOKEN_CHARS, true);
            this.currentToken = this.tokenizer.hasMoreTokens() ? this.tokenizer.nextToken() : null;
            this.nextToken = this.tokenizer.hasMoreTokens() ? this.tokenizer.nextToken() : null;
        }

        public boolean hasMoreTokens() {
            return this.currentToken != null;
        }

        public String getCurrentToken() {
            return this.currentToken;
        }

        public String getNextToken() {
            return this.nextToken;
        }

        public void advance(int count) {
            for (int i = 0; i < count; ++i) {
                this.currentToken = this.nextToken;
                this.nextToken = this.tokenizer.hasMoreTokens() ? this.tokenizer.nextToken() : null;
            }
        }
    }
}

