/*
 * Decompiled with CFR 0.152.
 */
package ghidra.app.plugin.core.function.editor;

import ghidra.app.plugin.core.function.editor.ModelChangeListener;
import ghidra.app.plugin.core.function.editor.VarnodeInfo;
import ghidra.app.plugin.core.function.editor.VarnodeType;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressOverflowException;
import ghidra.program.model.address.AddressRangeImpl;
import ghidra.program.model.lang.ProgramArchitecture;
import ghidra.program.model.lang.Register;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.VariableStorage;
import ghidra.program.model.pcode.Varnode;
import ghidra.util.SystemUtilities;
import ghidra.util.exception.AssertException;
import ghidra.util.exception.InvalidInputException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.swing.SwingUtilities;

class StorageAddressModel {
    private List<VarnodeInfo> varnodes = new ArrayList<VarnodeInfo>();
    private int requiredSize;
    private boolean unconstrained;
    private int[] selectedVarnodeRows = new int[0];
    private ModelChangeListener listener;
    private String statusText;
    private boolean isValid;
    private Program program;

    StorageAddressModel(Program program, VariableStorage storage, ModelChangeListener listener) {
        this.listener = listener;
        this.program = program;
        if (storage != null) {
            for (Varnode varnode : storage.getVarnodes()) {
                this.varnodes.add(new VarnodeInfo(program, varnode));
            }
        }
        this.validate();
    }

    void addVarnode() {
        this.varnodes.add(new VarnodeInfo(this.program, VarnodeType.Register));
        this.listener.tableRowsChanged();
        this.setSelectedRow(this.varnodes.size() - 1);
        this.notifyDataChanged();
    }

    void removeVarnodes() {
        if (!this.canRemoveVarnodes()) {
            throw new AssertException("Attempted to remove varnodes when not allowed.");
        }
        this.listener.tableRowsChanged();
        Arrays.sort(this.selectedVarnodeRows);
        for (int i = this.selectedVarnodeRows.length - 1; i >= 0; --i) {
            int index = this.selectedVarnodeRows[i];
            this.varnodes.remove(index);
        }
        if (this.varnodes.isEmpty()) {
            this.selectedVarnodeRows = new int[0];
        } else {
            int selectRow = Math.min(this.selectedVarnodeRows[0], this.varnodes.size() - 1);
            this.selectedVarnodeRows = new int[]{selectRow};
        }
        this.notifyDataChanged();
    }

    void moveSelectedVarnodeUp() {
        if (!this.canMoveVarnodeUp()) {
            throw new AssertException("Attempted to move a varnode up when not allowed.");
        }
        this.listener.tableRowsChanged();
        int index = this.selectedVarnodeRows[0];
        VarnodeInfo info = this.varnodes.remove(index);
        this.varnodes.add(index - 1, info);
        this.setSelectedRow(index - 1);
        this.notifyDataChanged();
    }

    void moveSelectedVarnodeDown() {
        if (!this.canMoveVarnodeDown()) {
            throw new AssertException("Attempted to move a parameter down when not allowed.");
        }
        this.listener.tableRowsChanged();
        int index = this.selectedVarnodeRows[0];
        VarnodeInfo info = this.varnodes.remove(index);
        this.varnodes.add(index + 1, info);
        this.setSelectedRow(index + 1);
        this.notifyDataChanged();
    }

    List<VarnodeInfo> getVarnodes() {
        return this.varnodes;
    }

    void setRequiredSize(int requiredSize, boolean unconstrained) {
        this.requiredSize = requiredSize;
        this.unconstrained = unconstrained;
        this.validate();
    }

    int getRequiredSize() {
        return this.requiredSize;
    }

    boolean isUnconstrained() {
        return this.unconstrained;
    }

    int getCurrentSize() {
        int size = 0;
        for (VarnodeInfo varnode : this.varnodes) {
            if (varnode.getSize() == null) continue;
            size += varnode.getSize().intValue();
        }
        return size;
    }

    String getStatusText() {
        return this.statusText;
    }

    boolean isValid() {
        return this.isValid;
    }

    void setSelectedVarnodeRows(int[] selectedRows) {
        this.selectedVarnodeRows = selectedRows;
    }

    private void setSelectedRow(int row) {
        this.selectedVarnodeRows = new int[]{row};
    }

    int[] getSelectedVarnodeRows() {
        return this.selectedVarnodeRows;
    }

    void notifyDataChanged() {
        this.validate();
        SwingUtilities.invokeLater(() -> this.listener.dataChanged());
    }

    private void validate() {
        this.statusText = "";
        this.isValid = this.hasValidVarnodes() && this.hasCorrectAllocatedSize();
    }

    private boolean hasCorrectAllocatedSize() {
        int currentSize = this.getCurrentSize();
        if (currentSize == 0) {
            this.statusText = "No storage has been allocated";
        } else {
            if (currentSize > 0 && this.unconstrained) {
                return true;
            }
            if (currentSize < this.requiredSize) {
                this.statusText = "Warning: Not enough storage space allocated";
                return false;
            }
            if (currentSize > this.requiredSize) {
                this.statusText = "Warning: Too much storage space allocated";
            }
        }
        return true;
    }

    private boolean hasValidVarnodes() {
        for (int i = 0; i < this.varnodes.size(); ++i) {
            VarnodeInfo varnode = this.varnodes.get(i);
            if (varnode.getAddress() == null) {
                this.statusText = "Row " + i + ": Storage not specified";
                return false;
            }
            if (this.isValidSize(varnode, i) || !this.isValidAddress(varnode, i)) continue;
            return false;
        }
        try {
            this.buildStorage();
            return true;
        }
        catch (InvalidInputException e) {
            this.statusText = e.getMessage();
            return false;
        }
    }

    private boolean isValidSize(VarnodeInfo varnode, int row) {
        Integer size = varnode.getSize();
        if (size == null) {
            this.statusText = "Row " + row + ": No size specified";
            return false;
        }
        if (size <= 0) {
            this.statusText = "Row " + row + ": Size must be > 0";
            return false;
        }
        return true;
    }

    private boolean isValidAddress(VarnodeInfo varnode, int row) {
        Address address = varnode.getAddress();
        if (address == null) {
            this.statusText = "Row " + row + ": Invalid Varnode Address";
            return false;
        }
        try {
            new AddressRangeImpl(address, (long)varnode.getSize().intValue());
        }
        catch (AddressOverflowException e) {
            this.statusText = "Row " + row + ": Varnode wraps within " + address.getAddressSpace().getName() + " space.";
            return false;
        }
        if (address.isStackAddress()) {
            long stackOffset = address.getOffset();
            if (stackOffset < 0L && -stackOffset < (long)varnode.getSize().intValue()) {
                this.statusText = "Row " + row + ": Stack varnode spans 0 offset";
                return false;
            }
            return true;
        }
        Register register = this.program.getRegister(address, varnode.getSize().intValue());
        if (register == null) {
            Register register2 = this.program.getRegister(address);
            if (register2 != null) {
                this.statusText = "Row " + row + ": Register (size=" + register2.getBitLength() / 8 + ") too small for specified size(" + varnode.getSize() + ")";
            } else if (address.isRegisterAddress()) {
                this.statusText = "Row " + row + ": Invalid Register";
                return false;
            }
            return address.isMemoryAddress();
        }
        return true;
    }

    boolean canRemoveVarnodes() {
        return this.selectedVarnodeRows.length > 0;
    }

    boolean canMoveVarnodeUp() {
        return this.selectedVarnodeRows.length == 1 && this.selectedVarnodeRows[0] > 0;
    }

    boolean canMoveVarnodeDown() {
        return this.selectedVarnodeRows.length == 1 && this.selectedVarnodeRows[0] < this.varnodes.size() - 1;
    }

    void setVarnodeType(VarnodeInfo varnode, VarnodeType type) {
        if (type == varnode.getType()) {
            return;
        }
        varnode.setVarnodeType(type);
        this.notifyDataChanged();
    }

    void setVarnode(VarnodeInfo info, String registerName) {
        Register register = this.program.getRegister(registerName);
        if (register != null) {
            this.setVarnode(info, register);
        }
    }

    void setVarnode(VarnodeInfo info, Register reg) {
        int size;
        Address addr = reg.getAddress();
        int currentSize = this.getCurrentSize();
        int regSize = reg.getBitLength() / 8;
        if (this.unconstrained) {
            this.setVarnode(info, addr, regSize);
            return;
        }
        Integer curVarnodeSize = info.getSize();
        if (reg.hasChildren() || curVarnodeSize == null || regSize < curVarnodeSize || currentSize < this.requiredSize) {
            size = reg.getBitLength() / 8;
            if (curVarnodeSize != null) {
                currentSize -= curVarnodeSize.intValue();
            }
            if (currentSize < this.requiredSize) {
                size = Math.min(size, this.requiredSize - currentSize);
            }
        } else {
            size = curVarnodeSize;
        }
        if (reg.isBigEndian()) {
            int s = Math.min(reg.getMinimumByteSize(), size);
            addr = addr.add((long)(reg.getMinimumByteSize() - s));
        }
        this.setVarnode(info, addr, size);
    }

    void setVarnode(VarnodeInfo info, Address address, Integer size) {
        if (SystemUtilities.isEqual((Object)info.getAddress(), (Object)address) && SystemUtilities.isEqual((Object)info.getSize(), (Object)size)) {
            return;
        }
        info.setVarnode(address, size);
        this.notifyDataChanged();
    }

    Program getProgram() {
        return this.program;
    }

    private VariableStorage buildStorage() throws InvalidInputException {
        if (this.varnodes.size() == 0) {
            if (this.requiredSize == 0) {
                return VariableStorage.VOID_STORAGE;
            }
            return VariableStorage.UNASSIGNED_STORAGE;
        }
        ArrayList<Varnode> varnodeList = new ArrayList<Varnode>(this.varnodes.size());
        for (VarnodeInfo varnodeInfo : this.varnodes) {
            Address addr = varnodeInfo.getAddress();
            if (addr == null) {
                return VariableStorage.UNASSIGNED_STORAGE;
            }
            varnodeList.add(new Varnode(addr, varnodeInfo.getSize().intValue()));
        }
        return new VariableStorage((ProgramArchitecture)this.program, varnodeList);
    }

    VariableStorage getStorage() {
        if (!this.isValid) {
            return null;
        }
        try {
            return this.buildStorage();
        }
        catch (InvalidInputException e) {
            throw new AssertException("Unexpected exception", (Throwable)e);
        }
    }
}

