"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateCustomError = generateCustomError;
exports.getAgentsFromResponses = getAgentsFromResponses;
exports.getConnectorsFromResponses = getConnectorsFromResponses;
exports.getModelsFromResponses = getModelsFromResponses;
exports.getResourcesCreatedFromResponse = getResourcesCreatedFromResponse;
exports.getWorkflowStateFromResponse = getWorkflowStateFromResponse;
exports.getWorkflowsFromResponses = getWorkflowsFromResponses;
exports.isIgnorableError = isIgnorableError;
exports.toAgentObj = toAgentObj;
exports.toWorkflowObj = toWorkflowObj;
var _common = require("../../common");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// OSD does not provide an interface for this response, but this is following the suggested
// implementations. To prevent typescript complaining, leaving as loosely-typed 'any'
function generateCustomError(res, err) {
  var _err$body;
  if (isDatasourceError(err)) {
    return res.customError({
      statusCode: 404,
      body: {
        message: 'Data source not found'
      }
    });
  }
  return res.customError({
    statusCode: err.statusCode || 500,
    body: {
      message: err.message,
      attributes: {
        error: ((_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.error) || err.message
      }
    }
  });
}
function isDatasourceError(err) {
  if (err.message !== undefined && typeof err.message === 'string') {
    return err.message.includes(_common.INVALID_DATASOURCE_MSG);
  }
}

// Helper fn to filter out backend errors that we don't want to propagate on the frontend.
function isIgnorableError(error) {
  var _error$body, _error$body2, _error$body3;
  return ((_error$body = error.body) === null || _error$body === void 0 || (_error$body = _error$body.error) === null || _error$body === void 0 ? void 0 : _error$body.type) === _common.INDEX_NOT_FOUND_EXCEPTION || ((_error$body2 = error.body) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.error) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.caused_by) === null || _error$body2 === void 0 ? void 0 : _error$body2.type) === _common.INDEX_NOT_FOUND_EXCEPTION || ((_error$body3 = error.body) === null || _error$body3 === void 0 ? void 0 : _error$body3.error) === _common.NO_MODIFICATIONS_FOUND_TEXT;
}

// Convert backend workflow into frontend workflow obj
function toWorkflowObj(hitSource, id) {
  return {
    id,
    name: hitSource.name,
    use_case: hitSource.use_case,
    description: hitSource.description || '',
    version: hitSource.version,
    workflows: hitSource.workflows,
    ui_metadata: hitSource.ui_metadata,
    lastUpdated: hitSource.last_updated_time,
    lastLaunched: hitSource.last_provisioned_time
  };
}

// TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
// https://github.com/opensearch-project/flow-framework/issues/171
// Current implementation combines 2 search responses to create a single set of workflows with
// static information + state information
function getWorkflowsFromResponses(workflowHits, workflowStateHits) {
  const workflowDict = {};
  workflowHits.forEach(workflowHit => {
    var _workflowStateHit$_so, _workflowStateHit$_so2, _workflowStateHit$_so3;
    const hitSource = workflowHit._source;
    workflowDict[workflowHit._id] = toWorkflowObj(hitSource, workflowHit._id);
    const workflowStateHit = workflowStateHits.find(workflowStateHit => workflowStateHit._id === workflowHit._id);
    const workflowState = getWorkflowStateFromResponse(workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so = workflowStateHit._source) === null || _workflowStateHit$_so === void 0 ? void 0 : _workflowStateHit$_so.state);
    const workflowError = workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so2 = workflowStateHit._source) === null || _workflowStateHit$_so2 === void 0 ? void 0 : _workflowStateHit$_so2.error;
    const workflowResourcesCreated = getResourcesCreatedFromResponse(workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so3 = workflowStateHit._source) === null || _workflowStateHit$_so3 === void 0 ? void 0 : _workflowStateHit$_so3.resources_created);
    workflowDict[workflowHit._id] = {
      ...workflowDict[workflowHit._id],
      // @ts-ignore
      state: workflowState,
      error: workflowError,
      resourcesCreated: workflowResourcesCreated
    };
  });
  return workflowDict;
}
function getModelsFromResponses(modelHits) {
  const modelDict = {};
  modelHits.forEach(modelHit => {
    // search model API returns hits for each deployed model chunk. ignore these hits
    if (modelHit._source.chunk_number === undefined) {
      var _modelHit$_source, _modelHit$_source2, _modelHit$_source3, _modelHit$_source4, _modelHit$_source5, _modelHit$_source6, _modelHit$_source7;
      const modelId = modelHit._id;

      // the persisted model interface (if available) is a mix of an obj and string.
      // We parse the string values for input/output to have a complete
      // end-to-end JSONSchema obj
      let indexedModelInterface = modelHit._source.interface;
      let modelInterface = undefined;
      if (indexedModelInterface !== undefined) {
        let parsedInput = undefined;
        let parsedOutput = undefined;
        try {
          parsedInput = JSON.parse(indexedModelInterface.input);
        } catch {}
        try {
          parsedOutput = JSON.parse(indexedModelInterface.output);
        } catch {}
        modelInterface = {
          input: parsedInput,
          output: parsedOutput
        };
      }

      // in case of schema changes from ML plugin, this may crash. That is ok, as the error
      // produced will help expose the root cause
      modelDict[modelId] = {
        id: modelId,
        name: (_modelHit$_source = modelHit._source) === null || _modelHit$_source === void 0 ? void 0 : _modelHit$_source.name,
        // @ts-ignore
        algorithm: _common.MODEL_ALGORITHM[(_modelHit$_source2 = modelHit._source) === null || _modelHit$_source2 === void 0 ? void 0 : _modelHit$_source2.algorithm],
        // @ts-ignore
        state: _common.MODEL_STATE[(_modelHit$_source3 = modelHit._source) === null || _modelHit$_source3 === void 0 ? void 0 : _modelHit$_source3.model_state],
        modelConfig: {
          modelType: (_modelHit$_source4 = modelHit._source) === null || _modelHit$_source4 === void 0 || (_modelHit$_source4 = _modelHit$_source4.model_config) === null || _modelHit$_source4 === void 0 ? void 0 : _modelHit$_source4.model_type,
          embeddingDimension: (_modelHit$_source5 = modelHit._source) === null || _modelHit$_source5 === void 0 || (_modelHit$_source5 = _modelHit$_source5.model_config) === null || _modelHit$_source5 === void 0 ? void 0 : _modelHit$_source5.embedding_dimension
        },
        interface: modelInterface,
        connectorId: (_modelHit$_source6 = modelHit._source) === null || _modelHit$_source6 === void 0 ? void 0 : _modelHit$_source6.connector_id,
        connector: (_modelHit$_source7 = modelHit._source) === null || _modelHit$_source7 === void 0 ? void 0 : _modelHit$_source7.connector
      };
    }
  });
  return modelDict;
}
function getConnectorsFromResponses(modelHits) {
  const connectorDict = {};
  modelHits.forEach(connectorHit => {
    var _connectorHit$_source, _connectorHit$_source2, _connectorHit$_source3, _connectorHit$_source4, _connectorHit$_source5, _connectorHit$_source6;
    const connectorId = connectorHit._id;

    // in case of schema changes from ML plugin, this may crash. That is ok, as the error
    // produced will help expose the root cause
    connectorDict[connectorId] = {
      id: connectorId,
      name: (_connectorHit$_source = connectorHit._source) === null || _connectorHit$_source === void 0 ? void 0 : _connectorHit$_source.name,
      protocol: (_connectorHit$_source2 = connectorHit._source) === null || _connectorHit$_source2 === void 0 ? void 0 : _connectorHit$_source2.protocol,
      parameters: {
        model: (_connectorHit$_source3 = connectorHit._source) === null || _connectorHit$_source3 === void 0 || (_connectorHit$_source3 = _connectorHit$_source3.parameters) === null || _connectorHit$_source3 === void 0 ? void 0 : _connectorHit$_source3.model,
        dimensions: (_connectorHit$_source4 = connectorHit._source) === null || _connectorHit$_source4 === void 0 ? void 0 : _connectorHit$_source4.parameters.dimensions
      },
      actions: ((_connectorHit$_source5 = connectorHit._source) === null || _connectorHit$_source5 === void 0 ? void 0 : _connectorHit$_source5.actions) || [],
      client_config: ((_connectorHit$_source6 = connectorHit._source) === null || _connectorHit$_source6 === void 0 ? void 0 : _connectorHit$_source6.client_config) || {}
    };
  });
  return connectorDict;
}

// Convert the workflow state into a readable/presentable state on frontend
function getWorkflowStateFromResponse(state) {
  const finalState = state || _common.DEFAULT_NEW_WORKFLOW_STATE_TYPE;
  // @ts-ignore
  return _common.WORKFLOW_STATE[finalState];
}

// Convert the workflow resources into a readable/presentable state on frontend
function getResourcesCreatedFromResponse(resourcesCreated) {
  const finalResources = [];
  if (resourcesCreated) {
    resourcesCreated.forEach(backendResource => {
      finalResources.push({
        id: backendResource.resource_id,
        stepType: backendResource.workflow_step_name,
        type:
        // @ts-ignore
        _common.WORKFLOW_RESOURCE_TYPE[
        // the backend persists the types in lowercase. e.g., "pipeline_id"
        backendResource.resource_type.toUpperCase()]
      });
    });
  }
  return finalResources;
}

// Convert backend agent into a frontend agent obj
function toAgentObj(hitSource, id) {
  // MCP connectors are stringified when indexed. Convert back to an obj
  // to be consistently stored on client-side.
  let mcpConnectors;
  try {
    var _hitSource$parameters;
    mcpConnectors = JSON.parse(hitSource === null || hitSource === void 0 || (_hitSource$parameters = hitSource.parameters) === null || _hitSource$parameters === void 0 ? void 0 : _hitSource$parameters.mcp_connectors);
  } catch {}
  return {
    id,
    name: hitSource === null || hitSource === void 0 ? void 0 : hitSource.name,
    type: hitSource === null || hitSource === void 0 ? void 0 : hitSource.type,
    description: hitSource === null || hitSource === void 0 ? void 0 : hitSource.description,
    tools: hitSource === null || hitSource === void 0 ? void 0 : hitSource.tools,
    llm: hitSource === null || hitSource === void 0 ? void 0 : hitSource.llm,
    memory: hitSource === null || hitSource === void 0 ? void 0 : hitSource.memory,
    parameters: {
      ...(hitSource === null || hitSource === void 0 ? void 0 : hitSource.parameters),
      mcp_connectors: mcpConnectors
    }
  };
}

// Convert the agent hits into a readable/presentable state on frontend
function getAgentsFromResponses(agentHits) {
  const agents = {};
  for (const agentHit of agentHits) {
    const source = agentHit._source;
    const id = agentHit._id;
    // @ts-ignore
    agents[id] = toAgentObj(source, id);
  }
  return agents;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29tbW9uIiwicmVxdWlyZSIsImdlbmVyYXRlQ3VzdG9tRXJyb3IiLCJyZXMiLCJlcnIiLCJfZXJyJGJvZHkiLCJpc0RhdGFzb3VyY2VFcnJvciIsImN1c3RvbUVycm9yIiwic3RhdHVzQ29kZSIsImJvZHkiLCJtZXNzYWdlIiwiYXR0cmlidXRlcyIsImVycm9yIiwidW5kZWZpbmVkIiwiaW5jbHVkZXMiLCJJTlZBTElEX0RBVEFTT1VSQ0VfTVNHIiwiaXNJZ25vcmFibGVFcnJvciIsIl9lcnJvciRib2R5IiwiX2Vycm9yJGJvZHkyIiwiX2Vycm9yJGJvZHkzIiwidHlwZSIsIklOREVYX05PVF9GT1VORF9FWENFUFRJT04iLCJjYXVzZWRfYnkiLCJOT19NT0RJRklDQVRJT05TX0ZPVU5EX1RFWFQiLCJ0b1dvcmtmbG93T2JqIiwiaGl0U291cmNlIiwiaWQiLCJuYW1lIiwidXNlX2Nhc2UiLCJkZXNjcmlwdGlvbiIsInZlcnNpb24iLCJ3b3JrZmxvd3MiLCJ1aV9tZXRhZGF0YSIsImxhc3RVcGRhdGVkIiwibGFzdF91cGRhdGVkX3RpbWUiLCJsYXN0TGF1bmNoZWQiLCJsYXN0X3Byb3Zpc2lvbmVkX3RpbWUiLCJnZXRXb3JrZmxvd3NGcm9tUmVzcG9uc2VzIiwid29ya2Zsb3dIaXRzIiwid29ya2Zsb3dTdGF0ZUhpdHMiLCJ3b3JrZmxvd0RpY3QiLCJmb3JFYWNoIiwid29ya2Zsb3dIaXQiLCJfd29ya2Zsb3dTdGF0ZUhpdCRfc28iLCJfd29ya2Zsb3dTdGF0ZUhpdCRfc28yIiwiX3dvcmtmbG93U3RhdGVIaXQkX3NvMyIsIl9zb3VyY2UiLCJfaWQiLCJ3b3JrZmxvd1N0YXRlSGl0IiwiZmluZCIsIndvcmtmbG93U3RhdGUiLCJnZXRXb3JrZmxvd1N0YXRlRnJvbVJlc3BvbnNlIiwic3RhdGUiLCJ3b3JrZmxvd0Vycm9yIiwid29ya2Zsb3dSZXNvdXJjZXNDcmVhdGVkIiwiZ2V0UmVzb3VyY2VzQ3JlYXRlZEZyb21SZXNwb25zZSIsInJlc291cmNlc19jcmVhdGVkIiwicmVzb3VyY2VzQ3JlYXRlZCIsImdldE1vZGVsc0Zyb21SZXNwb25zZXMiLCJtb2RlbEhpdHMiLCJtb2RlbERpY3QiLCJtb2RlbEhpdCIsImNodW5rX251bWJlciIsIl9tb2RlbEhpdCRfc291cmNlIiwiX21vZGVsSGl0JF9zb3VyY2UyIiwiX21vZGVsSGl0JF9zb3VyY2UzIiwiX21vZGVsSGl0JF9zb3VyY2U0IiwiX21vZGVsSGl0JF9zb3VyY2U1IiwiX21vZGVsSGl0JF9zb3VyY2U2IiwiX21vZGVsSGl0JF9zb3VyY2U3IiwibW9kZWxJZCIsImluZGV4ZWRNb2RlbEludGVyZmFjZSIsImludGVyZmFjZSIsIm1vZGVsSW50ZXJmYWNlIiwicGFyc2VkSW5wdXQiLCJwYXJzZWRPdXRwdXQiLCJKU09OIiwicGFyc2UiLCJpbnB1dCIsIm91dHB1dCIsImFsZ29yaXRobSIsIk1PREVMX0FMR09SSVRITSIsIk1PREVMX1NUQVRFIiwibW9kZWxfc3RhdGUiLCJtb2RlbENvbmZpZyIsIm1vZGVsVHlwZSIsIm1vZGVsX2NvbmZpZyIsIm1vZGVsX3R5cGUiLCJlbWJlZGRpbmdEaW1lbnNpb24iLCJlbWJlZGRpbmdfZGltZW5zaW9uIiwiY29ubmVjdG9ySWQiLCJjb25uZWN0b3JfaWQiLCJjb25uZWN0b3IiLCJnZXRDb25uZWN0b3JzRnJvbVJlc3BvbnNlcyIsImNvbm5lY3RvckRpY3QiLCJjb25uZWN0b3JIaXQiLCJfY29ubmVjdG9ySGl0JF9zb3VyY2UiLCJfY29ubmVjdG9ySGl0JF9zb3VyY2UyIiwiX2Nvbm5lY3RvckhpdCRfc291cmNlMyIsIl9jb25uZWN0b3JIaXQkX3NvdXJjZTQiLCJfY29ubmVjdG9ySGl0JF9zb3VyY2U1IiwiX2Nvbm5lY3RvckhpdCRfc291cmNlNiIsInByb3RvY29sIiwicGFyYW1ldGVycyIsIm1vZGVsIiwiZGltZW5zaW9ucyIsImFjdGlvbnMiLCJjbGllbnRfY29uZmlnIiwiZmluYWxTdGF0ZSIsIkRFRkFVTFRfTkVXX1dPUktGTE9XX1NUQVRFX1RZUEUiLCJXT1JLRkxPV19TVEFURSIsImZpbmFsUmVzb3VyY2VzIiwiYmFja2VuZFJlc291cmNlIiwicHVzaCIsInJlc291cmNlX2lkIiwic3RlcFR5cGUiLCJ3b3JrZmxvd19zdGVwX25hbWUiLCJXT1JLRkxPV19SRVNPVVJDRV9UWVBFIiwicmVzb3VyY2VfdHlwZSIsInRvVXBwZXJDYXNlIiwidG9BZ2VudE9iaiIsIm1jcENvbm5lY3RvcnMiLCJfaGl0U291cmNlJHBhcmFtZXRlcnMiLCJtY3BfY29ubmVjdG9ycyIsInRvb2xzIiwibGxtIiwibWVtb3J5IiwiZ2V0QWdlbnRzRnJvbVJlc3BvbnNlcyIsImFnZW50SGl0cyIsImFnZW50cyIsImFnZW50SGl0Iiwic291cmNlIl0sInNvdXJjZXMiOlsiaGVscGVycy50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCB7XG4gIEFnZW50LFxuICBBZ2VudERpY3QsXG4gIENvbm5lY3RvcixcbiAgQ29ubmVjdG9yRGljdCxcbiAgREVGQVVMVF9ORVdfV09SS0ZMT1dfU1RBVEVfVFlQRSxcbiAgSU5ERVhfTk9UX0ZPVU5EX0VYQ0VQVElPTixcbiAgSU5WQUxJRF9EQVRBU09VUkNFX01TRyxcbiAgTUNQQ29ubmVjdG9yLFxuICBNT0RFTF9BTEdPUklUSE0sXG4gIE1PREVMX1NUQVRFLFxuICBNb2RlbCxcbiAgTW9kZWxEaWN0LFxuICBNb2RlbElucHV0LFxuICBNb2RlbEludGVyZmFjZSxcbiAgTW9kZWxPdXRwdXQsXG4gIE5PX01PRElGSUNBVElPTlNfRk9VTkRfVEVYVCxcbiAgU2VhcmNoSGl0LFxuICBXT1JLRkxPV19SRVNPVVJDRV9UWVBFLFxuICBXT1JLRkxPV19TVEFURSxcbiAgV29ya2Zsb3csXG4gIFdvcmtmbG93RGljdCxcbiAgV29ya2Zsb3dSZXNvdXJjZSxcbn0gZnJvbSAnLi4vLi4vY29tbW9uJztcblxuLy8gT1NEIGRvZXMgbm90IHByb3ZpZGUgYW4gaW50ZXJmYWNlIGZvciB0aGlzIHJlc3BvbnNlLCBidXQgdGhpcyBpcyBmb2xsb3dpbmcgdGhlIHN1Z2dlc3RlZFxuLy8gaW1wbGVtZW50YXRpb25zLiBUbyBwcmV2ZW50IHR5cGVzY3JpcHQgY29tcGxhaW5pbmcsIGxlYXZpbmcgYXMgbG9vc2VseS10eXBlZCAnYW55J1xuZXhwb3J0IGZ1bmN0aW9uIGdlbmVyYXRlQ3VzdG9tRXJyb3IocmVzOiBhbnksIGVycjogYW55KSB7XG4gIGlmIChpc0RhdGFzb3VyY2VFcnJvcihlcnIpKSB7XG4gICAgcmV0dXJuIHJlcy5jdXN0b21FcnJvcih7XG4gICAgICBzdGF0dXNDb2RlOiA0MDQsXG4gICAgICBib2R5OiB7XG4gICAgICAgIG1lc3NhZ2U6ICdEYXRhIHNvdXJjZSBub3QgZm91bmQnLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuICByZXR1cm4gcmVzLmN1c3RvbUVycm9yKHtcbiAgICBzdGF0dXNDb2RlOiBlcnIuc3RhdHVzQ29kZSB8fCA1MDAsXG4gICAgYm9keToge1xuICAgICAgbWVzc2FnZTogZXJyLm1lc3NhZ2UsXG4gICAgICBhdHRyaWJ1dGVzOiB7XG4gICAgICAgIGVycm9yOiBlcnIuYm9keT8uZXJyb3IgfHwgZXJyLm1lc3NhZ2UsXG4gICAgICB9LFxuICAgIH0sXG4gIH0pO1xufVxuXG5mdW5jdGlvbiBpc0RhdGFzb3VyY2VFcnJvcihlcnI6IGFueSkge1xuICBpZiAoZXJyLm1lc3NhZ2UgIT09IHVuZGVmaW5lZCAmJiB0eXBlb2YgZXJyLm1lc3NhZ2UgPT09ICdzdHJpbmcnKSB7XG4gICAgcmV0dXJuIGVyci5tZXNzYWdlLmluY2x1ZGVzKElOVkFMSURfREFUQVNPVVJDRV9NU0cpO1xuICB9XG59XG5cbi8vIEhlbHBlciBmbiB0byBmaWx0ZXIgb3V0IGJhY2tlbmQgZXJyb3JzIHRoYXQgd2UgZG9uJ3Qgd2FudCB0byBwcm9wYWdhdGUgb24gdGhlIGZyb250ZW5kLlxuZXhwb3J0IGZ1bmN0aW9uIGlzSWdub3JhYmxlRXJyb3IoZXJyb3I6IGFueSk6IGJvb2xlYW4ge1xuICByZXR1cm4gKFxuICAgIGVycm9yLmJvZHk/LmVycm9yPy50eXBlID09PSBJTkRFWF9OT1RfRk9VTkRfRVhDRVBUSU9OIHx8XG4gICAgZXJyb3IuYm9keT8uZXJyb3I/LmNhdXNlZF9ieT8udHlwZSA9PT0gSU5ERVhfTk9UX0ZPVU5EX0VYQ0VQVElPTiB8fFxuICAgIGVycm9yLmJvZHk/LmVycm9yID09PSBOT19NT0RJRklDQVRJT05TX0ZPVU5EX1RFWFRcbiAgKTtcbn1cblxuLy8gQ29udmVydCBiYWNrZW5kIHdvcmtmbG93IGludG8gZnJvbnRlbmQgd29ya2Zsb3cgb2JqXG5leHBvcnQgZnVuY3Rpb24gdG9Xb3JrZmxvd09iaihoaXRTb3VyY2U6IGFueSwgaWQ6IHN0cmluZyk6IFdvcmtmbG93IHtcbiAgcmV0dXJuIHtcbiAgICBpZCxcbiAgICBuYW1lOiBoaXRTb3VyY2UubmFtZSxcbiAgICB1c2VfY2FzZTogaGl0U291cmNlLnVzZV9jYXNlLFxuICAgIGRlc2NyaXB0aW9uOiBoaXRTb3VyY2UuZGVzY3JpcHRpb24gfHwgJycsXG4gICAgdmVyc2lvbjogaGl0U291cmNlLnZlcnNpb24sXG4gICAgd29ya2Zsb3dzOiBoaXRTb3VyY2Uud29ya2Zsb3dzLFxuICAgIHVpX21ldGFkYXRhOiBoaXRTb3VyY2UudWlfbWV0YWRhdGEsXG4gICAgbGFzdFVwZGF0ZWQ6IGhpdFNvdXJjZS5sYXN0X3VwZGF0ZWRfdGltZSxcbiAgICBsYXN0TGF1bmNoZWQ6IGhpdFNvdXJjZS5sYXN0X3Byb3Zpc2lvbmVkX3RpbWUsXG4gIH0gYXMgV29ya2Zsb3c7XG59XG5cbi8vIFRPRE86IGNhbiByZW1vdmUgb3Igc2ltcGxpZnkgaWYgd2UgY2FuIGZldGNoIGFsbCBkYXRhIGZyb20gYSBzaW5nbGUgQVBJIGNhbGwuIFRyYWNraW5nIGlzc3VlOlxuLy8gaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9mbG93LWZyYW1ld29yay9pc3N1ZXMvMTcxXG4vLyBDdXJyZW50IGltcGxlbWVudGF0aW9uIGNvbWJpbmVzIDIgc2VhcmNoIHJlc3BvbnNlcyB0byBjcmVhdGUgYSBzaW5nbGUgc2V0IG9mIHdvcmtmbG93cyB3aXRoXG4vLyBzdGF0aWMgaW5mb3JtYXRpb24gKyBzdGF0ZSBpbmZvcm1hdGlvblxuZXhwb3J0IGZ1bmN0aW9uIGdldFdvcmtmbG93c0Zyb21SZXNwb25zZXMoXG4gIHdvcmtmbG93SGl0czogU2VhcmNoSGl0W10sXG4gIHdvcmtmbG93U3RhdGVIaXRzOiBTZWFyY2hIaXRbXVxuKTogV29ya2Zsb3dEaWN0IHtcbiAgY29uc3Qgd29ya2Zsb3dEaWN0ID0ge30gYXMgV29ya2Zsb3dEaWN0O1xuICB3b3JrZmxvd0hpdHMuZm9yRWFjaCgod29ya2Zsb3dIaXQ6IFNlYXJjaEhpdCkgPT4ge1xuICAgIGNvbnN0IGhpdFNvdXJjZSA9IHdvcmtmbG93SGl0Ll9zb3VyY2U7XG4gICAgd29ya2Zsb3dEaWN0W3dvcmtmbG93SGl0Ll9pZF0gPSB0b1dvcmtmbG93T2JqKGhpdFNvdXJjZSwgd29ya2Zsb3dIaXQuX2lkKTtcbiAgICBjb25zdCB3b3JrZmxvd1N0YXRlSGl0ID0gd29ya2Zsb3dTdGF0ZUhpdHMuZmluZChcbiAgICAgICh3b3JrZmxvd1N0YXRlSGl0KSA9PiB3b3JrZmxvd1N0YXRlSGl0Ll9pZCA9PT0gd29ya2Zsb3dIaXQuX2lkXG4gICAgKTtcbiAgICBjb25zdCB3b3JrZmxvd1N0YXRlID0gZ2V0V29ya2Zsb3dTdGF0ZUZyb21SZXNwb25zZShcbiAgICAgIHdvcmtmbG93U3RhdGVIaXQ/Ll9zb3VyY2U/LnN0YXRlXG4gICAgKTtcbiAgICBjb25zdCB3b3JrZmxvd0Vycm9yID0gd29ya2Zsb3dTdGF0ZUhpdD8uX3NvdXJjZT8uZXJyb3I7XG4gICAgY29uc3Qgd29ya2Zsb3dSZXNvdXJjZXNDcmVhdGVkID0gZ2V0UmVzb3VyY2VzQ3JlYXRlZEZyb21SZXNwb25zZShcbiAgICAgIHdvcmtmbG93U3RhdGVIaXQ/Ll9zb3VyY2U/LnJlc291cmNlc19jcmVhdGVkXG4gICAgKTtcbiAgICB3b3JrZmxvd0RpY3Rbd29ya2Zsb3dIaXQuX2lkXSA9IHtcbiAgICAgIC4uLndvcmtmbG93RGljdFt3b3JrZmxvd0hpdC5faWRdLFxuICAgICAgLy8gQHRzLWlnbm9yZVxuICAgICAgc3RhdGU6IHdvcmtmbG93U3RhdGUsXG4gICAgICBlcnJvcjogd29ya2Zsb3dFcnJvcixcbiAgICAgIHJlc291cmNlc0NyZWF0ZWQ6IHdvcmtmbG93UmVzb3VyY2VzQ3JlYXRlZCxcbiAgICB9O1xuICB9KTtcbiAgcmV0dXJuIHdvcmtmbG93RGljdDtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldE1vZGVsc0Zyb21SZXNwb25zZXMobW9kZWxIaXRzOiBTZWFyY2hIaXRbXSk6IE1vZGVsRGljdCB7XG4gIGNvbnN0IG1vZGVsRGljdCA9IHt9IGFzIE1vZGVsRGljdDtcbiAgbW9kZWxIaXRzLmZvckVhY2goKG1vZGVsSGl0OiBTZWFyY2hIaXQpID0+IHtcbiAgICAvLyBzZWFyY2ggbW9kZWwgQVBJIHJldHVybnMgaGl0cyBmb3IgZWFjaCBkZXBsb3llZCBtb2RlbCBjaHVuay4gaWdub3JlIHRoZXNlIGhpdHNcbiAgICBpZiAobW9kZWxIaXQuX3NvdXJjZS5jaHVua19udW1iZXIgPT09IHVuZGVmaW5lZCkge1xuICAgICAgY29uc3QgbW9kZWxJZCA9IG1vZGVsSGl0Ll9pZDtcblxuICAgICAgLy8gdGhlIHBlcnNpc3RlZCBtb2RlbCBpbnRlcmZhY2UgKGlmIGF2YWlsYWJsZSkgaXMgYSBtaXggb2YgYW4gb2JqIGFuZCBzdHJpbmcuXG4gICAgICAvLyBXZSBwYXJzZSB0aGUgc3RyaW5nIHZhbHVlcyBmb3IgaW5wdXQvb3V0cHV0IHRvIGhhdmUgYSBjb21wbGV0ZVxuICAgICAgLy8gZW5kLXRvLWVuZCBKU09OU2NoZW1hIG9ialxuICAgICAgbGV0IGluZGV4ZWRNb2RlbEludGVyZmFjZSA9IG1vZGVsSGl0Ll9zb3VyY2UuaW50ZXJmYWNlIGFzXG4gICAgICAgIHwgeyBpbnB1dDogc3RyaW5nOyBvdXRwdXQ6IHN0cmluZyB9XG4gICAgICAgIHwgdW5kZWZpbmVkO1xuICAgICAgbGV0IG1vZGVsSW50ZXJmYWNlID0gdW5kZWZpbmVkIGFzIE1vZGVsSW50ZXJmYWNlIHwgdW5kZWZpbmVkO1xuICAgICAgaWYgKGluZGV4ZWRNb2RlbEludGVyZmFjZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGxldCBwYXJzZWRJbnB1dCA9IHVuZGVmaW5lZCBhcyBNb2RlbElucHV0IHwgdW5kZWZpbmVkO1xuICAgICAgICBsZXQgcGFyc2VkT3V0cHV0ID0gdW5kZWZpbmVkIGFzIE1vZGVsT3V0cHV0IHwgdW5kZWZpbmVkO1xuICAgICAgICB0cnkge1xuICAgICAgICAgIHBhcnNlZElucHV0ID0gSlNPTi5wYXJzZShpbmRleGVkTW9kZWxJbnRlcmZhY2UuaW5wdXQpO1xuICAgICAgICB9IGNhdGNoIHt9XG4gICAgICAgIHRyeSB7XG4gICAgICAgICAgcGFyc2VkT3V0cHV0ID0gSlNPTi5wYXJzZShpbmRleGVkTW9kZWxJbnRlcmZhY2Uub3V0cHV0KTtcbiAgICAgICAgfSBjYXRjaCB7fVxuICAgICAgICBtb2RlbEludGVyZmFjZSA9IHtcbiAgICAgICAgICBpbnB1dDogcGFyc2VkSW5wdXQsXG4gICAgICAgICAgb3V0cHV0OiBwYXJzZWRPdXRwdXQsXG4gICAgICAgIH0gYXMgTW9kZWxJbnRlcmZhY2U7XG4gICAgICB9XG5cbiAgICAgIC8vIGluIGNhc2Ugb2Ygc2NoZW1hIGNoYW5nZXMgZnJvbSBNTCBwbHVnaW4sIHRoaXMgbWF5IGNyYXNoLiBUaGF0IGlzIG9rLCBhcyB0aGUgZXJyb3JcbiAgICAgIC8vIHByb2R1Y2VkIHdpbGwgaGVscCBleHBvc2UgdGhlIHJvb3QgY2F1c2VcbiAgICAgIG1vZGVsRGljdFttb2RlbElkXSA9IHtcbiAgICAgICAgaWQ6IG1vZGVsSWQsXG4gICAgICAgIG5hbWU6IG1vZGVsSGl0Ll9zb3VyY2U/Lm5hbWUsXG4gICAgICAgIC8vIEB0cy1pZ25vcmVcbiAgICAgICAgYWxnb3JpdGhtOiBNT0RFTF9BTEdPUklUSE1bbW9kZWxIaXQuX3NvdXJjZT8uYWxnb3JpdGhtXSxcbiAgICAgICAgLy8gQHRzLWlnbm9yZVxuICAgICAgICBzdGF0ZTogTU9ERUxfU1RBVEVbbW9kZWxIaXQuX3NvdXJjZT8ubW9kZWxfc3RhdGVdLFxuICAgICAgICBtb2RlbENvbmZpZzoge1xuICAgICAgICAgIG1vZGVsVHlwZTogbW9kZWxIaXQuX3NvdXJjZT8ubW9kZWxfY29uZmlnPy5tb2RlbF90eXBlLFxuICAgICAgICAgIGVtYmVkZGluZ0RpbWVuc2lvbjpcbiAgICAgICAgICAgIG1vZGVsSGl0Ll9zb3VyY2U/Lm1vZGVsX2NvbmZpZz8uZW1iZWRkaW5nX2RpbWVuc2lvbixcbiAgICAgICAgfSxcbiAgICAgICAgaW50ZXJmYWNlOiBtb2RlbEludGVyZmFjZSxcbiAgICAgICAgY29ubmVjdG9ySWQ6IG1vZGVsSGl0Ll9zb3VyY2U/LmNvbm5lY3Rvcl9pZCxcbiAgICAgICAgY29ubmVjdG9yOiBtb2RlbEhpdC5fc291cmNlPy5jb25uZWN0b3IsXG4gICAgICB9IGFzIE1vZGVsO1xuICAgIH1cbiAgfSk7XG4gIHJldHVybiBtb2RlbERpY3Q7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXRDb25uZWN0b3JzRnJvbVJlc3BvbnNlcyhcbiAgbW9kZWxIaXRzOiBTZWFyY2hIaXRbXVxuKTogQ29ubmVjdG9yRGljdCB7XG4gIGNvbnN0IGNvbm5lY3RvckRpY3QgPSB7fSBhcyBDb25uZWN0b3JEaWN0O1xuICBtb2RlbEhpdHMuZm9yRWFjaCgoY29ubmVjdG9ySGl0OiBTZWFyY2hIaXQpID0+IHtcbiAgICBjb25zdCBjb25uZWN0b3JJZCA9IGNvbm5lY3RvckhpdC5faWQ7XG5cbiAgICAvLyBpbiBjYXNlIG9mIHNjaGVtYSBjaGFuZ2VzIGZyb20gTUwgcGx1Z2luLCB0aGlzIG1heSBjcmFzaC4gVGhhdCBpcyBvaywgYXMgdGhlIGVycm9yXG4gICAgLy8gcHJvZHVjZWQgd2lsbCBoZWxwIGV4cG9zZSB0aGUgcm9vdCBjYXVzZVxuICAgIGNvbm5lY3RvckRpY3RbY29ubmVjdG9ySWRdID0ge1xuICAgICAgaWQ6IGNvbm5lY3RvcklkLFxuICAgICAgbmFtZTogY29ubmVjdG9ySGl0Ll9zb3VyY2U/Lm5hbWUsXG4gICAgICBwcm90b2NvbDogY29ubmVjdG9ySGl0Ll9zb3VyY2U/LnByb3RvY29sLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICBtb2RlbDogY29ubmVjdG9ySGl0Ll9zb3VyY2U/LnBhcmFtZXRlcnM/Lm1vZGVsLFxuICAgICAgICBkaW1lbnNpb25zOiBjb25uZWN0b3JIaXQuX3NvdXJjZT8ucGFyYW1ldGVycy5kaW1lbnNpb25zLFxuICAgICAgfSxcbiAgICAgIGFjdGlvbnM6IGNvbm5lY3RvckhpdC5fc291cmNlPy5hY3Rpb25zIHx8IFtdLFxuICAgICAgY2xpZW50X2NvbmZpZzogY29ubmVjdG9ySGl0Ll9zb3VyY2U/LmNsaWVudF9jb25maWcgfHwge30sXG4gICAgfSBhcyBDb25uZWN0b3I7XG4gIH0pO1xuICByZXR1cm4gY29ubmVjdG9yRGljdDtcbn1cblxuLy8gQ29udmVydCB0aGUgd29ya2Zsb3cgc3RhdGUgaW50byBhIHJlYWRhYmxlL3ByZXNlbnRhYmxlIHN0YXRlIG9uIGZyb250ZW5kXG5leHBvcnQgZnVuY3Rpb24gZ2V0V29ya2Zsb3dTdGF0ZUZyb21SZXNwb25zZShcbiAgc3RhdGU6IHR5cGVvZiBXT1JLRkxPV19TVEFURSB8IHVuZGVmaW5lZFxuKTogV09SS0ZMT1dfU1RBVEUge1xuICBjb25zdCBmaW5hbFN0YXRlID0gc3RhdGUgfHwgREVGQVVMVF9ORVdfV09SS0ZMT1dfU1RBVEVfVFlQRTtcbiAgLy8gQHRzLWlnbm9yZVxuICByZXR1cm4gV09SS0ZMT1dfU1RBVEVbZmluYWxTdGF0ZV07XG59XG5cbi8vIENvbnZlcnQgdGhlIHdvcmtmbG93IHJlc291cmNlcyBpbnRvIGEgcmVhZGFibGUvcHJlc2VudGFibGUgc3RhdGUgb24gZnJvbnRlbmRcbmV4cG9ydCBmdW5jdGlvbiBnZXRSZXNvdXJjZXNDcmVhdGVkRnJvbVJlc3BvbnNlKFxuICByZXNvdXJjZXNDcmVhdGVkOiBhbnlbXSB8IHVuZGVmaW5lZFxuKTogV29ya2Zsb3dSZXNvdXJjZVtdIHtcbiAgY29uc3QgZmluYWxSZXNvdXJjZXMgPSBbXSBhcyBXb3JrZmxvd1Jlc291cmNlW107XG4gIGlmIChyZXNvdXJjZXNDcmVhdGVkKSB7XG4gICAgcmVzb3VyY2VzQ3JlYXRlZC5mb3JFYWNoKChiYWNrZW5kUmVzb3VyY2UpID0+IHtcbiAgICAgIGZpbmFsUmVzb3VyY2VzLnB1c2goe1xuICAgICAgICBpZDogYmFja2VuZFJlc291cmNlLnJlc291cmNlX2lkLFxuICAgICAgICBzdGVwVHlwZTogYmFja2VuZFJlc291cmNlLndvcmtmbG93X3N0ZXBfbmFtZSxcbiAgICAgICAgdHlwZTpcbiAgICAgICAgICAvLyBAdHMtaWdub3JlXG4gICAgICAgICAgV09SS0ZMT1dfUkVTT1VSQ0VfVFlQRVtcbiAgICAgICAgICAgIC8vIHRoZSBiYWNrZW5kIHBlcnNpc3RzIHRoZSB0eXBlcyBpbiBsb3dlcmNhc2UuIGUuZy4sIFwicGlwZWxpbmVfaWRcIlxuICAgICAgICAgICAgKGJhY2tlbmRSZXNvdXJjZS5yZXNvdXJjZV90eXBlIGFzIHN0cmluZykudG9VcHBlckNhc2UoKVxuICAgICAgICAgIF0sXG4gICAgICB9IGFzIFdvcmtmbG93UmVzb3VyY2UpO1xuICAgIH0pO1xuICB9XG4gIHJldHVybiBmaW5hbFJlc291cmNlcztcbn1cblxuLy8gQ29udmVydCBiYWNrZW5kIGFnZW50IGludG8gYSBmcm9udGVuZCBhZ2VudCBvYmpcbmV4cG9ydCBmdW5jdGlvbiB0b0FnZW50T2JqKGhpdFNvdXJjZTogYW55LCBpZDogc3RyaW5nKTogQWdlbnQge1xuICAvLyBNQ1AgY29ubmVjdG9ycyBhcmUgc3RyaW5naWZpZWQgd2hlbiBpbmRleGVkLiBDb252ZXJ0IGJhY2sgdG8gYW4gb2JqXG4gIC8vIHRvIGJlIGNvbnNpc3RlbnRseSBzdG9yZWQgb24gY2xpZW50LXNpZGUuXG4gIGxldCBtY3BDb25uZWN0b3JzO1xuICB0cnkge1xuICAgIG1jcENvbm5lY3RvcnMgPSBKU09OLnBhcnNlKFxuICAgICAgaGl0U291cmNlPy5wYXJhbWV0ZXJzPy5tY3BfY29ubmVjdG9yc1xuICAgICkgYXMgTUNQQ29ubmVjdG9yW107XG4gIH0gY2F0Y2gge31cbiAgcmV0dXJuIHtcbiAgICBpZCxcbiAgICBuYW1lOiBoaXRTb3VyY2U/Lm5hbWUsXG4gICAgdHlwZTogaGl0U291cmNlPy50eXBlLFxuICAgIGRlc2NyaXB0aW9uOiBoaXRTb3VyY2U/LmRlc2NyaXB0aW9uLFxuICAgIHRvb2xzOiBoaXRTb3VyY2U/LnRvb2xzLFxuICAgIGxsbTogaGl0U291cmNlPy5sbG0sXG4gICAgbWVtb3J5OiBoaXRTb3VyY2U/Lm1lbW9yeSxcbiAgICBwYXJhbWV0ZXJzOiB7IC4uLmhpdFNvdXJjZT8ucGFyYW1ldGVycywgbWNwX2Nvbm5lY3RvcnM6IG1jcENvbm5lY3RvcnMgfSxcbiAgfSBhcyBBZ2VudDtcbn1cblxuLy8gQ29udmVydCB0aGUgYWdlbnQgaGl0cyBpbnRvIGEgcmVhZGFibGUvcHJlc2VudGFibGUgc3RhdGUgb24gZnJvbnRlbmRcbmV4cG9ydCBmdW5jdGlvbiBnZXRBZ2VudHNGcm9tUmVzcG9uc2VzKGFnZW50SGl0czogU2VhcmNoSGl0W10pOiBBZ2VudERpY3Qge1xuICBjb25zdCBhZ2VudHMgPSB7fSBhcyBBZ2VudERpY3Q7XG4gIGZvciAoY29uc3QgYWdlbnRIaXQgb2YgYWdlbnRIaXRzKSB7XG4gICAgY29uc3Qgc291cmNlID0gYWdlbnRIaXQuX3NvdXJjZSBhcyBBZ2VudDtcbiAgICBjb25zdCBpZCA9IGFnZW50SGl0Ll9pZDtcbiAgICAvLyBAdHMtaWdub3JlXG4gICAgYWdlbnRzW2lkXSA9IHRvQWdlbnRPYmooc291cmNlLCBpZCk7XG4gIH1cbiAgcmV0dXJuIGFnZW50cztcbn1cbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7Ozs7Ozs7O0FBS0EsSUFBQUEsT0FBQSxHQUFBQyxPQUFBO0FBTEE7QUFDQTtBQUNBO0FBQ0E7O0FBMkJBO0FBQ0E7QUFDTyxTQUFTQyxtQkFBbUJBLENBQUNDLEdBQVEsRUFBRUMsR0FBUSxFQUFFO0VBQUEsSUFBQUMsU0FBQTtFQUN0RCxJQUFJQyxpQkFBaUIsQ0FBQ0YsR0FBRyxDQUFDLEVBQUU7SUFDMUIsT0FBT0QsR0FBRyxDQUFDSSxXQUFXLENBQUM7TUFDckJDLFVBQVUsRUFBRSxHQUFHO01BQ2ZDLElBQUksRUFBRTtRQUNKQyxPQUFPLEVBQUU7TUFDWDtJQUNGLENBQUMsQ0FBQztFQUNKO0VBQ0EsT0FBT1AsR0FBRyxDQUFDSSxXQUFXLENBQUM7SUFDckJDLFVBQVUsRUFBRUosR0FBRyxDQUFDSSxVQUFVLElBQUksR0FBRztJQUNqQ0MsSUFBSSxFQUFFO01BQ0pDLE9BQU8sRUFBRU4sR0FBRyxDQUFDTSxPQUFPO01BQ3BCQyxVQUFVLEVBQUU7UUFDVkMsS0FBSyxFQUFFLEVBQUFQLFNBQUEsR0FBQUQsR0FBRyxDQUFDSyxJQUFJLGNBQUFKLFNBQUEsdUJBQVJBLFNBQUEsQ0FBVU8sS0FBSyxLQUFJUixHQUFHLENBQUNNO01BQ2hDO0lBQ0Y7RUFDRixDQUFDLENBQUM7QUFDSjtBQUVBLFNBQVNKLGlCQUFpQkEsQ0FBQ0YsR0FBUSxFQUFFO0VBQ25DLElBQUlBLEdBQUcsQ0FBQ00sT0FBTyxLQUFLRyxTQUFTLElBQUksT0FBT1QsR0FBRyxDQUFDTSxPQUFPLEtBQUssUUFBUSxFQUFFO0lBQ2hFLE9BQU9OLEdBQUcsQ0FBQ00sT0FBTyxDQUFDSSxRQUFRLENBQUNDLDhCQUFzQixDQUFDO0VBQ3JEO0FBQ0Y7O0FBRUE7QUFDTyxTQUFTQyxnQkFBZ0JBLENBQUNKLEtBQVUsRUFBVztFQUFBLElBQUFLLFdBQUEsRUFBQUMsWUFBQSxFQUFBQyxZQUFBO0VBQ3BELE9BQ0UsRUFBQUYsV0FBQSxHQUFBTCxLQUFLLENBQUNILElBQUksY0FBQVEsV0FBQSxnQkFBQUEsV0FBQSxHQUFWQSxXQUFBLENBQVlMLEtBQUssY0FBQUssV0FBQSx1QkFBakJBLFdBQUEsQ0FBbUJHLElBQUksTUFBS0MsaUNBQXlCLElBQ3JELEVBQUFILFlBQUEsR0FBQU4sS0FBSyxDQUFDSCxJQUFJLGNBQUFTLFlBQUEsZ0JBQUFBLFlBQUEsR0FBVkEsWUFBQSxDQUFZTixLQUFLLGNBQUFNLFlBQUEsZ0JBQUFBLFlBQUEsR0FBakJBLFlBQUEsQ0FBbUJJLFNBQVMsY0FBQUosWUFBQSx1QkFBNUJBLFlBQUEsQ0FBOEJFLElBQUksTUFBS0MsaUNBQXlCLElBQ2hFLEVBQUFGLFlBQUEsR0FBQVAsS0FBSyxDQUFDSCxJQUFJLGNBQUFVLFlBQUEsdUJBQVZBLFlBQUEsQ0FBWVAsS0FBSyxNQUFLVyxtQ0FBMkI7QUFFckQ7O0FBRUE7QUFDTyxTQUFTQyxhQUFhQSxDQUFDQyxTQUFjLEVBQUVDLEVBQVUsRUFBWTtFQUNsRSxPQUFPO0lBQ0xBLEVBQUU7SUFDRkMsSUFBSSxFQUFFRixTQUFTLENBQUNFLElBQUk7SUFDcEJDLFFBQVEsRUFBRUgsU0FBUyxDQUFDRyxRQUFRO0lBQzVCQyxXQUFXLEVBQUVKLFNBQVMsQ0FBQ0ksV0FBVyxJQUFJLEVBQUU7SUFDeENDLE9BQU8sRUFBRUwsU0FBUyxDQUFDSyxPQUFPO0lBQzFCQyxTQUFTLEVBQUVOLFNBQVMsQ0FBQ00sU0FBUztJQUM5QkMsV0FBVyxFQUFFUCxTQUFTLENBQUNPLFdBQVc7SUFDbENDLFdBQVcsRUFBRVIsU0FBUyxDQUFDUyxpQkFBaUI7SUFDeENDLFlBQVksRUFBRVYsU0FBUyxDQUFDVztFQUMxQixDQUFDO0FBQ0g7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDTyxTQUFTQyx5QkFBeUJBLENBQ3ZDQyxZQUF5QixFQUN6QkMsaUJBQThCLEVBQ2hCO0VBQ2QsTUFBTUMsWUFBWSxHQUFHLENBQUMsQ0FBaUI7RUFDdkNGLFlBQVksQ0FBQ0csT0FBTyxDQUFFQyxXQUFzQixJQUFLO0lBQUEsSUFBQUMscUJBQUEsRUFBQUMsc0JBQUEsRUFBQUMsc0JBQUE7SUFDL0MsTUFBTXBCLFNBQVMsR0FBR2lCLFdBQVcsQ0FBQ0ksT0FBTztJQUNyQ04sWUFBWSxDQUFDRSxXQUFXLENBQUNLLEdBQUcsQ0FBQyxHQUFHdkIsYUFBYSxDQUFDQyxTQUFTLEVBQUVpQixXQUFXLENBQUNLLEdBQUcsQ0FBQztJQUN6RSxNQUFNQyxnQkFBZ0IsR0FBR1QsaUJBQWlCLENBQUNVLElBQUksQ0FDNUNELGdCQUFnQixJQUFLQSxnQkFBZ0IsQ0FBQ0QsR0FBRyxLQUFLTCxXQUFXLENBQUNLLEdBQzdELENBQUM7SUFDRCxNQUFNRyxhQUFhLEdBQUdDLDRCQUE0QixDQUNoREgsZ0JBQWdCLGFBQWhCQSxnQkFBZ0IsZ0JBQUFMLHFCQUFBLEdBQWhCSyxnQkFBZ0IsQ0FBRUYsT0FBTyxjQUFBSCxxQkFBQSx1QkFBekJBLHFCQUFBLENBQTJCUyxLQUM3QixDQUFDO0lBQ0QsTUFBTUMsYUFBYSxHQUFHTCxnQkFBZ0IsYUFBaEJBLGdCQUFnQixnQkFBQUosc0JBQUEsR0FBaEJJLGdCQUFnQixDQUFFRixPQUFPLGNBQUFGLHNCQUFBLHVCQUF6QkEsc0JBQUEsQ0FBMkJoQyxLQUFLO0lBQ3RELE1BQU0wQyx3QkFBd0IsR0FBR0MsK0JBQStCLENBQzlEUCxnQkFBZ0IsYUFBaEJBLGdCQUFnQixnQkFBQUgsc0JBQUEsR0FBaEJHLGdCQUFnQixDQUFFRixPQUFPLGNBQUFELHNCQUFBLHVCQUF6QkEsc0JBQUEsQ0FBMkJXLGlCQUM3QixDQUFDO0lBQ0RoQixZQUFZLENBQUNFLFdBQVcsQ0FBQ0ssR0FBRyxDQUFDLEdBQUc7TUFDOUIsR0FBR1AsWUFBWSxDQUFDRSxXQUFXLENBQUNLLEdBQUcsQ0FBQztNQUNoQztNQUNBSyxLQUFLLEVBQUVGLGFBQWE7TUFDcEJ0QyxLQUFLLEVBQUV5QyxhQUFhO01BQ3BCSSxnQkFBZ0IsRUFBRUg7SUFDcEIsQ0FBQztFQUNILENBQUMsQ0FBQztFQUNGLE9BQU9kLFlBQVk7QUFDckI7QUFFTyxTQUFTa0Isc0JBQXNCQSxDQUFDQyxTQUFzQixFQUFhO0VBQ3hFLE1BQU1DLFNBQVMsR0FBRyxDQUFDLENBQWM7RUFDakNELFNBQVMsQ0FBQ2xCLE9BQU8sQ0FBRW9CLFFBQW1CLElBQUs7SUFDekM7SUFDQSxJQUFJQSxRQUFRLENBQUNmLE9BQU8sQ0FBQ2dCLFlBQVksS0FBS2pELFNBQVMsRUFBRTtNQUFBLElBQUFrRCxpQkFBQSxFQUFBQyxrQkFBQSxFQUFBQyxrQkFBQSxFQUFBQyxrQkFBQSxFQUFBQyxrQkFBQSxFQUFBQyxrQkFBQSxFQUFBQyxrQkFBQTtNQUMvQyxNQUFNQyxPQUFPLEdBQUdULFFBQVEsQ0FBQ2QsR0FBRzs7TUFFNUI7TUFDQTtNQUNBO01BQ0EsSUFBSXdCLHFCQUFxQixHQUFHVixRQUFRLENBQUNmLE9BQU8sQ0FBQzBCLFNBRWhDO01BQ2IsSUFBSUMsY0FBYyxHQUFHNUQsU0FBdUM7TUFDNUQsSUFBSTBELHFCQUFxQixLQUFLMUQsU0FBUyxFQUFFO1FBQ3ZDLElBQUk2RCxXQUFXLEdBQUc3RCxTQUFtQztRQUNyRCxJQUFJOEQsWUFBWSxHQUFHOUQsU0FBb0M7UUFDdkQsSUFBSTtVQUNGNkQsV0FBVyxHQUFHRSxJQUFJLENBQUNDLEtBQUssQ0FBQ04scUJBQXFCLENBQUNPLEtBQUssQ0FBQztRQUN2RCxDQUFDLENBQUMsTUFBTSxDQUFDO1FBQ1QsSUFBSTtVQUNGSCxZQUFZLEdBQUdDLElBQUksQ0FBQ0MsS0FBSyxDQUFDTixxQkFBcUIsQ0FBQ1EsTUFBTSxDQUFDO1FBQ3pELENBQUMsQ0FBQyxNQUFNLENBQUM7UUFDVE4sY0FBYyxHQUFHO1VBQ2ZLLEtBQUssRUFBRUosV0FBVztVQUNsQkssTUFBTSxFQUFFSjtRQUNWLENBQW1CO01BQ3JCOztNQUVBO01BQ0E7TUFDQWYsU0FBUyxDQUFDVSxPQUFPLENBQUMsR0FBRztRQUNuQjVDLEVBQUUsRUFBRTRDLE9BQU87UUFDWDNDLElBQUksR0FBQW9DLGlCQUFBLEdBQUVGLFFBQVEsQ0FBQ2YsT0FBTyxjQUFBaUIsaUJBQUEsdUJBQWhCQSxpQkFBQSxDQUFrQnBDLElBQUk7UUFDNUI7UUFDQXFELFNBQVMsRUFBRUMsdUJBQWUsRUFBQWpCLGtCQUFBLEdBQUNILFFBQVEsQ0FBQ2YsT0FBTyxjQUFBa0Isa0JBQUEsdUJBQWhCQSxrQkFBQSxDQUFrQmdCLFNBQVMsQ0FBQztRQUN2RDtRQUNBNUIsS0FBSyxFQUFFOEIsbUJBQVcsRUFBQWpCLGtCQUFBLEdBQUNKLFFBQVEsQ0FBQ2YsT0FBTyxjQUFBbUIsa0JBQUEsdUJBQWhCQSxrQkFBQSxDQUFrQmtCLFdBQVcsQ0FBQztRQUNqREMsV0FBVyxFQUFFO1VBQ1hDLFNBQVMsR0FBQW5CLGtCQUFBLEdBQUVMLFFBQVEsQ0FBQ2YsT0FBTyxjQUFBb0Isa0JBQUEsZ0JBQUFBLGtCQUFBLEdBQWhCQSxrQkFBQSxDQUFrQm9CLFlBQVksY0FBQXBCLGtCQUFBLHVCQUE5QkEsa0JBQUEsQ0FBZ0NxQixVQUFVO1VBQ3JEQyxrQkFBa0IsR0FBQXJCLGtCQUFBLEdBQ2hCTixRQUFRLENBQUNmLE9BQU8sY0FBQXFCLGtCQUFBLGdCQUFBQSxrQkFBQSxHQUFoQkEsa0JBQUEsQ0FBa0JtQixZQUFZLGNBQUFuQixrQkFBQSx1QkFBOUJBLGtCQUFBLENBQWdDc0I7UUFDcEMsQ0FBQztRQUNEakIsU0FBUyxFQUFFQyxjQUFjO1FBQ3pCaUIsV0FBVyxHQUFBdEIsa0JBQUEsR0FBRVAsUUFBUSxDQUFDZixPQUFPLGNBQUFzQixrQkFBQSx1QkFBaEJBLGtCQUFBLENBQWtCdUIsWUFBWTtRQUMzQ0MsU0FBUyxHQUFBdkIsa0JBQUEsR0FBRVIsUUFBUSxDQUFDZixPQUFPLGNBQUF1QixrQkFBQSx1QkFBaEJBLGtCQUFBLENBQWtCdUI7TUFDL0IsQ0FBVTtJQUNaO0VBQ0YsQ0FBQyxDQUFDO0VBQ0YsT0FBT2hDLFNBQVM7QUFDbEI7QUFFTyxTQUFTaUMsMEJBQTBCQSxDQUN4Q2xDLFNBQXNCLEVBQ1A7RUFDZixNQUFNbUMsYUFBYSxHQUFHLENBQUMsQ0FBa0I7RUFDekNuQyxTQUFTLENBQUNsQixPQUFPLENBQUVzRCxZQUF1QixJQUFLO0lBQUEsSUFBQUMscUJBQUEsRUFBQUMsc0JBQUEsRUFBQUMsc0JBQUEsRUFBQUMsc0JBQUEsRUFBQUMsc0JBQUEsRUFBQUMsc0JBQUE7SUFDN0MsTUFBTVgsV0FBVyxHQUFHSyxZQUFZLENBQUNoRCxHQUFHOztJQUVwQztJQUNBO0lBQ0ErQyxhQUFhLENBQUNKLFdBQVcsQ0FBQyxHQUFHO01BQzNCaEUsRUFBRSxFQUFFZ0UsV0FBVztNQUNmL0QsSUFBSSxHQUFBcUUscUJBQUEsR0FBRUQsWUFBWSxDQUFDakQsT0FBTyxjQUFBa0QscUJBQUEsdUJBQXBCQSxxQkFBQSxDQUFzQnJFLElBQUk7TUFDaEMyRSxRQUFRLEdBQUFMLHNCQUFBLEdBQUVGLFlBQVksQ0FBQ2pELE9BQU8sY0FBQW1ELHNCQUFBLHVCQUFwQkEsc0JBQUEsQ0FBc0JLLFFBQVE7TUFDeENDLFVBQVUsRUFBRTtRQUNWQyxLQUFLLEdBQUFOLHNCQUFBLEdBQUVILFlBQVksQ0FBQ2pELE9BQU8sY0FBQW9ELHNCQUFBLGdCQUFBQSxzQkFBQSxHQUFwQkEsc0JBQUEsQ0FBc0JLLFVBQVUsY0FBQUwsc0JBQUEsdUJBQWhDQSxzQkFBQSxDQUFrQ00sS0FBSztRQUM5Q0MsVUFBVSxHQUFBTixzQkFBQSxHQUFFSixZQUFZLENBQUNqRCxPQUFPLGNBQUFxRCxzQkFBQSx1QkFBcEJBLHNCQUFBLENBQXNCSSxVQUFVLENBQUNFO01BQy9DLENBQUM7TUFDREMsT0FBTyxFQUFFLEVBQUFOLHNCQUFBLEdBQUFMLFlBQVksQ0FBQ2pELE9BQU8sY0FBQXNELHNCQUFBLHVCQUFwQkEsc0JBQUEsQ0FBc0JNLE9BQU8sS0FBSSxFQUFFO01BQzVDQyxhQUFhLEVBQUUsRUFBQU4sc0JBQUEsR0FBQU4sWUFBWSxDQUFDakQsT0FBTyxjQUFBdUQsc0JBQUEsdUJBQXBCQSxzQkFBQSxDQUFzQk0sYUFBYSxLQUFJLENBQUM7SUFDekQsQ0FBYztFQUNoQixDQUFDLENBQUM7RUFDRixPQUFPYixhQUFhO0FBQ3RCOztBQUVBO0FBQ08sU0FBUzNDLDRCQUE0QkEsQ0FDMUNDLEtBQXdDLEVBQ3hCO0VBQ2hCLE1BQU13RCxVQUFVLEdBQUd4RCxLQUFLLElBQUl5RCx1Q0FBK0I7RUFDM0Q7RUFDQSxPQUFPQyxzQkFBYyxDQUFDRixVQUFVLENBQUM7QUFDbkM7O0FBRUE7QUFDTyxTQUFTckQsK0JBQStCQSxDQUM3Q0UsZ0JBQW1DLEVBQ2Y7RUFDcEIsTUFBTXNELGNBQWMsR0FBRyxFQUF3QjtFQUMvQyxJQUFJdEQsZ0JBQWdCLEVBQUU7SUFDcEJBLGdCQUFnQixDQUFDaEIsT0FBTyxDQUFFdUUsZUFBZSxJQUFLO01BQzVDRCxjQUFjLENBQUNFLElBQUksQ0FBQztRQUNsQnZGLEVBQUUsRUFBRXNGLGVBQWUsQ0FBQ0UsV0FBVztRQUMvQkMsUUFBUSxFQUFFSCxlQUFlLENBQUNJLGtCQUFrQjtRQUM1Q2hHLElBQUk7UUFDRjtRQUNBaUcsOEJBQXNCO1FBQ3BCO1FBQ0NMLGVBQWUsQ0FBQ00sYUFBYSxDQUFZQyxXQUFXLENBQUMsQ0FBQztNQUU3RCxDQUFxQixDQUFDO0lBQ3hCLENBQUMsQ0FBQztFQUNKO0VBQ0EsT0FBT1IsY0FBYztBQUN2Qjs7QUFFQTtBQUNPLFNBQVNTLFVBQVVBLENBQUMvRixTQUFjLEVBQUVDLEVBQVUsRUFBUztFQUM1RDtFQUNBO0VBQ0EsSUFBSStGLGFBQWE7RUFDakIsSUFBSTtJQUFBLElBQUFDLHFCQUFBO0lBQ0ZELGFBQWEsR0FBRzdDLElBQUksQ0FBQ0MsS0FBSyxDQUN4QnBELFNBQVMsYUFBVEEsU0FBUyxnQkFBQWlHLHFCQUFBLEdBQVRqRyxTQUFTLENBQUU4RSxVQUFVLGNBQUFtQixxQkFBQSx1QkFBckJBLHFCQUFBLENBQXVCQyxjQUN6QixDQUFtQjtFQUNyQixDQUFDLENBQUMsTUFBTSxDQUFDO0VBQ1QsT0FBTztJQUNMakcsRUFBRTtJQUNGQyxJQUFJLEVBQUVGLFNBQVMsYUFBVEEsU0FBUyx1QkFBVEEsU0FBUyxDQUFFRSxJQUFJO0lBQ3JCUCxJQUFJLEVBQUVLLFNBQVMsYUFBVEEsU0FBUyx1QkFBVEEsU0FBUyxDQUFFTCxJQUFJO0lBQ3JCUyxXQUFXLEVBQUVKLFNBQVMsYUFBVEEsU0FBUyx1QkFBVEEsU0FBUyxDQUFFSSxXQUFXO0lBQ25DK0YsS0FBSyxFQUFFbkcsU0FBUyxhQUFUQSxTQUFTLHVCQUFUQSxTQUFTLENBQUVtRyxLQUFLO0lBQ3ZCQyxHQUFHLEVBQUVwRyxTQUFTLGFBQVRBLFNBQVMsdUJBQVRBLFNBQVMsQ0FBRW9HLEdBQUc7SUFDbkJDLE1BQU0sRUFBRXJHLFNBQVMsYUFBVEEsU0FBUyx1QkFBVEEsU0FBUyxDQUFFcUcsTUFBTTtJQUN6QnZCLFVBQVUsRUFBRTtNQUFFLElBQUc5RSxTQUFTLGFBQVRBLFNBQVMsdUJBQVRBLFNBQVMsQ0FBRThFLFVBQVU7TUFBRW9CLGNBQWMsRUFBRUY7SUFBYztFQUN4RSxDQUFDO0FBQ0g7O0FBRUE7QUFDTyxTQUFTTSxzQkFBc0JBLENBQUNDLFNBQXNCLEVBQWE7RUFDeEUsTUFBTUMsTUFBTSxHQUFHLENBQUMsQ0FBYztFQUM5QixLQUFLLE1BQU1DLFFBQVEsSUFBSUYsU0FBUyxFQUFFO0lBQ2hDLE1BQU1HLE1BQU0sR0FBR0QsUUFBUSxDQUFDcEYsT0FBZ0I7SUFDeEMsTUFBTXBCLEVBQUUsR0FBR3dHLFFBQVEsQ0FBQ25GLEdBQUc7SUFDdkI7SUFDQWtGLE1BQU0sQ0FBQ3ZHLEVBQUUsQ0FBQyxHQUFHOEYsVUFBVSxDQUFDVyxNQUFNLEVBQUV6RyxFQUFFLENBQUM7RUFDckM7RUFDQSxPQUFPdUcsTUFBTTtBQUNmIn0=