#!/bin/sh -efu
#
# Copyright (c) 2017 Dmitry V. Levin <ldv@strace.io>
# Copyright (c) 2017-2025 The strace developers.
# All rights reserved.
#
# SPDX-License-Identifier: GPL-2.0-or-later

usage()
{
	cat >&2 <<EOF
Usage: $0 [<input> [<output>]]

Generate test script(s) from <input> table.
If a <output> is specified, generate the script for this test only,
otherwise generate scripts for all tests specified in <input> table.
EOF
	exit 1
}

input="${0%/*}/gen_tests.in"
[ $# -eq 0 ] || { input="$1"; shift; }
output=
[ $# -eq 0 ] || { output="$1"; shift; }
[ $# -eq 0 ] || usage

if [ -n "$output" ]; then
	match="${output##*/}"
	match="${match%.gen.test}"
	[ -n "$match" ] || usage
else
	match=
	dir="$(dirname "$input")"
fi

names=

while read -r name arg0 args; do {
	[ -n "${name###*}" ] || continue
	if [ -z "$match" ]; then
		names="$names $name"
		output="$dir/$name.gen.test"
	else
		[ "$match" = "$name" ] || continue
	fi

	hdr="\
#!/bin/sh -efu
# Generated by $0 from $input ($name $arg0 $args); do not edit."

	case "$arg0" in
		+*)
		cat <<-EOF
		$hdr
		set -- $args
		. "\${srcdir=.}/${arg0#+}"
		EOF
		;;

		# Tests that rely on syscall tampering require additional checks
		# implemented in the scno_tamperng.sh.
		-einject=*|--inject=*)
		case " $args " in
			*\ -y\ *)
			cat <<-EOF
			$hdr
			. "\${srcdir=.}/scno_tampering.sh"
			run_strace_match_diff $arg0 $args
			EOF
			;;
			*)
			cat <<-EOF
			$hdr
			. "\${srcdir=.}/scno_tampering.sh"
			run_strace $arg0 $args "../$name" > "\$EXP"
			match_diff "\$LOG" "\$EXP"
			EOF
			;;
		esac
		;;

		''|-*)
		cat <<-EOF
		$hdr
		. "\${srcdir=.}/init.sh"
		run_strace_match_diff $arg0 $args
		EOF
		;;

		*)
		cat <<-EOF
		$hdr
		. "\${srcdir=.}/init.sh"
		$arg0 $args
		EOF
		;;
	esac > "$output"

	chmod a+x "$output"
} < /dev/null; done < "$input"

if [ -n "$names" ]; then
	{
		printf '# Generated by %s from %s; do not edit.\n' "$0" "$input"
		printf 'GEN_TESTS ='
		printf ' %s.gen.test' $names
		echo
		target='$(srcdir)/%s.gen.test'
		dep1='$(abs_srcdir)/gen_tests.sh'
		dep2='$(srcdir)/gen_tests.in'
		recipe='$(AM_V_GEN) $^ $@'
		printf "\\n$target: $dep1 $dep2\\n\\t$recipe\\n" $names
	} > "$dir/gen_tests.am"
fi
