/* SPDX-License-Identifier: GPL-2.0-or-later */
/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2006-2010  Nokia Corporation
 *  Copyright (C) 2004-2010  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 */

#include <stdbool.h>
#include <dbus/dbus.h>
#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/sdp.h>

#define ADAPTER_INTERFACE	"org.bluez.Adapter1"

#define MAX_NAME_LENGTH		248

/* Invalid SSP passkey value used to indicate negative replies */
#define INVALID_PASSKEY		0xffffffff

struct btd_adapter;
struct btd_device;
struct queue;

struct btd_adapter *btd_adapter_get_default(void);
bool btd_adapter_is_default(struct btd_adapter *adapter);
uint16_t btd_adapter_get_index(struct btd_adapter *adapter);
bool btd_adapter_has_cable_pairing_devices(struct btd_adapter *adapter);

typedef void (*adapter_cb) (struct btd_adapter *adapter, gpointer user_data);

typedef void (*oob_read_local_cb_t) (struct btd_adapter *adapter,
					const uint8_t *hash,
					const uint8_t *randomizer,
					void *user_data);
typedef void (*oob_bonding_cb_t) (struct btd_adapter *adapter,
					const bdaddr_t *bdaddr, uint8_t status,
					void *user_data);

struct oob_handler {
	oob_read_local_cb_t read_local_cb;
	oob_bonding_cb_t bonding_cb;
	bdaddr_t remote_addr;
	void *user_data;
};

int adapter_init(void);
void adapter_cleanup(void);
void adapter_shutdown(void);

void btd_adapter_foreach(adapter_cb func, gpointer user_data);

typedef void (*btd_disconnect_cb) (struct btd_device *device, uint8_t reason);
void btd_add_disconnect_cb(btd_disconnect_cb func);
void btd_remove_disconnect_cb(btd_disconnect_cb func);

typedef void (*btd_conn_fail_cb) (struct btd_device *device, uint8_t status);
void btd_add_conn_fail_cb(btd_conn_fail_cb func);
void btd_remove_conn_fail_cb(btd_conn_fail_cb func);

struct btd_adapter *adapter_find(const bdaddr_t *sba);
struct btd_adapter *adapter_find_by_id(int id);

bool btd_adapter_get_pairable(struct btd_adapter *adapter);
bool btd_adapter_get_powered(struct btd_adapter *adapter);
bool btd_adapter_get_connectable(struct btd_adapter *adapter);
bool btd_adapter_get_discoverable(struct btd_adapter *adapter);
bool btd_adapter_get_bredr(struct btd_adapter *adapter);

struct btd_gatt_database *btd_adapter_get_database(struct btd_adapter *adapter);

uint32_t btd_adapter_get_class(struct btd_adapter *adapter);
const char *btd_adapter_get_name(struct btd_adapter *adapter);
void btd_adapter_remove_device(struct btd_adapter *adapter,
				struct btd_device *dev);
struct btd_device *btd_adapter_get_device(struct btd_adapter *adapter,
					const bdaddr_t *addr,
					uint8_t addr_type);
sdp_list_t *btd_adapter_get_services(struct btd_adapter *adapter);

struct btd_device *btd_adapter_find_device(struct btd_adapter *adapter,
							const bdaddr_t *dst,
							uint8_t dst_type);
struct btd_device *btd_adapter_find_device_by_path(struct btd_adapter *adapter,
						   const char *path);
struct btd_device *btd_adapter_find_device_by_fd(int fd);

void btd_adapter_device_found(struct btd_adapter *adapter,
					const bdaddr_t *bdaddr,
					uint8_t bdaddr_type, int8_t rssi,
					uint32_t flags,
					const uint8_t *data, uint8_t data_len,
					bool monitoring);

const char *adapter_get_path(struct btd_adapter *adapter);
const bdaddr_t *btd_adapter_get_address(struct btd_adapter *adapter);
uint8_t btd_adapter_get_address_type(struct btd_adapter *adapter);
const char *btd_adapter_get_storage_dir(struct btd_adapter *adapter);

int adapter_service_add(struct btd_adapter *adapter, sdp_record_t *rec);
void adapter_service_remove(struct btd_adapter *adapter, uint32_t handle);

struct agent *adapter_get_agent(struct btd_adapter *adapter);

bool btd_adapter_uuid_is_allowed(struct btd_adapter *adapter, const char *uuid);

struct btd_adapter *btd_adapter_ref(struct btd_adapter *adapter);
void btd_adapter_unref(struct btd_adapter *adapter);

void btd_adapter_set_class(struct btd_adapter *adapter, uint8_t major,
							uint8_t minor);
int btd_adapter_set_name(struct btd_adapter *adapter, const char *name);

struct btd_adapter_driver {
	const char *name;
	int (*probe)(struct btd_adapter *adapter);
	void (*remove)(struct btd_adapter *adapter);
	void (*resume)(struct btd_adapter *adapter);
	void (*device_added)(struct btd_adapter *adapter,
						struct btd_device *device);
	void (*device_removed)(struct btd_adapter *adapter,
						struct btd_device *device);
	void (*device_resolved)(struct btd_adapter *adapter,
						struct btd_device *device);

	/* Indicates the driver is experimental and shall only be registered
	 * when experimental has been enabled (see: main.conf:Experimental).
	 */
	bool experimental;
};

void device_resolved_drivers(struct btd_adapter *adapter,
						struct btd_device *device);
typedef void (*service_auth_cb) (DBusError *derr, void *user_data);

void adapter_add_profile(struct btd_adapter *adapter, gpointer p);
void adapter_remove_profile(struct btd_adapter *adapter, gpointer p);
int btd_register_adapter_driver(const struct btd_adapter_driver *driver);
void btd_unregister_adapter_driver(const struct btd_adapter_driver *driver);
guint btd_request_authorization(const bdaddr_t *src, const bdaddr_t *dst,
		const char *uuid, service_auth_cb cb, void *user_data);
guint btd_request_authorization_cable_configured(const bdaddr_t *src, const bdaddr_t *dst,
		const char *uuid, service_auth_cb cb, void *user_data);
int btd_cancel_authorization(guint id);

int btd_adapter_restore_powered(struct btd_adapter *adapter);
int btd_adapter_set_blocked(struct btd_adapter *adapter);

typedef ssize_t (*btd_adapter_pin_cb_t) (struct btd_adapter *adapter,
			struct btd_device *dev, char *out, bool *display,
							unsigned int attempt);
void btd_adapter_register_pin_cb(struct btd_adapter *adapter,
						btd_adapter_pin_cb_t cb);
void btd_adapter_unregister_pin_cb(struct btd_adapter *adapter,
						btd_adapter_pin_cb_t cb);

struct btd_adapter_pin_cb_iter *btd_adapter_pin_cb_iter_new(
						struct btd_adapter *adapter);
void btd_adapter_pin_cb_iter_free(struct btd_adapter_pin_cb_iter *iter);
bool btd_adapter_pin_cb_iter_end(struct btd_adapter_pin_cb_iter *iter);

typedef void (*btd_msd_cb_t) (struct btd_adapter *adapter,
							struct btd_device *dev,
							uint16_t company,
							const uint8_t *data,
							uint8_t data_len);
void btd_adapter_register_msd_cb(struct btd_adapter *adapter,
							btd_msd_cb_t cb);
void btd_adapter_unregister_msd_cb(struct btd_adapter *adapter,
							btd_msd_cb_t cb);

/* If TRUE, enables fast connectabe, i.e. reduces page scan interval and changes
 * type. If FALSE, disables fast connectable, i.e. sets page scan interval and
 * type to default values. Valid for both connectable and discoverable modes. */
int btd_adapter_set_fast_connectable(struct btd_adapter *adapter,
							gboolean enable);

int btd_adapter_read_clock(struct btd_adapter *adapter, const bdaddr_t *bdaddr,
				int which, int timeout, uint32_t *clock,
				uint16_t *accuracy);

int btd_adapter_block_address(struct btd_adapter *adapter,
				const bdaddr_t *bdaddr, uint8_t bdaddr_type);
int btd_adapter_unblock_address(struct btd_adapter *adapter,
				const bdaddr_t *bdaddr, uint8_t bdaddr_type);

int btd_adapter_disconnect_device(struct btd_adapter *adapter,
							const bdaddr_t *bdaddr,
							uint8_t bdaddr_type);

int btd_adapter_remove_bonding(struct btd_adapter *adapter,
				const bdaddr_t *bdaddr, uint8_t bdaddr_type);

int btd_adapter_pincode_reply(struct btd_adapter *adapter,
					const  bdaddr_t *bdaddr,
					const char *pin, size_t pin_len);
int btd_adapter_confirm_reply(struct btd_adapter *adapter,
				const bdaddr_t *bdaddr, uint8_t bdaddr_type,
				gboolean success);
int btd_adapter_passkey_reply(struct btd_adapter *adapter,
				const bdaddr_t *bdaddr, uint8_t bdaddr_type,
				uint32_t passkey);

int adapter_create_bonding(struct btd_adapter *adapter, const bdaddr_t *bdaddr,
					uint8_t addr_type, uint8_t io_cap);

int adapter_bonding_attempt(struct btd_adapter *adapter, const bdaddr_t *bdaddr,
					uint8_t addr_type, uint8_t io_cap);

int adapter_cancel_bonding(struct btd_adapter *adapter, const bdaddr_t *bdaddr,
							uint8_t addr_type);

int adapter_set_io_capability(struct btd_adapter *adapter, uint8_t io_cap);

int btd_adapter_read_local_oob_data(struct btd_adapter *adapter);

int btd_adapter_add_remote_oob_data(struct btd_adapter *adapter,
					const bdaddr_t *bdaddr,
					uint8_t *hash, uint8_t *randomizer);

int btd_adapter_remove_remote_oob_data(struct btd_adapter *adapter,
							const bdaddr_t *bdaddr);

int btd_adapter_gatt_server_start(struct btd_adapter *adapter);
void btd_adapter_gatt_server_stop(struct btd_adapter *adapter);

bool btd_adapter_ssp_enabled(struct btd_adapter *adapter);

int adapter_connect_list_add(struct btd_adapter *adapter,
					struct btd_device *device);
void adapter_connect_list_remove(struct btd_adapter *adapter,
						struct btd_device *device);
typedef void (*adapter_set_device_flags_func_t)(uint8_t status, uint16_t length,
						const void *param,
						void *user_data);
int adapter_set_device_flags(struct btd_adapter *adapter,
				struct btd_device *device, uint32_t flags,
				adapter_set_device_flags_func_t func,
				void *user_data);
void adapter_auto_connect_add(struct btd_adapter *adapter,
					struct btd_device *device);
void adapter_auto_connect_remove(struct btd_adapter *adapter,
					struct btd_device *device);
void adapter_accept_list_add(struct btd_adapter *adapter,
						struct btd_device *dev);
void adapter_accept_list_remove(struct btd_adapter *adapter,
						struct btd_device *dev);

void btd_adapter_set_oob_handler(struct btd_adapter *adapter,
						struct oob_handler *handler);
gboolean btd_adapter_check_oob_handler(struct btd_adapter *adapter);

void btd_adapter_for_each_device(struct btd_adapter *adapter,
			void (*cb)(struct btd_device *device, void *data),
			void *data);

bool btd_le_connect_before_pairing(void);

bool btd_adapter_has_settings(struct btd_adapter *adapter, uint32_t settings);

enum experimental_features {
	EXP_FEAT_DEBUG			= 1 << 0,
	EXP_FEAT_LE_SIMULT_ROLES	= 1 << 1,
	EXP_FEAT_BQR			= 1 << 2,
	EXP_FEAT_RPA_RESOLUTION		= 1 << 3,
	EXP_FEAT_CODEC_OFFLOAD		= 1 << 4,
	EXP_FEAT_ISO_SOCKET		= 1 << 5,
};

bool btd_adapter_has_exp_feature(struct btd_adapter *adapter, uint32_t feature);

enum kernel_features {
	KERNEL_CONN_CONTROL		= 1 << 0,
	KERNEL_BLOCKED_KEYS_SUPPORTED	= 1 << 1,
	KERNEL_SET_SYSTEM_CONFIG	= 1 << 2,
	KERNEL_EXP_FEATURES		= 1 << 3,
	KERNEL_HAS_RESUME_EVT		= 1 << 4,
	KERNEL_HAS_EXT_ADV_ADD_CMDS	= 1 << 5,
	KERNEL_HAS_CONTROLLER_CAP_CMD	= 1 << 6,
};

bool btd_has_kernel_features(uint32_t feature);

bool btd_adapter_set_allowed_uuids(struct btd_adapter *adapter,
							struct queue *uuids);
bool btd_adapter_is_uuid_allowed(struct btd_adapter *adapter,
							const char *uuid_str);

void btd_adapter_load_conn_param(struct btd_adapter *adapter,
				const bdaddr_t *peer, uint8_t bdaddr_type,
				uint16_t min_interval, uint16_t max_interval,
				uint16_t latency, uint16_t timeout);

void btd_adapter_store_conn_param(struct btd_adapter *adapter,
				const bdaddr_t *peer, uint8_t bdaddr_type,
				uint16_t min_interval, uint16_t max_interval,
				uint16_t latency, uint16_t timeout);
void btd_adapter_cancel_service_auth(struct btd_adapter *adapter,
				struct btd_device *device);
